/*
 * rstr.h
 *
 * Global defines and data structures.
 */

#define	TAPE		"/dev/rmt0"	/* default tape device */

#define	TAPEBLK		10240	/* size (bytes) of physical tape records */

#ifndef	ULTRIX_11
# define	BSIZE	512	/* size (bytes) of logical tape records */
# define	PROMPT	"512restor> "	/* prompt for interactive "shell" */
#else	ULTRIX_11
# define	BSIZE	1024
# define	PROMPT	"1024restor> "
#endif	ULTRIX_11

#define	NTREC   	TAPEBLK/BSIZE	/* logical records per physical */
#define	PAD		BSIZE-88	/* chars in c_addr field (see below) */

#define	DISBLK		8192
#define	NDREC		DISBLK/BSIZE

#define	MLEN    	16	/* number of bits in a bit map word */
#define	MSIZ    	4096	/* number of words in a bit map */

#define	MAXINO		65536	/* number of directory inodes we can handle */
#define	MAXNAME		255	/* max number of chars in full pathname */
#define	BLOCKDIR	1	/* ignore this inode */
#define	ROOTDIR		2	/* inode 2 is always root of filsys */

#define	D_TYPE		07	/* mask for file type bits */
#define	REGFILE		01	/*  ...regular (non-directory) file */
#define DIRECT		02	/*  ...directory */
#define	MARKED		010	/* set if object marked for extraction */

/*
 * c_type field mnemonics for struct spcl
 */
#define	TS_TAPE 	1	/* tape volume label */
#define	TS_INODE	2	/* file or directory follows */
#define	TS_BITS 	3	/* bit map -- bit set if inode dumped */
#define	TS_ADDR 	4	/* subrecord of large file description */
#define	TS_END  	5	/* end of tape record */
#define	TS_CLRI 	6	/* bit map -- bit clear if inode empty on */
				/* dumped file system */

#define	MAGIC   	(ushort) 60011	/* all headers have this in c_magic */
#define	CHECKSUM	(ushort) 84446	/* all headers checksum to this */

/*
 * gives value of word containing bit corresponding to i in bitmap 
 */
#define	MWORD(m,i)	(m[(unsigned)(i-1) / MLEN])

/*
 * used to isolate the bit in a found (via MWORD) word
 */
#define	MBIT(i)		(1 << ((unsigned)(i-1) % MLEN))

/*
 * evaluates to true if bit i is set in bitmap w
 */
#define	BIT(i,w)	(MWORD(w,i) & MBIT(i))

/*
 * V7 structure -- entries in directories have this format
 */
struct dir {
	ushort	d_inode;
	char	d_name[14];
};

/*
 * Structure for inode description array.  We index into an array
 * of these structures by inode.  The array is malloc'd, and is
 * big enough to store the largest inode dumped to tape (according
 * to the directory information written first on tape).  Even so,
 * the array can be pretty big.
 *
 * The "extraction list" is really maintained here.
 */
struct dino {
	int	dn_type;	/* type: REGFILE, DIRECT plus MARKED */
	long	dn_seek;	/* seek adrress:
				 *	if DIRECT, into dirfile, where we
				 *	 find first a copy of the disk inode
				 *	 which described the original directory
				 *	 and then the contents of the directory
				 *	 itself.
				 *	if REGFILE, into namfile, where we
				 *	 can do an fgets to get the pathname
				 *	 of the file associated with the
				 *	 inode.
				 */
};

/*
 * Inode structure as it appeared on a disk block.  Data types here
 * massaged to fit intentions of original PDP code.
 */
struct dinode
{
	ushort	di_mode;     	/* mode and type of file */
	short	di_nlink;    	/* number of links to file */
	short	di_uid;      	/* owner's user id */
	short	di_gid;      	/* owner's group id */
	long	di_size;     	/* number of bytes in file */
	char  	di_addr[40];	/* disk block addresses */
	long	di_atime;   	/* time last accessed */
	long	di_mtime;   	/* time last modified */
	long	di_ctime;   	/* time created */
};

/*
 * Structure of dump tape header records.
 */
struct	spcl
{
	short	c_type;		/* type of header */
	long	c_date;		/* date dump was taken */
	long	c_ddate;	/* date the file system was last dumped from */
	short	c_volume;	/* current volume of the dump */
	long	c_tapea;	/* current number of this logical record */
	ushort	c_inumber;	/* if c_type=TS_INODE, number of inode dumped */
	ushort	c_magic;	/* contains value MAGIC, trucated as needed */
	short	c_checksum;	/* makes record sum to value CHECKSUM */
	struct	dinode	c_dinode; /* copy of the disk inode */
	short	c_count;	/* count of chars in c_addr, if TS_INODE -- */
				/* also, # of blocks in map for bitmap types */
	char	c_addr[PAD];	/* array of chars describing blocks in file. */
				/* char=0 --> block associated with char was */
				/* not present on filesystem (i.e., restore */
				/* as a hole).  If c_addr field is not large */
				/* enough to describe file, as many TS_ADDR */
				/* records as needed will follow */
};
