/*
 * getfile.c
 *
 * Modules related to reading files from tape.  What is done with
 * a particular file varies, so we use a general-purpose routine
 * getfile which is passed function pointers to perform whatever
 * tasks are appropriate for the particular file read.
 *
 * The attempt here is to provide a kind of package-like behavior;
 * before getfile can be safely called it is sometimes necessary to
 * call some support routines.  This avoids accessing program-wide
 * globals.
 */

#include <stdio.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/file.h>
#include "rstr.h"

extern char	*progname;

/*
 * getfile
 *
 * Read a file from tape, calling the functions f1 and f2 to
 * process what is read.  Expected behavior of functions:
 *
 *	(*f1)(buf, size)-- process the size characters in buf.
 *	(*f2)		-- perform whatever wrap-up might be necessary
 * 			   before leaving.
 *
 * Getfile returns 0 on success, -1 with problems, and expects f1 and
 * f2 to have the same return conventions.
 */
getfile(fd, hp, f1, f2)
struct spcl	*hp;
int		(*f1)(), (*f2)();
{
	struct dinode	*dp = &hp->c_dinode;
	long		size;
	char		buf[BSIZE];
	int		block_index = 0;

	/*
	 * We use size as the loop control here, since experience shows
	 * that we can't fully trust the value in hp->c_count; it seems
	 * to be 10 for lots of short files on V7 dump tapes (I haven't
	 * checked dump to see if there's any reason for this), although
	 * correct for the larger files.
	 */
	for (size = dp->di_size; size > 0; size -= BSIZE) {
		/*
		 * More blocks than can be described in one c_addr array
		 * require the reading in of additional TS_ADDR blocks.
		 */
		if (block_index >= hp->c_count) {
			if (read_hdr(fd, hp) == -1) {
				fprintf(stderr, "getfile: bad read_hdr\n");
				return -1;
			}
			if (hp->c_type != TS_ADDR) {
				fprintf(stderr,"getfile: didn't get TS_ADDR\n");
				return -1;
			}
			block_index = 0;
		}
		/*
		 * hp->c_addr[block_index] != 0 means that the next tape block
		 * contains valid file information; otherwise the block was
		 * all zeros in the original file and was not actually written
		 * to the archive.
		 */
		if (hp->c_addr[block_index]) {
			if (read_block(fd, buf) == -1) {
				fprintf(stderr, "getfile: bad read_block\n");
				return -1;
			}
		} else {
			bzero(buf, BSIZE);
		}
		/*
		 * f1 will process the valid charcters in buf.
		 */
		if ((*f1)(buf, size > BSIZE ? BSIZE : size) == -1) {
			fprintf(stderr, "getfile: f1 problem\n");
			return -1;
		}
		block_index++;
	}
	if ((*f2)() == -1) {
		fprintf(stderr, "getfile: f2 problem\n");
		return -1;
	}
	return 0;
}

/*
 * Code and data relating to the f1 and f2
 * for the "write tapefile to disk" function.
 */

static int	wfd;		/* file descriptor for written file */
static char	diskbuf[DISBLK], /* buffer for tape blocks */
		*dp = diskbuf;	/* pointer where next tape block goes */
static int	nwritten = 0;	/* number of tape blocks "written" to */
				/* disk buffer */

/*
 * write_block
 *
 * "Write" the block buf to file descriptor wfd.  We collect
 * blocks until we have a buffer of size DISBLK to write (try
 * to make better use of BSD file system).
 */
write_block(buf, sz)
char	*buf;
{
	long	lseek();

	if (nwritten == NDREC) {
		if (iszero(diskbuf, DISBLK)) {
			if (lseek(wfd, (long) DISBLK, L_INCR) == -1L) {
				perror("lseek in write_block");
				return -1;
			}
		} else {
			if (write(wfd, diskbuf, DISBLK) != DISBLK) {
				perror("write in write_block");
				return -1;
			}
		}
		nwritten = 0;
		dp = diskbuf;
	}
	bcopy(buf, dp, sz);
	dp += sz;
	nwritten++;

	return 0;
}

/*
 * flush_out
 *
 * Write any partial contents of diskbuf out to disk.
 */
flush_out()
{
	int	bytes_left = dp - diskbuf,
		retval = 0;
	long	lseek();

	if (!bytes_left)
		return 0;
	if (iszero(diskbuf, bytes_left)) {
		if (lseek(wfd, (long) bytes_left, L_INCR) == -1L) {
			perror("lseek in flush_out");
			retval = -1;
		}
	} else {
		if (write(wfd, diskbuf, bytes_left) == -1) {
			perror("write in flush_out");
			retval = -1;
		}
	}
	dp = diskbuf;
	nwritten = 0;

	return retval;
}

/*
 * open_file
 *
 * Opens name, and sets wfd (global to this module; see above)
 * to the opened file descriptor.
 */
open_file(fname)
char	*fname;
{
	char	nambuf[MAXNAME+5],
		rbuf[BUFSIZ];

	if (*fname == '/')	/* we ALWAYS restore relative to */
		fname++;	/* current directory */

	if (access(fname, F_OK) != -1) {
		fprintf(stderr, "%s warning: %s exists -- overwrite? <y or n> ",
			progname, fname);
		(void) fflush(stdout);
		(void) gets(rbuf);
		if (*rbuf != 'y' && *rbuf != 'Y') {
			(void) strcpy(nambuf, fname);
			(void) strcat(nambuf, ".RST");
			printf("restore file as %s? <y or n> ", nambuf);
			(void) fflush(stdout);
			(void) gets(rbuf);
			if (*rbuf == 'y' || *rbuf == 'Y')
				fname = nambuf;
			else
				fname = "/dev/null";	/* why screw up the */
		}					/* algorithm for a */
	}						/* special case? */
	/*
	 * We don't open the file with any modes set -- it's up
	 * to read_regfile to set owner/mode after the file is closed
	 */
	if ((wfd = open(fname, O_WRONLY|O_CREAT|O_TRUNC, 0)) < 0) {
		perror("open_file: open");
		return -1;
	}

	return 0;
}

/*
 * close_file
 *
 * Very simple -- just closes wfd.  (All data global to this
 * module can only be accessed by routines within it -- kind
 * of a "package" behavior).
 */

close_file()
{
	(void) close(wfd);
}

/*
 * iszero
 *
 * Check to see the the passed buffer buf of size sz is full
 * of zeroes.  Return 1 if so, 0 otherwise.
 */
static
iszero(buf, sz)
char	*buf;
{
	register	i;

	for (i = 0; i < sz; i++)
		if (buf[i])
			return 0;
	return 1;
}

/*
 * Code and date relating to the f1 and f2
 * for the "write tapefile to buffer" function.
 * (Used to read directory files in to core buffers.)
 */

static char	*membuf = NULL,	/* pointer to malloc'd buffer */
		*mp;		/* pointer within that buffer */
static int	memsize = 0;	/* size of malloc'd buffer */

/*
 * alloc_mem
 *
 * Malloc sz bytes, set membuf to point to the new buffer.
 */
char *
alloc_mem(sz)
unsigned	sz;
{
	char	*malloc();

	if (sz > memsize) {
		if (membuf)
			free(membuf);
		if ((membuf = malloc(sz)) == NULL) {
			perror("alloc_mem: malloc");
			return NULL;
		}
		memsize = sz;
	}
	mp = membuf;	/* "shouldn't be needed" */

	return membuf;
}

/*
 * copy_buf
 *
 * Copy the contents of buf to the global buffer membuf.
 * Update mp to point where the next block goes.
 */
copy_buf(buf, sz)
char	*buf;
{
	bcopy(buf, mp, sz);
	mp += sz;

	return 0;
}

/*
 * flush_buf
 *
 * Do whatever's necessary when we've seen end of tapefile.
 * Not as much to do here as in the "write file" case (no
 * buffering is done by copy_buf) -- just reset mp.
 */
flush_buf()
{
	mp = membuf;

	return 0;
}

/*
 * null
 *
 * null case -- if we're just ignoring a particular file
 */
null()
{
	return 0;
}
