/* nova_mta.c: NOVA magnetic tape simulator

   Copyright (c) 1995-1997,
   Robert M Supnik, Digital Equipment Corporation
   Commercial use prohibited

   mta		magnetic tape

   18-Jan-97	RMS	New MT format
   29-Jun-96	RMS	Added unit disable support

   Magnetic tapes are represented as a series of variable records
   of the form:

	32b byte count			byte count is little endian
	byte 0
	byte 1
	:
	byte n-2
	byte n-1
	32b byte count

   If the byte count is odd, the record is padded with an extra byte
   of junk.  File marks are represented by a byte count of 0 and are
   not duplicated; end of tape by end of file.
*/

#include "nova_defs.h"

#define MTA_NUMDR	8				/* #drives */
#define UNIT_V_WLK	(UNIT_V_UF + 0)			/* write locked */
#define UNIT_W_UF	2				/* saved flags width */
#define UNIT_WLK	1 << UNIT_V_WLK
#define USTAT		u3				/* unit status */
#define UNUM		u4				/* unit number */
#define DTSIZE		(1 << 14)			/* max data xfer */
#define DTMASK		(DTSIZE - 1)

/* Command/unit */

#define	CU_CI		0100000				/* clear interrupt */
#define CU_EP		0002000				/* poll enable */
#define CU_DE		0001000				/* disable erase */
#define CU_DA		0000400				/* disable autoretry */
#define CU_PE		0000400				/* PE mode */
#define CU_V_CMD	3				/* command */
#define CU_M_CMD	027
#define  CU_READ	 000
#define  CU_REWIND	 001
#define  CU_CMODE	 002
#define  CU_SPACEF	 003
#define  CU_SPACER	 004
#define  CU_WRITE	 005
#define  CU_WREOF	 006
#define  CU_ERASE	 007
#define  CU_READNS	 020
#define  CU_UNLOAD	 021
#define  CU_DMODE	 022
#define CU_V_UNIT	0				/* unit */
#define CU_M_UNIT	07
#define GET_CMD(x)	(((x) >> CU_V_CMD) & CU_M_CMD)
#define GET_UNIT(x)	(((x) >> CU_V_UNIT) & CU_M_UNIT)

/* Status 1 - stored in mta_sta<31:16> or (*) uptr -> USTAT<31:16> */

#define STA_ERR1	(0100000u << 16)		/* error */
#define STA_DLT		(0040000 << 16)			/* data late */
#define STA_REW		(0020000 << 16)			/* *rewinding */
#define STA_ILL		(0010000 << 16)			/* illegal */
#define STA_HDN		(0004000 << 16)			/* high density */
#define STA_DAE		(0002000 << 16)			/* data error */
#define STA_EOT		(0001000 << 16)			/* *end of tape */
#define STA_EOF		(0000400 << 16)			/* *end of file */
#define STA_BOT		(0000200 << 16)			/* *start of tape */
#define STA_9TK		(0000100 << 16)			/* nine track */
#define STA_BAT		(0000040 << 16)			/* bad tape */
#define STA_CHG		(0000010 << 16)			/* status change */
#define STA_WLK		(0000004 << 16)			/* *write lock */
#define STA_ODD		(0000002 << 16)			/* odd character */
#define STA_RDY		(0000001 << 16)			/* *drive ready */

/* Status 2 - stored in mta_sta<15:0> or (*) uptr -> USTAT<15:0> */

#define STA_ERR2	0100000				/* error */
#define STA_RWY		0040000				/* runaway tape */
#define STA_FGP		0020000				/* false gap */
#define STA_CDL		0004000				/* corrected dlt */
#define STA_V_UNIT	8
#define STA_M_UNIT	07				/* unit */
#define STA_WCO		0000200				/* word count ovflo */
#define STA_BDS		0000100				/* bad signal */
#define STA_OVS		0000040				/* overskew */
#define STA_CRC		0000020				/* check error */
#define STA_STE		0000010				/* single trk error */
#define STA_FPR		0000004				/* false preamble */
#define STA_FMT		0000002				/* format error */
#define STA_PEM		0000001				/* *PE mode */

#define STA_EFLGS1 	(STA_DLT | STA_ILL | STA_DAE | STA_EOT | \
			 STA_EOF | STA_BOT | STA_BAT | STA_ODD)
#define STA_EFLGS2	(STA_FGP | STA_CDL | STA_BDS | STA_OVS | \
			 STA_CRC | STA_FPR | STA_FPR)	/* set error 2 */
#define STA_CLR		((020 << 16) | 0010000)		/* always clear */
#define STA_SET		(STA_HDN | STA_9TK)		/* always set */
#define STA_DYN		(STA_REW | STA_EOT | STA_EOF | STA_BOT | \
			 STA_WLK | STA_RDY | STA_PEM)	/* kept in USTAT */
#define STA_MON		(STA_REW | STA_BOT | STA_WLK | STA_RDY | \
			 STA_PEM)			/* set status chg */

extern unsigned int16 M[];
extern UNIT cpu_unit;
extern int32 int_req, dev_busy, dev_done, dev_disable;
int32 mta_ma = 0;					/* memory address */
int32 mta_wc = 0;					/* word count */
int32 mta_cu = 0;					/* command/unit */
int32 mta_sta = 0;					/* status register */
int32 mta_ep = 0;					/* enable polling */
int32 mta_cwait = 100;					/* command latency */
int32 mta_rwait = 100;					/* record latency */
t_stat mta_svc (UNIT *uptr);
t_stat mta_reset (DEVICE *dptr);
t_stat mta_boot (int32 unitno);
t_stat mta_attach (UNIT *uptr, char *cptr);
t_stat mta_detach (UNIT *uptr);
int32 mta_updcsta (UNIT *uptr);
void mta_upddsta (UNIT *uptr, int32 newsta);
t_stat mta_vlock (UNIT *uptr, int32 val);
extern t_stat sim_activate (UNIT *uptr, int32 delay);
extern t_stat sim_cancel (UNIT *uptr);
extern int32 sim_is_active (UNIT *uptr);
extern size_t fxread (void *bptr, size_t size, size_t count, FILE *fptr);
extern size_t fxwrite (void *bptr, size_t size, size_t count, FILE *fptr);

static const int ctype[32] = {				/* c vs r timing */
 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1  };

/* MTA data structures

   mta_dev	MTA device descriptor
   mta_unit	MTA unit list
   mta_reg	MTA register list
   mta_mod	MTA modifier list
*/

UNIT mta_unit[] = {
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) },
	{ UDATA (&mta_svc, UNIT_ATTABLE + UNIT_DISABLE, 0) }  };

REG mta_reg[] = {
	{ ORDATA (CU, mta_cu, 16) },
	{ ORDATA (MA, mta_ma, 16) },
	{ ORDATA (WC, mta_wc, 16) },
	{ GRDATA (STA1, mta_sta, 8, 16, 16) },
	{ ORDATA (STA2, mta_sta, 16) },
	{ FLDATA (EP, mta_ep, 0) },
	{ FLDATA (BUSY, dev_busy, INT_V_MTA) },
	{ FLDATA (DONE, dev_done, INT_V_MTA) },
	{ FLDATA (DISABLE, dev_disable, INT_V_MTA) },
	{ FLDATA (INT, int_req, INT_V_MTA) },
	{ DRDATA (CTIME, mta_cwait, 24), PV_LEFT },
	{ DRDATA (RTIME, mta_rwait, 24), PV_LEFT },
	{ ORDATA (UST0, mta_unit[0].USTAT, 32) },
	{ ORDATA (UST1, mta_unit[1].USTAT, 32) },
	{ ORDATA (UST2, mta_unit[2].USTAT, 32) },
	{ ORDATA (UST3, mta_unit[3].USTAT, 32) },
	{ ORDATA (UST4, mta_unit[4].USTAT, 32) },
	{ ORDATA (UST5, mta_unit[5].USTAT, 32) },
	{ ORDATA (UST6, mta_unit[6].USTAT, 32) },
	{ ORDATA (UST7, mta_unit[7].USTAT, 32) },
	{ DRDATA (POS0, mta_unit[0].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS1, mta_unit[1].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS2, mta_unit[2].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS3, mta_unit[3].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS4, mta_unit[4].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS5, mta_unit[5].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS6, mta_unit[6].pos, 31), REG_RO + PV_LEFT },
	{ DRDATA (POS7, mta_unit[7].pos, 31), REG_RO + PV_LEFT },
	{ GRDATA (FLG0, mta_unit[0].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG1, mta_unit[1].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG2, mta_unit[2].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG3, mta_unit[3].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG4, mta_unit[4].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG5, mta_unit[5].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG6, mta_unit[6].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG7, mta_unit[7].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ NULL }  };

MTAB mta_mod[] = {
	{ UNIT_WLK, 0, "write enabled", "ENABLED", &mta_vlock },
	{ UNIT_WLK, UNIT_WLK, "write locked", "LOCKED", &mta_vlock },
	{ 0 }  };

DEVICE mta_dev = {
	"MT", mta_unit, mta_reg, mta_mod,
	MTA_NUMDR, 10, 31, 1, 8, 8,
	NULL, NULL, &mta_reset,
	&mta_boot, &mta_attach, &mta_detach };

/* IOT routine */

int32 mta (int32 pulse, int32 code, int32 AC)
{
UNIT *uptr;
int32 u, c, rval;

rval = 0;
uptr = mta_dev.units + GET_UNIT(mta_cu);		/* get unit */
switch (code) {						/* decode IR<5:7> */
case ioDIA:						/* DIA */
	rval = (mta_updcsta (uptr) >> 16) & 0177777;	/* return status 1 */
	break;
case ioDOA:						/* DOA */
/*	if (AC & CU_CI) ... clear ep int			*/
	mta_cu = AC;					/* save cmd/unit */
	uptr = mta_dev.units + GET_UNIT(mta_cu);	/* get unit */
	mta_updcsta (uptr);				/* update status */
	break;
case ioDIB:						/* DIB */
	rval = mta_ma & ADDRMASK;			/* return ma */
	break;
case ioDOB:						/* DOB */
	mta_ma = AC & ADDRMASK;				/* save ma */
	break;
case ioDIC:						/* DIC */
	rval = mta_updcsta (uptr) & 0177777;		/* return status 2 */
	break;
case ioDOC:						/* DOC */
	mta_wc = ((AC & 040000) << 1) | (AC & 077777);	/* save wc */
	break;  }					/* end switch code */

/* IOT, continued */

switch (pulse) {					/* decode IR<8:9> */
case iopS:						/* start */
	c = GET_CMD (mta_cu);				/* get command */
	if (dev_busy & INT_MTA) break;			/* ignore if busy */
	if ((uptr -> USTAT & STA_RDY) == 0) {		/* drive not ready? */
		mta_sta = mta_sta | STA_ILL;		/* illegal op */
		dev_busy = dev_busy & ~INT_MTA;		/* clear busy */
		dev_done = dev_done | INT_MTA;		/* set done */
		int_req = (int_req & ~INT_DEV) | (dev_done & ~dev_disable);  }
	else if ((c == CU_REWIND) || (c == CU_UNLOAD)) { /* rewind, unload? */
		mta_upddsta (uptr, (uptr -> USTAT &	/* update status */
			~(STA_BOT | STA_EOF | STA_EOT | STA_RDY)) | STA_REW);
		sim_activate (uptr, mta_rwait);		/* start IO */
		if (c == CU_UNLOAD) detach_unit (uptr);  }
	else {	mta_sta = 0;				/* clear errors */
		dev_busy = dev_busy | INT_MTA;		/* set busy */
		dev_done = dev_done & ~INT_MTA;		/* clear done */
		int_req = int_req & ~INT_MTA;		/* clear int */
		if (ctype[c]) sim_activate (uptr, mta_cwait);
		else {	mta_upddsta (uptr, uptr -> USTAT &
			   ~(STA_BOT | STA_EOF | STA_EOT | STA_RDY));
			sim_activate (uptr, mta_rwait);  }  }
	mta_updcsta (uptr);				/* update status */
	break;
case iopC:						/* clear */
	for (u = 0; u < MTA_NUMDR; u++) {		/* loop thru units */
		uptr = mta_dev.units + u;		/* cancel IO */
		if (sim_is_active (uptr) && !(uptr -> USTAT & STA_REW)) {
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
			sim_cancel (uptr);  }  }
	dev_busy = dev_busy & ~INT_MTA;			/* clear busy */
	dev_done = dev_done & ~INT_MTA;			/* clear done */
	int_req = int_req & ~INT_MTA;			/* clear int */
	mta_sta = mta_cu = mta_ma = 0;			/* clear registers */
	mta_updcsta (&mta_unit[0]);			/* update status */
	break;  }					/* end case pulse */
return rval;
}

/* Unit service

   If rewind done, reposition to start of tape, set status
   else, do operation, clear busy, set done, interrupt
*/

t_stat mta_svc (UNIT *uptr)
{
int32 c, i, p, u, err;
t_stat rval;
t_mtrlnt cbc, tbc, wc;
unsigned int16 c1, c2;
unsigned int8 dbuf[2 * DTSIZE];
static t_mtrlnt bceof = { 0 };

rval = SCPE_OK;
u = uptr -> UNUM;					/* get unit number */
if (uptr -> USTAT & STA_REW) {				/* rewind? */
	uptr -> pos = 0;				/* update position */
	mta_upddsta (uptr, (uptr -> USTAT & ~STA_REW) | STA_BOT | STA_RDY);
	if (u == GET_UNIT (mta_cu)) mta_updcsta (uptr);
	return rval;  }

err = 0;
c = GET_CMD (mta_cu);					/* command */
wc = DTSIZE - (mta_wc & DTMASK);			/* io wc */

if ((uptr -> flags & UNIT_ATT) == 0) {			/* not attached? */
	mta_upddsta (uptr, 0);				/* unit off line */
	mta_sta = mta_sta | STA_ILL;  }			/* illegal operation */

else if ((uptr -> flags & UNIT_WLK) &&			/* write locked? */
	((c == CU_WRITE) || (c == CU_WREOF) || (c == CU_ERASE))) {
	mta_upddsta (uptr, uptr -> USTAT | STA_WLK | STA_RDY);
	mta_sta = mta_sta | STA_ILL;  }			/* illegal operation */
	
else switch (c) {					/* case on command */
case CU_CMODE:						/* controller mode */
	mta_ep = mta_cu & CU_EP;
	break;
case CU_DMODE:						/* drive mode */
	if (uptr -> pos) mta_sta = mta_sta | STA_ILL;	/* must be BOT */
	else mta_upddsta (uptr, (mta_cu & CU_PE)?	/* update drv status */
		uptr -> USTAT | STA_PEM: uptr -> USTAT & ~ STA_PEM);
	break;

/* Unit service, continued */

case CU_READ:						/* read */
case CU_READNS:						/* read non-stop */
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	fxread (&tbc, sizeof (t_mtrlnt), 1, uptr -> fileref); /* read byte count */
	if ((err = ferror (uptr -> fileref)) ||		/* error or eof? */
	    (feof (uptr -> fileref))) {
		mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_EOT);
		break;  }
	if (tbc == 0) {					/* tape mark? */
		mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_EOF);
		uptr -> pos = uptr -> pos + sizeof (t_mtrlnt);
		break;  }
	cbc = wc * 2;					/* expected bc */
	if (tbc & 1) mta_sta = mta_sta | STA_ODD;	/* odd byte count? */
	if (tbc > cbc) mta_sta = mta_sta | STA_WCO;	/* too big? */
	else {	cbc = tbc;				/* no, use it */
		wc = (cbc + 1) / 2;  }			/* adjust wc */
	i = fxread (dbuf, sizeof (int8), cbc, uptr -> fileref);
	for ( ; i < cbc; i++) dbuf[i] = 0;
	err = ferror (uptr -> fileref);
	for (i = p = 0; i < wc; i++) {			/* copy buf to mem */
		c1 = dbuf[p++];
		c2 = dbuf[p++];
		if (MEM_ADDR_OK (mta_ma)) M[mta_ma] = (c1 << 8) | c2;
		mta_ma = (mta_ma + 1) & ADDRMASK;  }
	mta_wc = (mta_wc + wc) & 0177777;
	uptr -> pos = uptr -> pos + ((tbc + 1) & ~1) +
		(2 * sizeof (t_mtrlnt));
	mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
	break;
case CU_WRITE:						/* write */
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	tbc = wc * 2;					/* io byte count */
	fxwrite (&tbc, sizeof (t_mtrlnt), 1, uptr -> fileref);
	for (i = p = 0; i < wc; i++) {		/* copy to buffer */
		dbuf[p++] = (M[mta_ma] >> 8) & 0377;
		dbuf[p++] = M[mta_ma] & 0377;
		mta_ma = (mta_ma + 1) & ADDRMASK;  }
	fxwrite (dbuf, sizeof (int8), tbc, uptr -> fileref);
	fxwrite (&tbc, sizeof (t_mtrlnt), 1, uptr -> fileref);
	err = ferror (uptr -> fileref);
	mta_wc = 0;
	uptr -> pos = uptr -> pos + tbc + (2 * sizeof (t_mtrlnt));
	mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
	break;
case CU_WREOF:						/* write eof */
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	fxwrite (&bceof, sizeof (t_mtrlnt), 1, uptr -> fileref);
	err = ferror (uptr -> fileref);
	uptr -> pos = uptr -> pos + sizeof (t_mtrlnt);
	mta_upddsta (uptr, uptr -> USTAT | STA_EOF | STA_RDY);
	break;
case CU_ERASE:						/* erase */
	mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
	break;

/* Unit service, continued */

case CU_SPACEF:						/* space forward */
	do {	mta_wc = (mta_wc + 1) & 0177777;	/* incr wc */
		fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
		fxread (&tbc, sizeof (t_mtrlnt), 1, uptr -> fileref);
		if ((err = ferror (uptr -> fileref)) ||	/* error or eof? */
		    (feof (uptr -> fileref))) {
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_EOT);
			break;  }
		if (tbc == 0) {				/* zero bc? */
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_EOF);
			uptr -> pos = uptr -> pos + sizeof (t_mtrlnt);
			break;  }
		uptr -> pos = uptr -> pos + ((tbc + 1) & ~1) +
			(2 * sizeof (t_mtrlnt));  }
	while (mta_wc != 0);
	mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
	break;
case CU_SPACER:						/* space reverse */
	if (uptr -> pos == 0) {				/* at BOT? */
		mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_BOT);
		break;  }
	do {	mta_wc = (mta_wc + 1) & 0177777;	/* incr wc */
		fseek (uptr -> fileref, uptr -> pos - sizeof (t_mtrlnt),
			SEEK_SET);			/* bs to reclnt */
		fxread (&tbc, sizeof (t_mtrlnt), 1, uptr -> fileref);
		if ((err = ferror (uptr -> fileref)) ||	/* error or eof? */
		    (feof (uptr -> fileref))) {
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_BOT);
			uptr -> pos = 0;
			break;  }
		if (tbc == 0) {				/* zero bc? */
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_EOF);
			uptr -> pos = uptr -> pos - sizeof (t_mtrlnt);
			break;  }
		uptr -> pos = uptr -> pos - ((tbc + 1) & ~1) -
			(2 * sizeof (t_mtrlnt));
		if (uptr -> pos == 0) {			/* start of tape? */
			mta_upddsta (uptr, uptr -> USTAT | STA_RDY | STA_BOT);
			break;  }  }
	while (mta_wc != 0);
	mta_upddsta (uptr, uptr -> USTAT | STA_RDY);
	break;
default:						/* reserved */
	mta_sta = mta_sta | STA_ILL;
	break;  }					/* end case */

/* Unit service, continued */

if (err != 0) {						/* I/O error */
	mta_sta = mta_sta | STA_DAE;			/* flag error */
	perror ("MTA I/O error");
	rval = SCPE_IOERR;
	clearerr (uptr -> fileref);  }
mta_updcsta (uptr);					/* update status */
dev_busy = dev_busy & ~INT_MTA;				/* clear busy */
dev_done = dev_done | INT_MTA;				/* set done */
int_req = (int_req & ~INT_DEV) | (dev_done & ~dev_disable);
return rval;
}

/* Update controller status */

int32 mta_updcsta (UNIT *uptr)				/* update ctrl */
{
mta_sta = (mta_sta & ~(STA_DYN | STA_CLR | STA_ERR1 | STA_ERR2)) |
	(uptr -> USTAT & STA_DYN) | STA_SET;
if (mta_sta & STA_EFLGS1) mta_sta = mta_sta | STA_ERR1;
if (mta_sta & STA_EFLGS2) mta_sta = mta_sta | STA_ERR2;
return mta_sta;
}

/* Update drive status */

void mta_upddsta (UNIT *uptr, int32 newsta)		/* drive status */
{
int32 change;

if ((uptr -> flags & UNIT_ATT) == 0) newsta = 0;	/* offline? */
change = (uptr -> USTAT ^ newsta) & STA_MON;		/* changes? */
uptr -> USTAT = newsta & STA_DYN;			/* update status */
if (change) {
/*	if (mta_ep) {					/* if polling */
/*		u = uptr -> UNUM;			/* unit num */
/*		mta_sta = (mta_sta & ~STA_UNIT) | (u << STA_V_UNIT);
/*		set polling interupt...  }		*/
	mta_sta = mta_sta | STA_CHG;  }			/* flag change */
return;
}

/* Reset routine */

t_stat mta_reset (DEVICE *dptr)
{
int32 u;
UNIT *uptr;

dev_busy = dev_busy & ~INT_MTA;				/* clear busy */
dev_done = dev_done & ~INT_MTA;				/* clear done, int */
int_req = int_req & ~INT_MTA;
mta_cu = mta_wc = mta_ma = mta_sta = 0;			/* clear registers */
mta_ep = 0;
for (u = 0; u < MTA_NUMDR; u++) {			/* loop thru units */
	uptr = mta_dev.units + u;
	sim_cancel (uptr);				/* cancel activity */
	uptr -> UNUM = u;
	if (uptr -> flags & UNIT_ATT) uptr -> USTAT = STA_RDY |
			(uptr -> USTAT & STA_PEM) |
			((uptr -> flags & UNIT_WLK)? STA_WLK: 0) |
			((uptr -> pos)? 0: STA_BOT);
	else uptr -> USTAT = 0;  }
mta_updcsta (&mta_unit[0]);				/* update status */
return SCPE_OK;
}

/* Attach routine */

t_stat mta_attach (UNIT *uptr, char *cptr)
{
t_stat r;

r = attach_unit (uptr, cptr);
if (r != SCPE_OK) return r;
if (!sim_is_active (uptr)) mta_upddsta (uptr, STA_RDY | STA_BOT | STA_PEM |
	((uptr -> flags & UNIT_WLK)? STA_WLK: 0));
return r;
}

/* Detach routine */

t_stat mta_detach (UNIT* uptr)
{
if (!sim_is_active (uptr)) mta_upddsta (uptr, 0);
return detach_unit (uptr);
}

/* Write lock/unlock validate routine */

t_stat mta_vlock (UNIT *uptr, int32 val)
{
if ((uptr -> flags & UNIT_ATT) && val)
	mta_upddsta (uptr, uptr -> USTAT | STA_WLK);
else mta_upddsta (uptr, uptr -> USTAT & ~STA_WLK);
return SCPE_OK;
}

/* Bootstrap routine */

#define BOOT_START 02000
#define BOOT_UNIT 02020
#define BOOT_LEN (sizeof (boot_rom) / sizeof (int))

static const int32 boot_rom[] = {
	060222,			/* NIOC 0,MTA		; clear disk */
	020417,			/* LDA 0,UNIT		; unit */
	024417,			/* LDA 1,REWIND		; cmd */
	0107000,		/* ADD 0,1		; cmd + unit */
	065122,			/* DOAS 1,MTA		; start rewind */
	070422,			/* DIA 2,MTA		; get status */
	0151213,		/* MOVR# 2,2,SNC	; skip if done */
	000776,			/* JMP .-2 */
	0126400,		/* SUB 1,1 		; ma, wc = 0 */
	066022,			/* DOB 1,MTA */
	067022,			/* DOC 1,MTA */
	061122,			/* DOAS 0,MTA		; start read */
	070422,			/* DIA 2,MTA		; get status */
	0151213,		/* MOVR# 2,2,SNC	; skip if done */
	000776,			/* JMP .-2 */
	000377,			/* JMP 377 */
	000000,			/* UNIT: */
	000010			/* REWIND: 10 */
};

t_stat mta_boot (int32 unitno)
{
int32 i;
extern int32 saved_PC;

for (i = 0; i < BOOT_LEN; i++) M[BOOT_START + i] = boot_rom[i];
M[BOOT_UNIT] = (unitno & CU_M_UNIT) << CU_V_UNIT;
saved_PC = BOOT_START;
return SCPE_OK;
}
