/* IBM 1401 magnetic tape simulator

   Copyright (c) 1996, Robert M Supnik, Digital Equipment Corporation
   Commercial use prohibited

   Magnetic tapes are represented as a series of variable 16b records
   of the form:

	byte count low			byte count is little endian
	byte count high
	byte 0
	byte 1
	:
	byte n-2
	byte n-1

   If the byte count is odd, the record is padded with an extra byte
   of junk.  File marks are represented by a byte count of 0.
*/

#include "i1401_defs.h"

#define MT_NUMDR	7				/* #drives */
#define UNIT_V_WLK	(UNIT_V_UF + 0)			/* write locked */
#define UNIT_WLK	(1 << UNIT_V_WLK)
#define UNIT_W_UF	2				/* #save flags */

extern unsigned char M[];				/* memory */
extern int ind[64];					/* indicator */
extern int iochk;					/* I/O check stop */
extern UNIT cpu_unit;
unsigned char dbuf[MAXMEMSIZE * 2];			/* tape buffer */
int mt_reset (DEVICE *dptr);
UNIT *get_unit (int unit);
extern int detach_unit (UNIT *uptr);

/* MT data structures

   mt_dev	MT device descriptor
   mt_unit	MT unit list
   mt_reg	MT register list
   mt_mod	MT modifier list
*/

UNIT mt_unit[] = {
	{ UDATA (NULL, UNIT_DIS, 0) },			/* doesn't exist */
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) },
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) },
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) },
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) },
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) },
	{ UDATA (NULL, UNIT_DISABLE + UNIT_ATTABLE + UNIT_BCD, 0) }  };

REG mt_reg[] = {
	{ FLDATA (END, ind[IN_END], 0) },
	{ FLDATA (ERR, ind[IN_TAP], 0) },
	{ FLDATA (PAR, ind[IN_PAR], 0) },
	{ DRDATA (POS1, mt_unit[1].pos, 31), PV_LEFT + REG_RO },
	{ DRDATA (POS2, mt_unit[2].pos, 31), PV_LEFT + REG_RO },
	{ DRDATA (POS3, mt_unit[3].pos, 31), PV_LEFT + REG_RO },
	{ DRDATA (POS4, mt_unit[4].pos, 31), PV_LEFT + REG_RO },
	{ DRDATA (POS5, mt_unit[5].pos, 31), PV_LEFT + REG_RO },
	{ DRDATA (POS6, mt_unit[6].pos, 31), PV_LEFT + REG_RO },
	{ GRDATA (FLG1, mt_unit[1].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG2, mt_unit[2].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG3, mt_unit[3].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG4, mt_unit[4].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG5, mt_unit[5].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ GRDATA (FLG6, mt_unit[6].flags, 8, UNIT_W_UF, UNIT_V_UF - 1),
		  REG_HRO },
	{ NULL }  };

MTAB mt_mod[] = {
	{ UNIT_WLK, 0, "write enabled", "ENABLED", NULL },
	{ UNIT_WLK, UNIT_WLK, "write locked", "LOCKED", NULL }, 
	{ 0 }  };

DEVICE mt_dev = {
	"MT", mt_unit, mt_reg, mt_mod,
	MT_NUMDR, 10, 31, 1, 8, 6,
	NULL, NULL, &mt_reset,
	NULL, NULL, NULL };

/* Function routine

   Inputs:
	unit	=	unit character
	mod	=	modifier character
   Outputs:
	status	=	status
*/

int mt_func (int unit, int mod)
{
int p, prior, lnt, err;
UNIT *uptr;
static const unsigned char bceof[2] = { 0 };

if ((uptr = get_unit (unit)) == NULL) return STOP_INVMTU; /* valid unit? */
if ((uptr -> flags & UNIT_ATT) == 0) return SCPE_UNATT;	/* attached? */
switch (mod) {						/* case on modifier */
case BCD_B:						/* backspace */
	ind[IN_END] = 0;				/* clear end of reel */
	if (uptr -> pos == 0) return SCPE_OK;		/* at bot? */
	for (p = prior = 0; p < uptr -> pos; ) {	/* find prev record */
		fseek (uptr -> fileref, p, SEEK_SET);
		fread (dbuf, sizeof (char), 2, uptr -> fileref);
		lnt = ((unsigned int) dbuf[1] << 8) | (unsigned int) dbuf[0];
		if ((err = ferror (uptr -> fileref)) ||
		    (feof (uptr -> fileref))) {
			uptr -> pos = p;
			break;  }
		prior = p;				/* save loc of prev */
		p = p + ((lnt + 3) & ~1);  }		/* calc new loc */
	uptr -> pos = prior;				/* backspace */
	break;  					/* end case */
case BCD_E:						/* erase = nop */
	if (uptr -> flags & UNIT_WLK) return STOP_MTL;
	return SCPE_OK;
case BCD_M:						/* write tapemark */
	if (uptr -> flags & UNIT_WLK) return STOP_MTL;
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	fwrite (bceof, sizeof (char), 2, uptr -> fileref); /* write eof */
	err = ferror (uptr -> fileref);
	uptr -> pos = uptr -> pos + 2;			/* update position */
	break;
case BCD_R:						/* rewind */
	uptr -> pos = 0;				/* update position */
	return SCPE_OK;
case BCD_U:						/* unload */
	uptr -> pos = 0;				/* update position */
	return detach_unit (uptr);			/* detach */
default:
	return STOP_INVM;  }
if (err != 0) {						/* I/O error */
	perror ("MT I/O error");
	clearerr (uptr -> fileref);
	if (iochk) return SCPE_IOERR;
	ind[IN_TAP] = 1;  }
return SCPE_OK;
}

/* Read and write routines

   Inputs:
	unit	=	unit character
	flag	=	normal, word mark, or binary mode
	addr	=	buffer starting address
	mod	=	modifier character
	*new	=	pointer to output buffer ending address
   Outputs:
	status	=	status
*/

int mt_io (int unit, int flag, int addr, int mod, int *new)
{
int err, i, t, tbc, wm_seen;
UNIT *uptr;

if ((uptr = get_unit (unit)) == NULL) return STOP_INVMTU; /* valid unit? */
if ((uptr -> flags & UNIT_ATT) == 0) return SCPE_UNATT;	/* attached? */
switch (mod) {
case BCD_R:						/* read */
	ind[IN_TAP] = ind[IN_END] = 0;			/* clear error */
	wm_seen = 0;					/* no word mk seen */
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	fread (dbuf, sizeof (char), 2, uptr -> fileref); /* read byte count */
	tbc = ((unsigned int) dbuf[1] << 8) | (unsigned int) dbuf[0];
	if ((err = ferror (uptr -> fileref)) ||	(feof (uptr -> fileref))) {
		ind[IN_END] = 1;			/* err or eof? */
		break;  }
	if (tbc == 0) {					/* tape mark? */
		ind[IN_END] = 1;			/* set end mark */
		uptr -> pos = uptr -> pos + 2;		/* update pos */
		break;  }
	i = fread (dbuf, sizeof (char), tbc, uptr -> fileref);
	for ( ; i < tbc; i++) dbuf[i] = 0;		/* fill with 0's */
	err = ferror (uptr -> fileref);
	uptr -> pos = uptr -> pos + ((tbc + 3) & ~1);
	for (i = 0; (i < tbc) && (M[addr] != (BCD_GRPMRK + WM)); i++) {
		t = dbuf[i];				/* get char */
		if ((flag != MD_BIN) && (t == BCD_ALT)) t = BCD_BLANK;
		if (flag == MD_WM) {			/* word mk mode? */
			if ((t == BCD_WM) && (wm_seen == 0)) wm_seen = WM;
			else {	M[addr] = wm_seen | (t & CHAR);
				wm_seen = 0;  }  }
		else M[addr] = (M[addr] & WM) | (t & CHAR);
		if (!wm_seen) addr++;
		if (addr >= MAXMEMSIZE) {
			*new = addr % MAXMEMSIZE;
			return STOP_NXM;  }  }
	M[addr++] = BCD_GRPMRK + WM;			/* end of record */
	break;

case BCD_W:
	if (uptr -> flags & UNIT_WLK) return STOP_MTL;	/* locked? */
	if (M[addr] == (BCD_GRPMRK + WM)) return STOP_MTZ;	/* eor? */
	ind[IN_TAP] = ind[IN_END] = 0;			/* clear error */
	for (i = 2; (t = M[addr++]) != (BCD_GRPMRK + WM); ) {
		if ((t & WM) && (flag == MD_WM)) dbuf[i++] = BCD_WM;
		if (((t & CHAR) == BCD_BLANK) && (flag != MD_BIN))
			dbuf[i++] = BCD_ALT;
		else dbuf[i++] = t & CHAR;
		if (addr >= MEMSIZE) {
			*new = addr % MAXMEMSIZE;
			return STOP_NXM;  }  }
	tbc = i - 2;
	dbuf[0] = tbc & 0377;
	dbuf[1] = (tbc >> 8) & 0377;
	fseek (uptr -> fileref, uptr -> pos, SEEK_SET);
	fwrite (dbuf, sizeof (char), (tbc + 3) & ~1, uptr -> fileref);
	err = ferror (uptr -> fileref);
	uptr -> pos = uptr -> pos + ((tbc + 3) & ~1);
	break;
default:
	return STOP_INVM;  }

*new = addr;						/* output address */
if (err != 0) {						/* I/O error */
	perror ("MT I/O error");
	clearerr (uptr -> fileref);
	if (iochk) return SCPE_IOERR;
	ind[IN_TAP] = 1;  }				/* flag error */
return SCPE_OK;
}

/* Get unit pointer from unit number */

UNIT *get_unit (int unit)
{
if ((unit <= 0) || (unit >= MT_NUMDR)) return NULL;
else return mt_dev.units + unit;
}

/* Reset routine */

int mt_reset (DEVICE *dptr)
{
ind[IN_END] = ind[IN_PAR] = ind[IN_TAP] = 0;		/* clear indicators */
return SCPE_OK;
}
