/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

# include "proc.h"


/*
 * Processor internal registers
 */
enum {
	Reg_PSW		= 017777776,	/* processor status word */
	Reg_PIRQ	= 017777772,	/* programmed interrupt request */
	Reg_CPUErr	= 017777766,	/* cpu error register */
	Reg_LTC		= 017777546,	/* line time clock */
	Reg_Maint	= 017777750,	/* maintenance register */
	Reg_CSW		= 017777570,	/* console switch */
};

typedef struct IData	IData;

struct IData {
	int	timerreq;	/* pending timer interrupt */
	ushort	csw_switch;
	ushort	csw_led;
	int	csw_enable;
	int	pwrfail;
	u_int	clock_rate;
	u_int	lbolt;
	UQuad	seconds;
	u_int	start;
};


static void	proc_ctrl_complete(IODev *);
static void	proc_reset(IODev *);
static ushort	proc_fetch(IODev *, unsigned);
static void	proc_store(IODev *, unsigned, int, ushort);
static ushort	proc_vector(IODev *);
static void	proc_info(IODev *);

static void 	proc_setreq(IODev *);


static int	procc_info(IODev *dev, int argc, char **argv);
static int	procc_csw(IODev *dev, int argc, char **argv);

IODevCmd proc_cmds[] = {
	{ "?",		"[command ...]",	dev_help },
	{ "help",	"[command ...]",	dev_help },
	{ "info",	"",			procc_info },
	{ "csw",	"[on|off|<oct>]",	procc_csw },
	{ NULL },
};

IOOps proc_ops = {
	0,			/* ctrl_create */
	0,			/* dev_create */
	proc_ctrl_complete,	/* ctrl_complete */
	proc_reset,		/* reset */
	proc_fetch,		/* fetch */
	proc_store,		/* store */
	proc_vector,		/* vector */
	0,			/* dma */
	0,			/* async */
	proc_info,		/* info */
	0,			/* flush */

	proc_cmds,		/* cmds */
};

static void	ltc(void *);
static void	pbin(ushort v);


/*
 * create processor
 */
static void
proc_ctrl_complete(IODev *dev)
{
	IData *ip = dev->data = xalloc(sizeof(IData));
	struct timeval tv;

	(void)memset(ip, 0, sizeof(IData));

	ip->csw_enable = csw_enable;

	/*
	 * Load power-up values (assume proc is zeroed)
	 */
	proc.pri = 7;
	proc.halt = 0;
	proc.reg[7] = startloc;

	switch((cpu_options & 06) >> 1) {

	  case 3:	/* user boot address */
		proc.reg[7] = cpu_options & 0177000;
		break;

	  case 2:	/* start 173000 */
		break;

	  case 1:	/* enter monitor */
		proc.halt = 1;
		break;

	  case 0:	/* trap to 24 */
		break;
	}

	proc.reg[6] = 01000;
	proc.maint = (cpu_options & 0177016) | 021;

	/*
	 * Install in iopage
	 */
	proc.iopage[IOP(Reg_PSW   )] = dev;
	proc.iopage[IOP(Reg_CPUErr)] = dev;
	proc.iopage[IOP(Reg_PIRQ  )] = dev;
	proc.iopage[IOP(Reg_LTC   )] = dev;
	proc.iopage[IOP(Reg_Maint )] = dev;
	proc.iopage[IOP(Reg_CSW   )] = dev;

	ip->clock_rate = 1000 / clock_rate;
	if(ip->clock_rate == 0)
		conf_panic("clock_rate too high: %u", clock_rate);

	gettimeofday(&tv, NULL);
	ip->start = tv.tv_sec;
	ip->lbolt = 0;
	ip->seconds = 0;

	register_timer(ip->clock_rate, ltc, dev);
}


/*
 * line clock handle, called with processor 'device'
 */
static void
ltc(void *d)
{
	IData	*ip = ((IODev *)d)->data;
	struct timeval tv;
	u_int real_secs;

	if(!proc.bevent && (proc.ltc & 0100)) {
		ip->timerreq = 1;
		proc_setreq((IODev *)d);
	}

	if(++ip->lbolt == clock_rate) {
		ip->lbolt = 0;
		ip->seconds++;

		gettimeofday(&tv, NULL);
		real_secs = tv.tv_sec - ip->start;

# if 0
		printf("real=%d sys=%d\n", real_secs, ip->seconds);
# endif
		if(real_secs > ip->seconds) {
			/*
			 * clock too slow
			 */
			timer_faster();
		} else if(real_secs < ip->seconds) {
			/*
			 * clock too fast
			 */
			timer_slower();
		}
	}
}

/*
 * reset signal
 */
static void
proc_reset(IODev *dev)
{
	proc.pirq = 0;
	proc.ltc = 0;
	dev->reqpri = 0;
	((IData *)dev->data)->timerreq = 0;
}

/*
 * fetch value from register
 */
ushort
proc_fetch(IODev *dev, unsigned e)
{
	IData *ip = dev->data;
	ushort v;

	switch(e) {
	case Reg_PSW:
		v = ((proc.c&1)<<0) | ((proc.v&1)<<1) | ((proc.z&1)<<2) | ((proc.n&1)<<3)
		  | (proc.t<<4) | (proc.pri << 5) | (proc.regs << 11)
		  | (proc.prevmode << 12) | (proc.curmode << 14);
		break;

	case Reg_CPUErr:	v = proc.cpuerr; 	break;
	case Reg_PIRQ:		v = proc.pirq;		break;
	case Reg_LTC:		v = proc.ltc;		break;
	case Reg_Maint:		v = proc.maint;		break; 
	case Reg_CSW:		v = ip->csw_switch;	break;

	default:
		warn("proc_fetch(%u)", e);
		Trap4(020);
	}	

	return v;
}

/*
 * load internal registers, do with care
 * Note:
 *	the T-bit in the psw can be set only by rti and vectors
 *	but no by direct moves. So don't set/clear it here but do
 *	it at the appropriate places manually.
 */
static void
proc_store(IODev *dev, unsigned pa, int mode, ushort v)
{
	IData *ip = dev->data;
	ushort old;

	switch(pa) {

	case Reg_PSW:
		/*
		 * set all, expect unused and T-bit
		 */
		if(!(mode & M_Low)) {
			int newmode = (v >> 14) & 3, regs = (v >> 11) & 1;
			switchmode(newmode, regs);
			proc.curmode = newmode;
			proc.prevmode = (v >> 12) & 3;
			proc.regs = regs;
		}
		if(!(mode & M_High)) {
			proc.c = (v >> 0) & 1;
			proc.v = (v >> 1) & 1;
			proc.z = (v >> 2) & 1;
			proc.n = (v >> 3) & 1;
			proc.pri = (v >> 5) & 7;
		}
		break;

	case Reg_CPUErr:
		/*
		 * cleared by write
		 */
		proc.cpuerr = 0; 	
		break;


	case Reg_PIRQ:
		/*
		 * high byte writeable
		 */
		if(mode & M_Low)
			break;
		proc.pirq = v & 0177000;

		proc_setreq(dev);
		break;

	case Reg_LTC:
		/*
		 * Bit 0100 only
		 */
		if(mode & M_High)
			break;
		proc.ltc = v & 0100;
		break;

	case Reg_Maint:
		/*
		 * Not writeable
		 */
		break;

	case Reg_CSW:
		old = ip->csw_led;
		if(mode & M_High)
			SHB(ip->csw_led, v);
		else if(mode & M_Low)
			SLB(ip->csw_led, v);
		else
			ip->csw_led = v;
		if(old != ip->csw_led)
			printf("CSW: %06ho\n", ip->csw_led);
		break;

	default:
		warn("proc_store(%u)", pa);
		Trap4(020);
	}	
}

/*
 * return vector
 */
static ushort
proc_vector(IODev *dev)
{
	int pirq = (proc.pirq >> 1) & 7;
	IData *id = (IData *)dev->data;

	if(id->pwrfail) {
		id->pwrfail = 0;
		proc_setreq(dev);
		return 024;
	}
	if(pirq >= 6)
		return 0240;
	if(id->timerreq) {
		id->timerreq = 0;
		proc_setreq(dev);
		return 0100;
	}
	if(pirq > 0)
		return 0240;

	warn("bad call to proc_vector");
	return 4;
}

/*
 * set interrupt requests
 */
static void
proc_setreq(IODev *dev)
{
	IData *ip = dev->data;
	int i, req, pri;

	dev->reqpri = pri = 0;

	if(ip->pwrfail) {
		IRQ(dev, NMIPRI);
		return;
	}

	if(ip->timerreq)
		pri = 6;

	for(i = 0100000, req = 7; i >= 01000; i >>= 1, req--)
		if(proc.pirq & i) {
			proc.pirq = (proc.pirq & 0177000) | (req * 042);
			if(req > pri)
				pri = req;
			break;
		}

	if(pri)
		IRQ(dev, pri);
}

/*
 * assert DCOK
 *
 * There are two cases:
 *	1. real power fail.
 *	   We setup a call to proc_vector, which returns vector 24.
 *	   We start a timer of 1 sec and after that timer expires exit.
 *	2. Other sources of DCOK (for example DEQNA).
 *	   We setup a call to proc_vector, which returns vector 24.
 */
void
proc_pwrfail(int real)
{
	IData	*ip = proc.proc->data;

	if(ip->pwrfail)
		return;		/* already power off */

# if 1
	printf("DCOK = %d asserted\n", real);
# endif
	ip->pwrfail = 1;
	proc_setreq(proc.proc);

	if(real) {
		proc.maint &= ~1;	/* switch off PWOK */
		register_timer(1000, (void (*)(void *))pwrfail_exit, NULL);
	}
}

/*
 * called if power is off
 */
void
pwrfail_exit(void *p UNUSED)
{
/*	printf("power fail exit\n"); */
	exit(0);
}

static int
procc_info(IODev *dev, int argc, char **argv UNUSED)
{
	if(argc != 0)
		return 1;
	proc_info(dev);
	return 0;
}


/*
 * enable/disable/set/print switch register
 */
static int
procc_csw(IODev *dev, int argc, char **argv)
{
	ulong	vv;
	char	*end;
	IData	*ip = dev->data;

	if(argc > 1)
		return 1;
	if(argc == 0) {
		printf("LED    %06o = ", ip->csw_led); pbin(ip->csw_led); putchar('\n');
		printf("SWITCH %06o = ", ip->csw_switch); pbin(ip->csw_switch); putchar('\n');
		printf("CSW %sabled\n", ip->csw_enable ? "en" : "dis");
		return 0;
	}
	if(strcmp(argv[0], "off") == 0) {
		ip->csw_enable = 0;
		printf("CSW disabled\n");
		proc.iopage[IOP(Reg_CSW)] = NULL;
		return 0;
	}
	if(strcmp(argv[0], "on") == 0) {
		ip->csw_enable = 1;
		printf("CSW enabled\n");
		proc.iopage[IOP(Reg_CSW)] = dev;
		return 0;
	}
	vv = strtoul(argv[0], &end, 8);
	if(*end || vv >= 0200000)
		return 1;

	ip->csw_switch = vv;

	return 0;
}

static void
pbin(ushort v)
{
	int	i, j;

	for(i = 0; i < 6; i++) {
		printf("|");
		for(j = (i == 0) ? 2 : 0; j < 3; j++, v <<= 1)
			printf("%c", ".o"[(v & 0100000) != 0]);
	}
	printf("|");
}


static void
proc_info(IODev *dev)
{
	IData *id = dev->data;

	printf("KDJ11A - internal registers\n");
	printf("LTC     %08o: %s   %s%s\n", Reg_LTC, 
		id->timerreq ? "req. pending" : "no req.",
		(proc.ltc & 0100) ? "int-enabled  " : "int-disabled  ",
		id->pwrfail ? "pwrfail-init " : "");
	printf("PSW     %08o: cur=%d prev=%d reg=%d pri=%d %c%c%c%c%c\n", Reg_PSW,
		proc.curmode, proc.prevmode, proc.regs, proc.pri,
		".T"[proc.t], ".N"[proc.n], ".Z"[proc.z], ".V"[proc.v], ".C"[proc.c]);
	printf("PIRQ    %08o: %c%c%c%c%c%c%c %o %o\n", Reg_PIRQ,
		".7"[(proc.pirq & 0100000) != 0],
		".6"[(proc.pirq & 0040000) != 0],
		".5"[(proc.pirq & 0020000) != 0],
		".4"[(proc.pirq & 0010000) != 0],
		".3"[(proc.pirq & 0004000) != 0],
		".2"[(proc.pirq & 0002000) != 0],
		".1"[(proc.pirq & 0001000) != 0],
		(proc.pirq >> 5) & 7,
		(proc.pirq >> 1) & 7);
	printf("CPUErr  %08o: %s%s%s%s%s%s\n", Reg_CPUErr,
		(proc.cpuerr & 0200) ? "HALT " : "",
		(proc.cpuerr & 0100) ? "ADDR " : "",
		(proc.cpuerr & 0040) ? "NXM " : "",
		(proc.cpuerr & 0020) ? "IOTO " : "",
		(proc.cpuerr & 0010) ? "YELL " : "",
		(proc.cpuerr & 0004) ? "RED " : "");
	printf("Maint   %08o: %06o\n", Reg_Maint, proc.maint);
	printf("CSW LED %08o: %06o %sabled\n", Reg_CSW, id->csw_led, id->csw_enable ? "en":"dis");
	printf("SWITCH  %08o: %06o\n", Reg_CSW, id->csw_switch);
}
