/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * KL11 serial line controller
 *
 * arguments:
 *	ctrl
 *	  csr_base ivector ovector irq driver driver-args ...
 *	end
 */

# include "proc.h"

# define NTTY	4
# define FIFO	1024
# define TTYCSRMASK 017777770
# define TTYREGMASK (07)
# define CSR(x)	((x)&TTYCSRMASK)
# define REG(x) ((x)&TTYREGMASK)
# define D(C)	/* C */

typedef struct TTY TTY;
typedef	struct KL KL;

struct TTY {
	int		fd;

	int		tks;		/* keyb ready			*/
	uchar		tpb;

	int		tkint;		/* keyb interrupt enable 	*/
	int		tpint;		/* cons interrupt enable 	*/
	int		tkreq;
	int		tpreq;
	unsigned	addr;		/* basis address	 	*/
	int		tkvec;		/* keyboard vector	 	*/
	int		tpvec;		/* print vector		 	*/
	int		ireq;		/* interrupt request level 	*/

	uchar		fifo[FIFO];	/* receiver queue 	 	*/
	int		ffree;		/* 1st free cell		*/
};

struct KL {
	TTY		ttys[NTTY];
	int		ntty;
};

void	tty_ctrl_create(IODev *, int, char **);
void	tty_dev_create(IODev *, int, char **);
void	tty_ctrl_complete(IODev *);
void	tty_reset(IODev *);
ushort	tty_fetch(IODev *, unsigned);
void	tty_store(IODev *, unsigned, int, ushort);
ushort	tty_vector(IODev *);
void	tty_onsig(IODev *, void *);
void	tty_info(IODev *);

IOOps	kl_ops = {
	tty_ctrl_create,
	tty_dev_create,
	tty_ctrl_complete,
	tty_reset,
	tty_fetch,
	tty_store,
	tty_vector,
	0,
	tty_onsig,
	tty_info,
};

static void comp_req(IODev *);
static void set_tkreq(IODev *, TTY *);
static void shift_fifo(IODev *, TTY *);

void
tty_ctrl_create(IODev *dev, int argc, char **argv)
{
	if(argc)
		conf_panic("kl: to many args in controler configuration");

	dev->data = xalloc(sizeof(KL), "kl_ctrl_create");
}

void
tty_dev_create(IODev *dev, int argc, char **argv)
{
	KL	*t = (KL *)dev->data;
	int	i;

	if(argc < 6)
		conf_panic("kl: device requires min 6 args");

	i = t->ntty++;
	if(t->ntty > 4)
		conf_panic("kl: cannot have more than 4 lines");

	t->ttys[i].fd = register_handler(argv[4], argv+4, dev, &t->ttys[i]);

	t->ttys[i].addr = parse_csr(argv[0], "kl");
	t->ttys[i].tkvec = parse_vec(argv[1], "kl");
	t->ttys[i].tpvec = parse_vec(argv[2], "kl");
	t->ttys[i].ireq = parse_irq(argv[3], "kl");
	proc.iopage[IOP(t->ttys[i].addr+0)] = dev;
	proc.iopage[IOP(t->ttys[i].addr+2)] = dev;
	proc.iopage[IOP(t->ttys[i].addr+4)] = dev;
	proc.iopage[IOP(t->ttys[i].addr+6)] = dev;
}

void
tty_ctrl_complete(IODev *dev)
{
	/* nix */
}

/*
 * to get a signal on the next write into the write end
 * we need to read all of the pipe without blocking and
 * discard all but the last byte
 */
void
tty_onsig(IODev *dev, void *argp)
{
	TTY *t = (TTY *)argp;
	int got;
	int drop;
	int newchars;
	char buf[1024];

	if((got = read(t->fd, buf, sizeof(buf))) < 0)
		panic("error reading pipe: %s", strerror(errno));
	else if(got == 0) {
		Warning("XXX error reading pipe: %s", strerror(errno));
		return;
	}

	D(printf("got %d chars; ", got));
	if((drop = t->ffree + got - FIFO) > 0) {
		D(printf("dropping %d; ", drop));
		if(drop < t->ffree) {
			bcopy(&t->fifo[drop], &t->fifo[0], t->ffree - drop);
			t->ffree -= drop;
		} else {
			t->ffree = 0;
		}
	}
	newchars = MIN(got, FIFO - t->ffree);
	D(printf("new %d\n", newchars));
	bcopy(&buf[got - newchars], &t->fifo[t->ffree], newchars);
	t->ffree += newchars;
	set_tkreq(dev, t);
}

/*
 * if there are any buffered chars, request new interrupt
 */
static void
set_tkreq(IODev *dev, TTY *t)
{
	t->tks |= 0200;
	if(t->tkint & 0100) {
		IRQ(dev, t->ireq);
		t->tkreq++;
	}
}

void
tty_reset(IODev *dev)
{
	int 	i;
	KL	*t = dev->data;

	for( i = 0 ; i < t->ntty ; i++ ) {
		t->ttys[i].ffree = 0;
		t->ttys[i].fifo[0] = 0;
		t->ttys[i].tks = 0;
		t->ttys[i].tkint = 0;
		t->ttys[i].tpint = 0;
		t->ttys[i].tkreq = t->ttys[i].tpreq = 0;
	}
	comp_req(dev);
}


ushort
tty_fetch(IODev *dev, unsigned a)
{
	KL	*k = dev->data;
	TTY	*t;
	ushort 	v;

	for(t = k->ttys; t < &k->ttys[k->ntty]; t++)
		if(CSR(a) == t->addr)
			goto found;
	printf("tty_fetch(%o)\n", a);
	Trap4(0100);
	return v;
found:
	switch(REG(a)) {

	case 0:		/* TKS */
		v = t->tks | t->tkint;
		break;

	case 2:
		v = t->fifo[0];
		t->tks = 0;
		if(t->ffree != 0)
			shift_fifo(dev, t);
		break;

	case 4:
		v = t->tpint | 0200;
		break;

	case 6:
		v = t->tpb;
		break;

	default:
		printf("tty_fetch(%o)\n", a);
		Trap4(0100);
	}

	return v;
}

void
tty_store(IODev *dev, unsigned a, int mode, ushort v)
{
	KL	*t = dev->data;
	TTY	*k;
	uchar c;
	int	n;

	for( n = 0 ; n < t->ntty ; n++ )
		if( CSR(a) == t->ttys[n].addr )
			goto found;
	printf("tty_fetch(%o)\n", a);
	Trap4(0100);
	return;
found:
	k = &t->ttys[n];

	switch(REG(a&~1)) {

	case 0:		/* TKS */
		if(mode & M_High)
			break;
		if(!k->tkint && (v & 0100) && k->tks) {
			IRQ(dev, k->ireq);
			k->tkreq++;
		}
		if(!(k->tkint = v & 0100) && k->tkreq) {
			k->tkreq = 0;
			comp_req(dev);
		}
		break;


	case 2:
		break;


	case 4:
		if(mode & M_High)
			break;
		if(!k->tpint && (v & 0100)) {
			IRQ(dev, k->ireq);
			k->tpreq++;
		}
		if(!(k->tpint = v & 0100) && k->tpreq) {
			k->tpreq = 0;
			comp_req(dev);
		}
		break;


	case 6:
		if(mode & M_High)
			break;
		c = v;
		write(k->fd, &c, 1);
		if(k->tpint) {
			IRQ(dev, k->ireq);
			k->tpreq++;
		}
		break;

	default:
		printf("tty_store(%o)\n", a);
		Trap4(0100);
	}
}


ushort
tty_vector(IODev *dev)
{
	ushort vec = 0;
	int	i;
	int	pending = 0;
	KL	*k = dev->data;
	int	irq = 0;

	for( i = 0 ; i < k->ntty ; i++ )
		if( k->ttys[i].tkreq) {
			if(!vec) {
				k->ttys[i].tkreq = 0;
				vec = k->ttys[i].tkvec;
				irq = k->ttys[i].ireq;
			}
			pending++;
		}

	for( i = 0 ; i < k->ntty ; i++ )
		if(k->ttys[i].tpreq) {
			if(!vec) {
				k->ttys[i].tpreq = 0;
				vec = k->ttys[i].tpvec;
				irq = k->ttys[i].ireq;
			}
			pending++;
		}
	dev->reqpri = 0;
	if( !pending )
		return 04;

	if(--pending)
		IRQ(dev, irq);
	return vec;
}

/*
 * if the receiver fifo was not empty, shift all cells down
 * and request new interrupt if nt empty after shifting
 */
static void
shift_fifo(IODev *dev, TTY *t)
{
	sigset_t nsig, osig;

	sigemptyset(&nsig);
	sigaddset(&nsig, SIGIO);
	sigprocmask(SIG_BLOCK, &nsig, &osig);

	if(--t->ffree > 0) {
		bcopy(&t->fifo[1], &t->fifo[0], t->ffree);
		set_tkreq(dev, t);
	}

	sigprocmask(SIG_SETMASK, &osig, NULL);
}

/*
 * recompute request word, this is needed if a pending
 * request is to be dropped by clearing the interrupt enable in the csr
 */
static void
comp_req(IODev *dev)
{
	KL *d = dev->data;
	TTY *k;

	dev->reqpri = 0;
	for(k = d->ttys; k < &d->ttys[d->ntty]; k++)
		if(k->tkreq || k->tpreq)
			IRQ(dev, k->ireq);
}


void
tty_info(IODev *dev)
{
	KL	*t = dev->data;
	TTY	*k;

	printf("KL serial line controller\n");
	printf("%d lines attached\n", t->ntty);

	printf("UNIT\tTKREQ\tTPREQ\tCSR\t\tVEC\n");

	for(k = t->ttys; k < t->ttys + t->ntty; k++)
		printf("%d\t%d\t%d\t%08o\t%03o/%d\n", (int)(k - t->ttys),
			k->tkreq, k->tpreq, k->addr, k->tkvec, k->ireq);
}
