#
/*
 *	l  remote_tty
 *
 *	peter i. & ian j.	jun '77
 *
 *  File exchange package for remote unices.
 *  Also has transparent mode for talking to 
 *   other operating systems as if local tty.
 *
 *  Needs to run at fairly high priority,
 *   so should be setuid to root.
 *
 *	Modified for flow control & ^q & buffered copy
 *	 Piers Lauder - Sydney University - Aug '77
 *
 *	Modified for binary file protocol
 *	 Piers Lauder - Sydney University - Sep '77
 *
 *	modified Nov '77 to use shared data for locking.
 */

#define	HIPRIORITY	-20
#define	C_FILEMODE	021	/* new substitute for ^d is ^q */

/* #define buffered_copy	1	/* buffer reads from ( possibly ) faster remote host in tty() */

/*
 * Regulate file transfer with a file protocol as follows:-
 *
 *  Blocks consist of a header followed by data followed by a trailer.
 *  The header contains a sequence number (0/1) and the size
 *   ( transmitted twice, 2nd as complement of first ).
 *  The trailer contains a longitudinal parity check of the data,
 *   followed by a sum check of the data.
 *
 *  Block format is
 *	STX SEQ SIZ ~SIZ ...DATA... LPC SUM
 *
 *  Blocks are acknowledged by a 2 byte reply consisting of
 *   ACK or NAK followed by the relevant sequence number.
 */

#define	REPLYZ		2
#define	RTYPE		0
#define	RSEQ		1

struct { char r_typ; char r_seq; } reply;
#define	MREPLY		&reply,REPLYZ

#define	HEADER		4
#define	BSEQ		1
#define	TRAILER		2
#define OVERHEAD	(HEADER+TRAILER)
#define	DATAZ		64
#define	BUFZ		(DATAZ+OVERHEAD)

/*
 *  Timeouts - these must be tailored to the line speed,
 *   the basic formulae used being:-
 *	receive timeout = receive time + turnaround time
 *	transmit   "    = read retrys * receive timeout
 *	( all times in seconds )
 */
#define	STARTTIMEOUT	10
#define	FLUSHTIMEOUT	1
#define	TURNAROUND	3

#define	XRETRYS		3
#define	RRETRYS		3
#define	EXTA		200	/* per DZ11 */
#define	EXTB		1920	/* per DZ11 */

int	byterate[16]	/* per DH11 */
{
  0,5,8,10,15,15,20,30,60,120,
  180,240,480,960,EXTA,EXTB
};

int	xtimeout, rtimeout;

#define	STX		2
#define	ETX		3	
#define	ACK		6	
#define	NAK		025

#define	SPECIALON	0340
#define	SPECIALOFF	037
#define	SOH		001	/*start of header*/
#define	EOT		004	/*end of text*/
#define	NL		012
#define	MSOH		"\01\01",2
#define	MEOT		"\04\04",2
#define	METX		"\03\03",2
#define	PROMPT		"* ",2
#define	INITMESG	"* tty (^q=quit)\n",16
#define	SIGTIMEOUT	15

#define	UNHOOKMSG	"wait for ripples to subside\n",28
#define	CASERR		"unknown command!\n",17
#define	CREATERR	"cant create local file!\n",24
#define	ACREATERR	"cant create remote file!\n",25
#define	OPENERR		"cant open local file!\n",22
#define	AOPENERR	"cant open remote file!\n",23
#define	ALREADYINUSE	"already in use!\n",16
#define	ARGERR		"arg should be remote special file\n",34
#define	ILLFUNK		illfunc,20
char	illfunc[]	"illegal function ??\n";
#define	MTIMEDOUT	"timed out\n",10
#define	MPROTOERROR	"protocol error\n",15
char	abort[]		"transfer aborted\n";

int	modes[16];
int	ttymode[3];
int	narg;
char	*cmd[10];
int	xfd;		/* fd for remote port */
char	otherm[]	"/dev/xxxx";
char	c;
int	uid;
long	filesize;

struct	err
{
  int	e_count;
  char	*e_mess;
}
  errs[]
{
#define	E_RETRYS	0
  { 0,	"retrys" },
#define	E_SEQUENCE	1
  { 0,	"seq" },
#define	E_SIZE		2
  { 0,	"size" },
#define	E_OUTSIZE	3
  { 0,	"outsize" },
#define	E_TIMEOUTS	4
  { 0,	"timeouts" },
#define	E_CRC		5
  { 0,	"crc" },
#define	E_WRITE		6
  { 0,	"local write" },
#define	E_NAKS		7
  { 0,	"naks" },
#define E_SYNC		8
  { 0,	"sync" }
};
#define	NERRS		((sizeof errs)/(sizeof errs[0]))

char	buf[BUFZ];



/****************************************************************/

main (argc,argv)
char	**argv;
int	argc;
{
	register  x;

	uid = ((x = getuid() & 0377)<< 8 ) | x;
	nice( HIPRIORITY );

	switch ( argv[0][0])
		{
		case 'l':
			l (argc,argv);
			break;
		case 's':
			send (argc,argv);
			break;
		case 'r':
			receive (argc,argv);
			break;
		}

	errors();
}

/*****************************************************************/

l (argc,argv)
char	**argv;
int	argc;
{
	register char	*arg;
	extern uselock;

	if (argc != 2)
			{
			write (2,ARGERR);
			exit(1);
			}
	arg = argv[1];
	otherm[5] = arg[0];
	otherm[6] = arg[1];
	otherm[7] = arg[2];
	otherm[8] = arg[3];

	catch();

	if ( ++uselock || (xfd = open (otherm, 2)) < 0)
		{
		write (2,ALREADYINUSE);
		exit (2);
		}
	raw (xfd);  set_timeouts();

	write (2,INITMESG);
	tty();

	for (;;)  {
			errors();

			switch(getline())
			{
			/* put a file to remote */
			case 'p':
				if (narg == 3)  { put();  continue; }
				break;
			/* get a file from remote */
			case 'g':
				if (narg == 3)  { get();  continue; }
				break;
			/* talk to remote as a terminal */
			case 't':
				if (narg == 1)  { tty();  continue; }
				break;
			/* stop this sillyness */
			case 's':
				if (narg == 1)  { finish();  continue; }
				break;
			/* unknown command */
			default:
				help ();
				continue;
			}

			write( 2 , ILLFUNK );
	}
}

/**************/
getline()
{
	static char	line[200];
	register char	*q;
	register	*p;
	register int	i;
	int		n;

	narg = 1;
	write(2,PROMPT);
	if (( n = read(2,line,sizeof line)) < 1) return('s');
	if( n==1 ) return(-1);
	c = line[0];
	p = &cmd[0];
	q = &line[0];
	*p++ = q;

	for (i = 1; ; i++)
		{
		if (line[i] == ' ')
			{
			line[i] = 0;
			narg++;
			*p++ = q + 1 + i;
			continue;
			}
		if (line[i] == '\n')
			{
			line[i] = 0;
			*p++ = 0;
			break;
			}
		}

	return(c);
}

/*******************/
cook(fd)
{
	gtty(fd,ttymode);
	if( modes[fd] ) ttymode[2] = modes[fd];
	/* set to default system values, ie map cr -> cr-lf, echo,
		map upper case to lower, echo tabs as spaces     */
	stty (fd,ttymode);
}

/********************/
raw(fd)
{
	gtty(fd,ttymode);
	if( !modes[fd] ) modes[fd] = ttymode[2];
	ttymode[2] =| SPECIALON;
	ttymode[2] =& ~SPECIALOFF;
	/* set for raw, even or odd parity - nothing else ! */
	stty (fd,ttymode);
}

/**************************/

set_timeouts()
{
  xtimeout = RRETRYS * (rtimeout = BUFZ / byterate[ttymode[0] & 0377] + TURNAROUND);
}

/*********************/
finish()
{
	cook(2);
	cook(xfd);
	exit(0);
}

/*********************/
catch()
{
	register	i;

	/* for interrupt, quit; tell remote, reset modes and exit */

	for (i = 2; i <= 3; i++)
		signal(i, finish);
}

/*************************************************************/

tty()
{
	register n;
	sleep(2);	/* let any previous message out before "raw" - piers - Aug '77 */
	raw(2);		/* get terminal to ignore the control
			   functions - we want them	*/
#ifndef buffered_copy
	if(!(n=fork())) copy(xfd,2); /* dump all other computer sends
					  to the terminal  */
#endif
#ifdef	buffered_copy
	if(!(n=fork())) bcopy(xfd,2);
#endif
	if ( !fork() ) copy(2,xfd);   /* dump all entered at terminal
					  to other computer  */
	wait();	/* first process to stop will be dump to 
		   other computer from terminal  */
/*	sleep(5);	/* let things kool off */
	kill(n,1);	/* stop the receiving process */
	wait();
	cook(2);
/*	raw(xfd);	*/
}

/*************************************************************/

copy(fromid,toid)
{
	register	i;
	for(i=0;;) {
		read(fromid,&c,1);
		if ( (c & 0177) == C_FILEMODE )
				exit();
		write(toid,&c,1);
	}
}

/*******************************************************************/

#ifdef	buffered_copy

int	bchild;

/*
 *	catch interrupt to end buffered copy
 */
bend()
{
	kill( bchild , 1 );
	wait();
	exit();
}

/*
 *	buffered copy - use pipes
 */
bcopy( fromid , toid )
{
	int	f[2];

	if ( pipe( f ) < 0 )	copy( fromid , toid );
	if ( !(bchild = fork()) )  copy( fromid , f[1] );
	signal( 1 , bend );
	copy( f[0] , toid );
}

#endif

/*************************************************************/

get()
{
	register	ofd;

	if( (ofd=creat(cmd[2],0600)) < 0 ) {
		write(2,CREATERR);
		return;
	};

	chown( cmd[2] , uid );
	transfer("send",cmd[1]);

	if ( sync( "open" ) )
		if ( rproto( ofd , xfd ) == 0 )
			printf2( "%s: %l bytes received\n" , cmd[2] , filesize );
		else
			printf2( abort );

	close(ofd);
}

/*************************************************************/

put()
{
	register ifd;
	register n;

	if( (ifd=open(cmd[1],0)) < 0 ) {
		write(2,OPENERR);
		return;
	}

	transfer("receive",cmd[2]);

	if ( sync( "create" ) )
		if ( xproto( ifd , xfd ) )
			printf2( abort );

	close (ifd);
}

/*******************************/

transfer(s1,s2)
char	*s1,*s2;
{
	register char	*p, *q;

	q = buf;

	*q++ = '@';

	for (p = s1; *q++ = *p++; );	*q++ = ' ';
	for (p = s2; *q++ = *p++; );	*q++ = '\n';

	write (xfd, buf, q-buf);
}


/*******************************/

send(argc,argv)
char **argv;
{
	register  fd;

	raw(2);  set_timeouts();

	if( (fd=open(argv[1],0)) > 0)
		{
		write(2,MSOH);
		sentinel();
		xproto( fd , 2 );
		}
	else
		write (2,METX);

	cook(2);
}

/*******************************/

receive(argc,argv)
char **argv;
{
	register	fd;

	raw(2);  set_timeouts();

	if( (fd=creat(argv[1],0600)) > 0)
		{
		write(2,MSOH);
		chown( argv[1] , uid );
		sentinel();
		rproto( fd , 2 );
		}
	else write(2,METX);

	flush( 2 );
	cook(2);
}

/********************************************************************/

int	timedout;

sentinel()
{
	signal( SIGTIMEOUT , sentinel );
	timedout++;
}


tread( f , b , s , timeout )
{
	clktim( timeout );
	timedout = 0;
	read( f , b , s );
	clktim( 0 );
	return( timedout );
}


flush( fd )
{
  while ( !tread( fd , &c , 1 , FLUSHTIMEOUT ) );
  return( 1 );
}

/********************************************************************/

sync( remote_err )
  char	*remote_err;
{
	sentinel();
	flush( xfd );

	do  {
		if ( tread( xfd , &c , 1 , STARTTIMEOUT ) )  {
			printf2( "no response from remote!\n" );
			return( 0 );
		}
		if( c == ETX ) {
			printf2( "can't %s remote file!\n" , remote_err );
			return( 0 );
		}
	} while( c != SOH);

	return( 1 );
}

/******************************************************************/

xproto( local , remote )
{
	register  n, retrys;

  flush( remote );
  buf[BSEQ] = 0;

  while ((n = read( local , buf+HEADER , DATAZ )) > 0 )  {

	crc( n );	retrys = 0;

	do  {
		write( remote , buf , n+OVERHEAD );

		while ( tread( remote , &reply , REPLYZ , xtimeout ) == 0 )  {
			while ( reply.r_typ != ACK && reply.r_typ != NAK )  {
				errs[E_SYNC].e_count++;
				reply.r_typ = reply.r_seq;
				if ( tread( remote , &reply.r_seq , 1 , xtimeout ) )
					goto break2;
			}
			if ( reply.r_seq == buf[BSEQ] )
				break;
		}
		break2:;
	}while
		(((timedout && ++errs[E_TIMEOUTS].e_count)
		  || (reply.r_typ != ACK && ++errs[E_NAKS].e_count)
		 )
		 && retrys++ < XRETRYS
		);

	errs[E_RETRYS].e_count =+ retrys;
	if ( retrys > XRETRYS )  {
		errs[E_RETRYS].e_count--;
		return( 1 );
	}
  }

  write( remote , MEOT );

  return( 0 );
}

/*******************************************************************/

/* protocol states */
#define	NEWBLOCK	0
#define	SEQUENCE	1
#define	SIZE1		2
#define	SIZE2		3
#define	STATES		4

/* case labels */
#define	newblock	0
#define	sequence	1
#define	size1		2
#define	size2		3

/* decision table */ /*
char	dtbl[STATES]
{
	newblock,
	sequence,
	size1,
	size2
}; */

rproto( local , remote )
{
	register  size, state, x;
	int  wretrys, lastsize, retrys;
	char  lastseq;

  state = NEWBLOCK;
  retrys = 0;
  reply.r_typ = NAK;
  reply.r_seq = 1;
  filesize = 0;
  lastseq = -1;

  for (;;)  {

	while ( tread( remote , &c , 1 , rtimeout ) )
		if ( retrys++ < RRETRYS )  {
			errs[E_TIMEOUTS].e_count++;
ack:
			write( remote , MREPLY );
		}else
			return( 1 );

	if ( (x = c & 0377) == EOT )  return( 0 );

	switch ( /*dtbl[*/ state /*]*/ )  {

	 case newblock:	if ( x == STX )  state = SEQUENCE;
			break;

	 case sequence:	if ( x & ~1 )  {
				errs[E_SEQUENCE].e_count++;
				state = NEWBLOCK;
fnak:
				flush( remote );
nak:
				reply.r_typ = NAK;
				goto ack;
			}
			reply.r_seq = x;
			state = SIZE1;
			break;

	 case size1:	state = SIZE2;
			if ( (size = x) > DATAZ )  {
				errs[E_OUTSIZE].e_count++;
				goto fnak;
			}
			break;

	 case size2:	state = NEWBLOCK;
			if ( size != (~x & 0377) )  {
				errs[E_SIZE].e_count++;
				goto fnak;
			}
			if ( (tread( remote , buf+HEADER , size+TRAILER , rtimeout ) && ++errs[E_TIMEOUTS].e_count)
				|| (crc( size ) && ++errs[E_CRC].e_count)
			   )
				goto nak;
			retrys = 0;
			wretrys = 0;
			if ( reply.r_seq == lastseq && lastsize > 0 )  {
				seek( local , -lastsize , 1 );
				filesize =- lastsize;
			}
			while ( (lastsize = write( local , buf+HEADER , size )) < size )  {
				errs[E_WRITE].e_count++;
				if ( wretrys++ >= XRETRYS )
					return( 1 );
			}
			filesize =+ size;
			reply.r_typ = ACK;
			lastseq = reply.r_seq;
			goto ack;
	}
  }
}

/********************************************************************/

crc( s )
{
	register char *p;
	register char lpc, sum;
	char  *end;
	int  error;

  p = buf;  lpc = 0;  sum = 0;  error = 0;

  *p++ = STX;
  *p++ =^ 1;		/* flip sequence number */
  *p++ = s;  *p++ = ~s;

  end = buf+HEADER+s;
  for ( ; p < end ; p++ )  {
	lpc =^ *p;
	sum =+ *p;
  }

  if ( lpc != *p )  error++;
  *p++ = lpc;
  if ( sum != *p )  error++;
  *p++ = sum;

  return( error );
}

/**********************************************************************/

errors()
{
	register struct err *ep;
	register nerrs;

  nerrs = 0;

  for ( ep = errs ; ep < &errs[NERRS] ; ep++ )
	if ( ep->e_count )  {
		if ( nerrs++ == 0 )
			printf2( "\nfile transfer protocol errors:-\n" );
		printf2( "\t%s: %d" , ep->e_mess , ep->e_count );
		ep->e_count = 0;
	}

  if ( nerrs )  printf2( "\n" );
  return( nerrs );
}

/***********************************************************************/

printf2( s , a )
  register char *s;
{
	register *p;
	long n;  struct { long *longp; };

  p = &a;

  while ( c = *s++ )  {
	if ( c == '%' )  switch ( *s++ )  {
		case 's':	printf2( *p++ );	/* u r warned */
				continue;
		case 'd':	n = *p++;  goto nn;
		case 'l':	n = *p.longp++;
		nn:		if (n == 0)  printf2( "0" );
				else
					printl2( n );
				continue;
	}
	write( 2 , &c , 1 );
  }

  return( s );
}


printl2( n )
  long n;
{
  if ( n == 0 ) return;  printl2( n/10 );  c = n%10+'0';  write( 2 , &c , 1 );
}

/************************************************************************/

help()
{
	write(2,"\n",1);
	write(2,"use:\n",5);
	write(2,"\tp[ut] local-file remote-file\n",30);
	write(2,"\tg[et] remote-file local-file\n",30);
	write(2,"\tt[ty]\n",7);
	write(2,"\ts[top]\n",8);
	write(2,"\n",1);
}
