#
/*
 *	driver for BURROUGHS TD800 POLL/SELECT communications protocol
 *
 *		    by
 *	       piers lauder
 *	 DEPT OF COMPUTER SCIENCE
 *	   UNIVERSITY OF SYDNEY
 *	      november  1977
 */


#include	"../param.h"
#include	"../user.h"
#include	"../buf.h"
#include	"../dl11.h"


/*
 *	tunable constants
 */
#define	READPRI		3		/* priority for sleep when reading */
#define	WRITEPRI	2		/* priority for sleep when writing */
#define	T_DELAY		2		/* b1726 turn around delay in 'tics' */
#define	REMOSECHO	0100000		/* set switch 15 if echoes expected */
#define	splbx		spl6		/* priority of controller */
/* following bytes should be even parity */
#define	AD1		'0'		/* protocol address 1 */
#define	AD2		'0'		/* protocol address 2 */
#define	WMN		'0'		/* message number for write */


struct
{
  char		flag;		/* user synchronisation */
  char		echoes;		/* hardware echoes expected */
  char		rstate;		/* receive state */
  char		wstate;		/* write state */
  char		rlpc;		/* recieve lpc */
  char		wlpc;		/* write lpc */
  struct buf	*rbp;		/* read buffer pointer */
  struct buf	*wbp;		/* write buffer pointer */
  char		*posn;		/* position of data in buffer */
  int		count;		/* data count */
/*char		ad1, ad2;	/* current address bytes */
  char		lmn;		/* last read message number */
  char		rmn;		/* read message number */
  char		wmn;		/* write message number */
  int		blks_sent;	/* blocks sent - statistics */
  int		blks_rcvd;	/* blocks recieved - statistics */
}
  bx;

/*
 *	flag bits
 */
#define	OPENR		1	/* open for reading */
#define	OPENW		2	/* open for writing */
#define	RD_DATA_RDY	4	/* data ready for reading */
#define	GET_RD_BLK	010	/* block needed for input */
#define	WTG_FOR_WRT	020	/* waiting to transfer data to output block */
#define	WRT_DATA_RDY	040	/* data ready for output */
#define	WCLOSE		0100	/* write null block on close */

/*
 *	significant bytes
 */
#define	SOH		1
#define STX		2
#define ETX		3
#define EOT		4
#define ENQ		5
#define ACK		6
#define NAK		025
#define FS		034
#define	POL		'p'
#define	SEL		'q'

/*
 *	significant bytes + parity
 */
#define	SOHp		0201
#define STXp		0202
#define ETXp		3
#define EOTp		0204
#define ENQp		5
#define ACKp		6
#define NAKp		0225
#define FSp		0234

/*
 *	decision table selectors
 */
#define Y_EOT		0
#define Y_default	1
#define Y_error 	2

/*
 *	define structure to log errors
 */
struct	error
{
	char	*e_mess;
	int	e_count;
}
	bxerr[]
{
  {	"resets", 0	},
#define	RESET		0
  {	"write errors",0},
#define	WERR		1
  {	"write parity",0},
#define	WPAR		2
  {	"recv bad", 0	},
#define	RECVBAD		3
  {	"recv lpc", 0	},
#define	RECVLPC		4
  {	"breaks", 0	},
#define	BREAK		5
  {	"overuns", 0	},
#define	OVERUN		6
  {	"recv parity", 0},
#define	PARITY		7
  0
};


char	partab[];



/*
 *	general open:
 *	- check exclusive use & device free,
 *	- set vectors.
 *	- enable read interrupts
 */
bxopen( FLAG )
  register FLAG;
{
	register char *port = &dl11[BP];
	extern bxrint(), bxwint();

  if ( bx.flag & FLAG || (*port && *port != COM) )  {
	u.u_error = ENXIO;
	return( 0 );
  }

  bx.flag =| FLAG;
  if ( !*port )  {
	dl11v[BP].rintad = bxrint;
	dl11v[BP].wintad = bxwint;
	BPADDR->dlrs = IENABLE;
	*port = COM;
  }
  return( 1 );
}


	/*********
	 * write *
	 *********/

/*
 *	open for writing and set up write block
 */
bxopenw()
{
  splbx();
	if ( bxopen( OPENW ) )  {
		if ( !bx.wbp )
			bx.wbp = getblk( NODEV );
		if ( !bx.wmn )
			bx.wmn = WMN;
	}
  spl0();
}


/*
 *	close off write and free write block
 */
bxwclose()
{
  splbx();
	bx.flag =& ~WTG_FOR_WRT;
	bx.flag =| WCLOSE;
	bxerrors();
  spl0();
}


/*
 *	write data to line
 */
bxwrite()
{
	register struct buf *bp = bx.wbp;
	register char *flag = &bx.flag;
	register count;

  do  {
	splbx();
		while ( *flag & WRT_DATA_RDY )  {
			*flag =| WTG_FOR_WRT;
			sleep( bp , WRITEPRI );
			*flag =& ~WTG_FOR_WRT;
		}
	spl0();

	if ( bp->b_wcount = count = min( 512 , u.u_count ) )  {
		if ( count =& 01776 )
			iomove( bp , 0 , count , B_WRITE );
		if ( bp->b_wcount & 1 )
			bp->b_addr[count++] = cpass();
		bxwparity( count );
		*flag =| WRT_DATA_RDY;
	}
  } while
	( u.u_count > 0 );
}


/*
 *	set parity bits on data in block and calculate lpc
 */
bxwparity( count )
  register count;
{
	register char *cp = bx.wbp->b_addr;
	register c;

  bx.wlpc = 0;
  do  {
	c = *cp & 0177;
	if ( partab[c] & 0100 )	/* illegal char */
		c = '?';
	c = (partab[c] & 0200) | c;
	bx.wlpc =^ c;
	*cp++ = c;
  } while
	( --count );
  bx.wlpc =^ bx.wmn^(AD1^AD2^STXp^ETXp);
}


	/********
	 * read *
	 ********/

/*
 *	open for reading
 */
bxopenr()
{
  splbx();
	bxopen( OPENR );
  spl0();
}


/*
 *	close off read
 */
bxrclose()
{
  splbx();
	bx.flag =& ~OPENR ;
	bx.lmn = 0;
	bxerrors();
  spl0();
}


/*
 *	read data from line
 */
bxread()
{
	register struct buf *bp = bx.rbp;
	register char *flag = &bx.flag;
	register count;
	int n;

  splbx();

  for (;;)  {
	if ( *flag & RD_DATA_RDY )  {
		spl0();
		if ( count = (n = min( bp->b_wcount , u.u_count )) & 01776 )
			iomove( bp , 0 , count , B_READ );
		if ( n & 1 )
			passc( bp->b_addr[count] );
		/* geterror( bp ); */
		splbx();
		*flag =& ~RD_DATA_RDY;
		if ( *flag & GET_RD_BLK )
			*flag =& ~GET_RD_BLK;
		else  {
			bx.rbp = 0;
			brelse( bp );
		}
		break;
	}

	if ( *flag & GET_RD_BLK )  {
		spl0();
		bx.rbp = bp = getblk( NODEV );
		splbx();
		*flag =& ~GET_RD_BLK;
	}

	sleep( flag , READPRI );
  }

  spl0();
}


/*
 *	report any errors
 */
bxerrors()
{
	register struct error *ep = bxerr;
	register errs = 0;

  do
  	if ( ep->e_count )  {
		if ( errs++ == 0 )
			printf( "\nb1726 err:" );
  		printf( " %s:%d" , ep->e_mess , ep->e_count );
  		ep->e_count = 0;
  	}
  while
  	( (++ep)->e_mess );

  if ( errs )
	putchar( '\n' );
}


	/**************
	 * interrupts *
	 **************/


/*
 *	deal with read interrupts
 */
bxrint()
{
	register c, x;
	register char *state = &bx.rstate;
	int y;
	extern bxstart(), bxone();

  x = BPADDR->dlrb;
  if ( bx.echoes > 0 )  {
	if ( !(SW->integ & REMOSECHO) )  bx.echoes = 0;
	else  {
		bx.echoes--;
		return;
	}
  }

  if ( x < 0 )  {
	y = Y_error;
	if ( x & 020000 )  bxerr[BREAK].e_count++;
	else  bxerr[OVERUN].e_count++;
  }else
	if ( ( x ^ partab[(c = x & 0177)] ) & 0200 )  {  bxerr[PARITY].e_count++;  y = Y_error;  }
	else
		if ( c == EOT )  y = Y_EOT;
		else  {
			y = Y_default;
			bx.rlpc =^ c;
		}

  goto in;

/* 0 */
next:	(*state)++;
null:	return;

rset:	bxerr[RESET].e_count++;
reset1:	*state = 1;
	return;

/* 1/8 */
ad1_c:	if ( c != AD1 )  goto idl;
	goto next;

/* 2/9 */
ad2_c:	if ( c != AD2 )  goto idl;
	goto next;

/* 3 */
polsel:	if ( c == POL )  goto next;
	if ( c != SEL )  goto idl;
select:	*state = 6;
	return;

/* 4 */
testx:	if ( c != ENQ )  {
idl:		*state = 0;
		return;
	}
	if ( (x = bx.flag) & WRT_DATA_RDY )  {
send:		bx.wstate = 0;
		timeout( bxstart , SOHp , T_DELAY );
		goto next;
	}else  if ( x & WCLOSE )  {
		bx.wbp->b_wcount = 0;
		bx.wlpc = bx.wmn^(AD1^AD2^STXp^ETXp);
		goto send;
	}else  {
		if ( !(x & OPENW) )  {
			if ( bx.wbp )  {
				brelse( bx.wbp );
				bx.wbp = 0;
			}
			if ( !(x & OPENR) )  {
				dl11[BP] = 0;
				BPADDR->dlrs = 0;
			}
		}
w_finish:	c = EOTp;
w_idle:		*state = 0;
	}
w_out:	timeout( bxone , c , T_DELAY );
	return;

/* 5 */
w_err:	bxerr[WERR].e_count++;
	goto reset1;
acknak:	if ( c == ACK )  goto sendok;
	if ( c != NAK )  goto w_err;
resend:	bxerr[WPAR].e_count++;
	*state = 4;
	goto send;
sendok:	bx.blks_sent++;
	bx.wmn =^ 0201;	/* flip seqn. no. ( & parity ) */
	if ( (x = bx.flag) & WTG_FOR_WRT )
		wakeup( bx.wbp );
	if ( x & WRT_DATA_RDY )
		bx.flag =& ~WRT_DATA_RDY;
	else  if ( x & WCLOSE )
		bx.flag =& ~(WCLOSE|OPENW);
	goto w_finish;

/* 6 */
testr:	if ( c != ENQ )  goto idl;
	if ( bx.flag & OPENR )  {
		if ( bx.rbp && !(bx.flag & RD_DATA_RDY) )  {
			c = ACKp;
			bx.posn = bx.rbp->b_addr;
			bx.count = 0;
			(*state)++;
			goto w_out;
		}else  {
			bx.flag =| GET_RD_BLK;
			wakeup( &bx.flag );
		}
	}
	c = NAKp;
	goto w_idle;

/* 7 */
soh:	if ( c != SOH )  goto idl;
	bx.rlpc = 0;
	goto next;

/* 10 */
xmn:	if ( c != '0' && c != '1' )  goto idl;
	bx.rmn = c;
	goto next;

/* 11 */
stx:	if ( c != STX )
		goto idl;
	goto next;

/* 12 */
recv:	if ( c == ETX )  goto next;
	if ( bx.count++ < 512 )
		*bx.posn++ = c;
	return;
rerr:	bxerr[RECVBAD].e_count++;
	return;

/* 13 */
lpc:	if ( !bx.rlpc )  {
		if ( bx.rmn != bx.lmn )  {
			bx.blks_rcvd++;
			bx.lmn = bx.rmn;
			bx.rbp->b_wcount = bx.count;
			bx.flag =| RD_DATA_RDY;
			wakeup( &bx.flag );
		}
		c = ACKp;
		goto w_idle;
	}
	bxerr[RECVLPC].e_count++;
	c = NAKp;
	goto w_idle;


in:
  {
	static *bxrm_c[][3]
	{
	/*        EOT   ?   BAD  */
	/* 0  */ next,null,null,
	/* 1  */ rset,ad1_c,idl,
	/* 2  */ rset,ad2_c,idl,
	/* 3  */ rset,polsel,idl,
	/* 4  */ rset,testx,idl,
	/* 5  */ w_err,acknak,idl,
	/* 6  */ rset,testr,idl,
	/* 7  */ rset, soh, idl,
	/* 8  */ rset,ad1_c,idl,
	/* 9  */ rset,ad2_c,idl,
	/* 10 */ rset, xmn, idl,
	/* 11 */ rset, stx, idl,
	/* 12 */ rset,recv,rerr,
	/* 13 */  lpc, lpc, lpc
	};

	goto bxrm_c[*state][y];	/* yes folks - a computed goto */
  }
}


/*
 *	transmit 1 char & start write interrupts
 */
bxstart( c )
{
  BPADDR->dlrs = 0;
  BPADDR->dlwb = c;
  BPADDR->dlws = IENABLE;
}



/*
 *	transmit 1 char
 */
bxone( c )
{
	register  x;

  BPADDR->dlwb = c;
  x = BPADDR->dlrb;
  bx.echoes = 1;
}


/*
 *	deal with write interrupts
 */
bxwint()
{
	register c, x;

  switch ( bx.wstate )  {
	case 0:	c = AD1; break;
	case 1: c = AD2; break;
	case 2:	c = bx.wmn; break;
	case 3: c = STXp;
		bx.count = bx.wbp->b_wcount;
		bx.posn = bx.wbp->b_addr;
		break;
	case 4: if ( bx.count-- == 0 )  {
			c = ETXp; break;
		}
		BPADDR->dlwb = *bx.posn++;
		return;
	case 5: c = bx.wlpc;
		bx.echoes = 2;
		x = BPADDR->dlrb;
		BPADDR->dlws = 0;
		BPADDR->dlrs = IENABLE;
  }

  bx.wstate++;
  BPADDR->dlwb = c;
}
