#
/*
 *	line driver for B1726 9600bd async protocol
 *
 *		    by
 *	       piers lauder
 *	DEPT. OF COMPUTER SCIENCE
 *	   UNIVERSITY OF SYDNEY
 *		july  1977
 */

#include	"../param.h"
#include	"../user.h"
#include	"../dl11.h"

char	partab[];

/*
 *	tunable constants
 */
#define BPRIORITY	8		/* priority for sleep */
#define T_DELAY 	2		/* turnaround delay in ticks */
#define RETRYS		10		/* for message retransmissions */
#define NBUF		2		/* double buffered */
#define DATASIZ 	512		/* max. no. of data bytes in blocks transmitted to B1726 ( even ) */
#define WAIT		60		/* timeout in seconds for read from b1726 */
#define	REMOSECHO	0100000		/* set switch 15 if echos expected */

#include	"bprot.h"


struct	Bp			/* data stucture for driver */
{
	int	state;			/* protocol state */
	int	flag;			/* driver flag */
	int	outcount;		/* count for echoes */
	int	waitcount;		/* count of seconds while waiting for read */

	struct	buf	*u_p, *d_p;	/* pointers for Bpwrite & Bpread, Bpstatemc */

	struct	buf	buffers[NBUF];	/* the buffers */
} Bp;

/*
 *	errors
 */
#define RETRY		0
#define LPC		1
#define PARITY		2
#define BREAK		3
#define EOT		4
#define OVERUN		5
#define SIZE		6

struct	error	Bperr[]
{
	"retrys",	0,
	"lpc",		0,
	"parity",	0,
	"break",	0,
	"eot",		0,
	"overun",	0,
	"size",		0,
	0
};




/*
 *	open link
 */
Bpopen( dev , writeflag )
{
	register struct buf *bp,*cp;
	extern Bprint(),Bpwint(),Bptime();

  if ( dl11[BP] )  {  u.u_error = ENXIO;  return;  }	/* exclusive use */

  dl11[BP] = COM;
  dl11v[BP].rintad = Bprint;
  dl11v[BP].wintad = Bpwint;

  Bp.flag  set	( OPEN | (writeflag ? WRITE : READ) );	/* open for reading OR writing - half dulplex */
  Bp.state = IDLE;

  bp = &Bp.buffers[0];

  do {
	cp = bp;
	cp->b_flag = 0;
	cp->b_count = 0;
	cp->b_lpc = 0;
	cp->b_retry = 0;
	cp->b_next = ++bp;
  } while
	( bp < &Bp.buffers[NBUF] );

  cp->b_next = &Bp.buffers[0];

  Bp.u_p = Bp.d_p = cp;

  BPADDR->dlrs set IENABLE;

  spl6();
	if ( ( Bp.flag & TIMING ) == 0 )  {
		timeout( Bptime , IDLE , HZ );
		Bp.flag set TIMING;
	}
  spl0();
}



/*
 *	handle 1 sec timouts on line
 */
Bptime( arg )
{
  if ( Bp.flag & OPEN )  {
	if ( (Bp.flag & READ) == WRITE )  {  if ( Bp.state == arg )  Bpstatemc(Y_timout);  }
	else
		if ( (++Bp.waitcount >= WAIT) && (Bp.u_p->b_flag & B_ASLEEP) )  {
			Bp.flag set WCLOSE;
			Bp.u_p->b_flag reset B_ASLEEP;
			wakeup( Bp.u_p );
		}
	timeout( Bptime , Bp.state , HZ );
  }else
	Bp.flag reset TIMING;
}



/*
 *	close link
 */
Bpclose( dev )
{
	register struct buf  *bp;
	register struct error *ep;

  if ( ( Bp.flag & READ ) == WRITE )  {
	bp = Bp.u_p;
	Bpxblk( 1 );
	spl6();
		Bp.flag set XCLOSE;
		sleep( &Bp.flag , BPRIORITY );
	spl0();
	if ( bp->b_flag & B_ERR )
		u.u_error = EIO;
  }

  Bp.flag =& TIMING;
  BPADDR->dlrs reset IENABLE;
  dl11[BP] = 0;

  if ( Bperr[RETRY].e_count )  {
	printf( "\nb1726 err\n" );
	ep = Bperr;
	do
		if ( ep->e_count )  {
			printf( "%s %d\n" , ep->e_mess , ep->e_count );
			ep->e_count = 0;
		}
	while
		( (++ep)->e_mess );
  }
}




/*
 *	read from link
 */
Bpread()
{
	register struct buf *bp;
	register char *cp;
	register c;
	static oldblock;

  bp = Bp.u_p;
  cp = &bp->b[bp->b_count];

  do  {
	if ( oldblock == 0 )  {
		Bp.waitcount = 0;
		spl6();
			while ( ( bp->b_flag & B_READ ) == 0 )
				if ( Bp.flag & (RCLOSE|WCLOSE) )  {
					spl0();
					return; 		/* EOF or timout */
				}else  {
					bp->b_flag set B_ASLEEP;
					sleep( bp , BPRIORITY );
				}
		spl0();
		if ( bp->b_flag & B_ERR )  {
			u.u_error = EIO;
			return;
		}
		bp->b_size = bp->b_count;
		bp->b_count = 0;
		cp = bp->b;
		oldblock++;
	}
	c = *cp++ & 0177;
	if ( ++bp->b_count == bp->b_size )  {
		bp->b_count = 0;
		bp->b_lpc = 0;
		bp->b_retry = 0;
		bp->b_flag = 0;
		Bp.u_p = bp = bp->b_next;
		oldblock = 0;
		if ( c == FS )  {	/* EOF */
			Bp.flag set RCLOSE;
			return;
		}
	}
  }while
	( passc( c ) >= 0 );
}



/*
 *	write to link
 */
Bpwrite()
{
	register struct buf *bp;
	register char *cp;
	register c;

  bp = Bp.u_p;
  cp = &bp->b[bp->b_count];

  while ( ( c = cpass() ) >= 0 )  {
	c =& 0177;
	if ( ( c<040 && (c>015 || c<7) ) || c==0177 )  c = '?';
	c set ( partab[c] & 0200);
	*cp++ = c;
	bp->b_lpc =^ c;

	if ( ++bp->b_count == DATASIZ )  {
		bp = Bpxblk( 0 );
		if ( u.u_error )  break;
		cp = bp->b;
	}
  }
}



/*
 *	initiate message transmission
 */
struct buf *Bpxblk( close )
{
	register struct buf *bp;
	register char *cp;

  bp = Bp.u_p;
  cp = &bp->b[bp->b_count];

  if ( close )  {
	spl6();
		if ( bp->b_flag & B_XMIT )  {	/* killed */
			bp->b_flag = 0;
			bp->b_lpc = 0;
			bp->b_count = 0;
			cp = bp->b;
		}
	spl0();
	*cp++ = FSp;
	bp->b_lpc =^ FSp;
	bp->b_count++;
  }

  *cp++ = ETXp;
  *cp = bp->b_lpc ^ ETXp;
  bp->b_size = bp->b_count+2;
  bp->b_count = 0;
  bp->b_retry = 0;

  bp->b_flag = B_XMIT;
  spl6();
	if ( ( Bp.flag & RUN ) == 0 )
		Bpstatemc(Y_timout);
  spl0();

  bp = bp->b_next;
  Bp.u_p = bp;

  spl6();
	while ( bp->b_flag & B_XMIT )  {
		bp->b_flag set B_ASLEEP;
		sleep( bp , BPRIORITY );
	}
  spl0();

  if ( bp->b_flag & B_ERR )  u.u_error = EIO;

  bp->b_flag = 0;
  bp->b_count = 0;
  bp->b_lpc = 0;
  return( bp );
}



/*
 *	deal with read interrupts
 */
Bprint()
{
	register c,x,y;

  x = BPADDR->dlrb;
  if ( Bp.outcount > 0 )  {
	if ( !(SW->integ & REMOSECHO) )  Bp.outcount = 0;
	else
		Bp.outcount--;
	return;
  }

  if ( x < 0 )	if ( x & 020000 )  {  Bperr[BREAK].e_count++;	y = Y_break;  }
		else  {  Bperr[OVERUN].e_count++;  y = Y_overun;  }
  else
	if ( ( x ^ partab[(c = x & 0177)] ) & 0200 )  {  Bperr[PARITY].e_count++;  y = Y_parity;  }
	else  {
		y = Y_default;
		if ( c < ' ' )  switch ( c )  {
					case STX:	y = Y_STX;  break;
					case ETX:	y = Y_ETX;  break;
					case EOT:	y = Y_EOT;  break;
					case ENQ:	y = Y_ENQ;  break;
					case ACK:	y = Y_ACK;  break;
					case NAK:	y = Y_NAK;  break;
				}
	}

  Bpstatemc( y , x );
}



/*
 *	deal with write interrupts
 */
Bpwint()
{
  Bpstatemc( Y_xmit );
}



/*
 *	message protocol state machine
 */
Bpstatemc( yy , cc )
{
	register struct buf *bp;
	register c,*state;
	extern  Bpxone(), Bpstart();

  bp = Bp.d_p;
  state = &Bp.state;

  switch ( Bp_dtbl[*state][yy] )  {

	case NULL:
		return;

	case ACK_R1:
		if ( ((Bp.flag & READ) == WRITE) || bp->b_flag & B_READ || Bp.flag & RCLOSE )  {
rreset:
			*state = IDLE;
			c = NAKp;
			break;
		}
		*state = RECV1;
	case ACK_:
		c = ACKp;
		break;

	/* TRANSMITTER */

	case e_xr_ID:
		Bperr[EOT].e_count++;
	case xr_IDLE:
		Bperr[RETRY].e_count++;
		if ( ++bp->b_retry <= RETRYS )	{  bp->b_count = 0;  goto retry;  }
		bp->b_flag set B_ERR;
	case ok_IDLE:
		bp->b_flag reset B_XMIT;
		if ( bp->b_flag & B_ASLEEP )  {  bp->b_flag reset B_ASLEEP;  wakeup(bp);  }
		bp = bp->b_next;
		Bp.d_p = bp;
	case _IDLE:
idle:
		*state = IDLE;
	case t_XFLAG:
		if ( bp->b_flag & B_XMIT )  {
retry:
			*state = XMIT1;
			Bp.flag set RUN;
	case ENQ_:
			c = ENQp;
			break;
		}

		Bp.flag reset RUN;
		if ( Bp.flag & XCLOSE )  {
			c = EOTp;
			Bp.flag reset XCLOSE;
			wakeup(&Bp.flag);
			break;
		}
		return;

	case STX_X2:
		*state = XMIT2;
		timeout( Bpstart , STXp , T_DELAY );	/* start block transmission after delay */
		return;

	case _X2:
		BPADDR->dlwb = bp->b[bp->b_count];
		if ( ++bp->b_count == bp->b_size )  {
			*state = XMIT3;
			Bp.outcount = 2;		/* echoes expected */
			c = BPADDR->dlrb;		/* flush input */
			BPADDR->dlrs set IENABLE;
			BPADDR->dlws reset IENABLE;
		}
		return;

	/* RECEIVER */

	case lpc_R2:
		*state = RECV2;
		return; 		/* lpc & count already set to zero by Bpread */

	case e_byte:
	case byte_:
		if ( bp->b_count >= DATASIZ )  Bperr[SIZE].e_count++;
		else {
			c = cc;
			bp->b_lpc =^ c;
			bp->b[bp->b_count] = c;
			bp->b_count++;
		}
		return;

	case _R3:
		bp->b_lpc =^ cc;
		*state = RECV3;
		return;

	case lpc_IDL:
		if ( cc != bp->b_lpc )	{
			Bperr[LPC].e_count++;
	case r_IDLE:
recverr:
			Bperr[RETRY].e_count++;
			if ( ++bp->b_retry <= RETRYS )	{
				bp->b_count = 0;
				bp->b_lpc = 0;
				goto rreset;
			}
			bp->b_flag set B_ERR;
		}
		bp->b_flag set B_READ;
		if ( bp->b_flag & B_ASLEEP )  {
			bp->b_flag reset B_ASLEEP;
			wakeup( bp );
		}
		Bp.d_p = bp->b_next;
		*state = IDLE;
		c = ( bp->b_flag & B_ERR ? NAKp : ACKp );
		break;

	/* EOT */

	case END:
		if ( Bp.flag & READ )  {
			Bp.flag set RCLOSE;
			wakeup( Bp.u_p );
		}
		return;

	case e_IDLE:
		Bperr[EOT].e_count++;
		goto idle;

	case e_r_IDL:
		Bperr[EOT].e_count++;
		goto recverr;
  }

  timeout( Bpxone , c , T_DELAY );		/* send char after delay */
}



/*
 *	transmit 1 char & start write interrupts
 */
Bpstart( c )
{
  BPADDR->dlrs reset IENABLE;
  BPADDR->dlwb = c;
  BPADDR->dlws set IENABLE;
}



/*
 *	transmit 1 char
 */
Bpxone( c )
{
	register  x;

  BPADDR->dlwb = c;
  x = BPADDR->dlrb;
  Bp.outcount = 1;
}
