#
/* 
        Floppy disk driver for RX02
        E.G. Keizer
        Computer Graphics Departement
        KUN
        Nijmegen
        Holland
*/

#include "../h/param.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"

#define DXADDR  0177170         /* address of first register */
#define MAXTRCOUNT 5000         /* give it some time, but not forever */
#define NTRACK  77
#define NSECT   26
#define SECSIZE 64

/* some useful values */
#define TRUE    1
#define FALSE   0

/* control & status register layout */
#define GO      1

#define FILL    0
#define EMPTY   02
#define WSECT   04
#define RSECT   06
#define FORMAT  010
#define RSTATUS 012
#define WDELDAT 014
#define RERRCOD 016
#define UNIT1   020
#define DONE    040
#define IENABLE 0100
#define TR      0200
#define DDENS   0400
#define INIT    040000
#define ERROR   0100000

/* error register layout */
#define ERCRC   01              /* CRC error, retry transfer */
#define ERINITD 04              /* not an error, initialize done */
#define ERACLO  010             /* AC low */
#define ERDENS  020             /* incorrect density, retry */
#define ERDRDEN 040             /* not an error, drive density */
#define ERDELD  0100            /* not an error, deleted data flag */
#define ERDREADY 0200           /* not an error, drive ready */
#define ERUNIT  0400            /* not an error, unit */
#define ERWCOVFL 02000          /* word count overflow, should not occur, retry */
#define ERNXM   04000           /* non existent memory, bus timeout, retry */

struct dx11 {
        char    x_status ;      /* status indicator */
        char    x_map ;         /* current mapping */
        char    x_unit ;        /* unit number */
        char    x_dens ;        /* bit array of densities */
        int     x_erreg ;       /* copy of error register */
        char    x_track, x_sect;/* track and sector */
        union {
                long    x_busaddr;
                unsigned int x_addr[2];
        } x_ad;
        unsigned x_wcount ;     /* # of words to transfer */
        unsigned x_offset ;     /* current logical sector (unit 64 words ) */
} dx11;
struct buf dxtab ;
int     dxnlo ;

struct {
        int     rx2cs ;         /* control and status */
        int     rx2db ;         /* data buffer */
} ;
struct {
        char    rx2lcs ;        /* to access rx2cs as a byte */
} ;

/* possible values of d_active */
#define XREAD   1               /* read in progress */
#define XWRITE  2               /* write in progress */
#define XFMT    3               /* format or read status in progress */

/* bits in minor device no to indicate mapping */
#define MAPBITS 0300
#define M_STR   0               /* staightforward mapping */
#define M_DEC   0100            /* Digital Equipment's mapping */

/* status bits in dx11.x_status */
#define XS_BUFOK 01             /* buffer has valid contents */
#define XS_FMTW 02              /* somebody waits for format/read status */
#define XS_FATAL 04             /* fatal error, stop processing this buffer */
#define XS_RECAL 010            /* Hard error occured init dx was given */

/************************************************************************/

dxstrategy(bp)
register struct buf *bp ;
{

#ifdef UNIBMAP
        if(bp->b_flags&B_PHYS)
                mapalloc(bp);
#endif UNIBMAP
        bp->av_forw=0 ;
        spl5() ;
        if ( dxtab.b_actf==NULL ) {
                dxtab.b_actf=bp ;
        } else {
                dxtab.b_actl->av_forw= bp ;
        }
        dxtab.b_actl=bp ;
        if ( dxtab.b_active== NULL ) dxstart() ;
        spl0() ;
}

dxstart()
{
        /* must be called at interrupt priority level */
        register struct buf *bp ;
        register int unit ;
        register int command;
        int wcount;

start:
        bp=dxtab.b_actf ;
        if ( dxtab.b_active==NULL ) {
                if (dx11.x_status&XS_FMTW) {
                        /* transfer control */
                        wakeup((caddr_t)&dx11) ;
                        return ;
                }
                /* fill dx11 structure */
                if ( bp==NULL ) return ;
                dxtab.b_active=(bp->b_flags&B_READ ? XREAD : XWRITE ) ;
                dx11.x_status &= ~(XS_BUFOK|XS_FATAL|XS_RECAL) ;
                dx11.x_offset= (short)bp->b_blkno*4 ;
                dx11.x_ad.x_addr[1]=bp->b_un.b_addr ;
                dx11.x_wcount= (bp->b_bcount>>1) ;
                dx11.x_ad.x_addr[0]=bp->b_xmem&03 ;
                /* disassemble device no */
                dx11.x_map=(minor(bp->b_dev))&MAPBITS ;
                dxtab.b_errcnt=0 ;
                dx11.x_unit=(minor(bp->b_dev))& ~MAPBITS ;
        }
        if ( dx11.x_status&XS_FATAL || dx11.x_wcount==0 ) {
                if ( dx11.x_status&XS_FATAL ) {
                        bp->b_flags |= B_ERROR ;
                }
                bp->b_resid= -dx11.x_wcount ;
                dxtab.b_actf=bp->av_forw ;
                iodone(bp) ;
                dxtab.b_active= NULL ;
                goto start ;
        }
        if ( ((dx11.x_status&XS_BUFOK)!=0) ^ (dxtab.b_active==XREAD) ) {
                /* if READ & BUFFER not ok 
                        or
                      WRITE & BUFFER ok
                   do io
                   else do dma
                */
                if ( !dxaddr() ) {
                        bp->b_error=ENXIO ;
                        dx11.x_status |= XS_FATAL ;
                        goto start ;
                }
                command= (dxtab.b_active==XREAD ? RSECT : WSECT ) ;
                command |= IENABLE|GO|(dx11.x_unit<<4) ;
                if ( (dx11.x_dens>>dx11.x_unit)&1) command |= DDENS ;
                DXADDR->rx2cs=command ;
                if ( dxld(dx11.x_sect) ) dxld(dx11.x_track) ;
        } else {
                command= (dxtab.b_active==XREAD ? EMPTY : FILL ) ;
                command |= IENABLE|GO|(dx11.x_ad.x_addr[0]<<12) ;
                if ( (dx11.x_dens>>dx11.x_unit)&1) command |= DDENS ;
                wcount=min(dx11.x_wcount,( command&DDENS ? SECSIZE*2 : SECSIZE ) ) ;
                DXADDR->rx2cs=command ;
                if ( dxld(wcount) )  dxld(dx11.x_ad.x_addr[1]) ;
        }
}

dxaddr()
{
        /* tries to map the current offset, tries double density if
            single does not satisfy
            called just once; from dxstart
        */
        if ( dx11.x_unit>1 ) return(FALSE) ;
        if ( !dxmap() ) {
                if ( (dx11.x_dens>>dx11.x_unit)&1 ){
                        /* double density already  */
                        return(FALSE) ;
                }
                dx11.x_dens |= 1<<dx11.x_unit ; /* set double density */
                return(dxmap()) ;
        }
        return(TRUE) ;
}
/* mapping function, returns true if mapping possible */
/*      called twice; both times from 'dxaddr'        */

int dxmap()
{
        register blkno, sector, density ;

        blkno=dx11.x_offset ;
        density=(dx11.x_dens>>dx11.x_unit)&1 ;

        if ( density ) {
                /* double density */
                blkno >>= 1 ;
        }
        switch(dx11.x_map) {
        case M_STR :
                /* straigthforward mapping */
                if ( blkno>=NSECT*NTRACK ) {
                        return(FALSE) ;
                }
                dx11.x_track=blkno/NSECT ;
                dx11.x_sect=blkno%NSECT+1 ;
                break ;
        case M_DEC :
                if ( blkno>=NSECT*(NTRACK-1) ) {
                        return(FALSE) ;
                }
                dx11.x_track=blkno/NSECT ;
                sector=(blkno%NSECT)<<1 ;
                if ( sector>=NSECT ) sector += 1 ;
                dx11.x_sect= ( sector+(6*dx11.x_track) )%NSECT + 1 ;
                dx11.x_track++ ; /* leave track 0 alone */
                break ;
        default :
                return(FALSE) ;
        }
}
/******************************************************************

        Interrupt routine

******************************************************************/

dxintr()
{
        register error,density,unit ;

        dx11.x_erreg = error = DXADDR->rx2db ;

        if ( dxtab.b_active == XFMT ) {
                wakeup((caddr_t)&dx11) ;
                return ;
        }

        unit=dx11.x_unit ;
        if ( dx11.x_status&XS_RECAL ) {
                if ( ! (error&ERDREADY) ) {
                        /* drive not ready, announce */
                        printf("DX%d not ready\n",unit) ;
                        dx11.x_status |= XS_FATAL ;
                }
                dx11.x_status &= ~XS_BUFOK ;
        } else {
                if ( error&ERDENS ) {
                        dx11.x_status &= ~XS_BUFOK ;
                        dx11.x_dens ^= 1<<unit ;
                }

                if ( DXADDR->rx2cs<0 || error&ERINITD ) {
                        /* error !!! */
                        if ( dxtab.b_errcnt++>=10 ) {
                                dx11.x_status |= XS_FATAL ;
                        }
                if ( (error&(ERINITD|ERCRC|ERDENS|ERWCOVFL|ERNXM))!=ERDENS ||
                                dx11.x_status&XS_FATAL ) {
                                /* if only ERDENS asserted dont give a message,
                                   unless it is the tenth time */
                                deverror(dxtab.b_actf,DXADDR->rx2cs,error) ;
                        }
                        if ( (error&(ERCRC|ERDENS|ERWCOVFL|ERNXM)) == 0 ) {
                                /* must be seek error */
                                dxinit(unit) ;
                                return ;
                        }
                } else {
                        /* command executed o.k. */
                        /* BUFOK indicates phase action went through */
                        if ( dx11.x_status&XS_BUFOK ) {
                                density= (dx11.x_dens>>unit)&1 ;
                                dx11.x_offset += 1<<density ;
                                dx11.x_wcount -=
                                        min(dx11.x_wcount,SECSIZE<<density) ;
                                dx11.x_ad.x_busaddr += SECSIZE*2<<density;
                        }
                        dx11.x_status ^= XS_BUFOK ;
                }
        }
        dxstart() ;
}

/******************************************************************

        service subroutines; put a word to the unit's dataregister
        and initialise the controller.

******************************************************************/

int dxld(word)
{
        register int count ;
        for ( count=MAXTRCOUNT ; count>0 ; count-- ) {
                if ( DXADDR->rx2lcs<0 ) {
                        DXADDR->rx2db=word ;
                        dxnlo=MAXTRCOUNT-count ;
                        return(TRUE) ;
                }
        }
        printf("RX02 TR timeout\n") ;
        /* initialize the controller */
        dxinit(dx11.x_unit) ;
        return(FALSE) ;
}

dxinit(unit)
{
        /* the GO and FUNCTION bits read back a zeroes */
        DXADDR->rx2cs= (unit<<4)|RSTATUS|GO|IENABLE ;
        dx11.x_status |= XS_RECAL ;
}

/*********************************************************************

        The character device ('raw') interface.
        No real raw device service, just entries for
        diskette formatting (dxfmt; the driver's 'write' routine, and
        diskette density lookup (dxstat; the device's 'read' routine.

*********************************************************************/

/* all sleeps are in negative priority, otherwise synchronization
   would be impossible between the block & character devices */
dxfmt(dev)
dev_t dev;
{
        /* is write call to character floppy device.
           Does the following:
                checks whether ONE character is passed.
                formats the floppy
                        in single density if it is a I.
                        in double density if it is a D.

                gives the error EFAULT in all other occasions,
                except when the controller indicates an error while
                formatting in which case EIO is given.
        */

        register int type, command ;

        if ( !dxgrab(dev) ) return ;

        command= IENABLE|GO|FORMAT ;
        if ( (minor(dev))&1 ) command |= UNIT1 ;
        type=cpass() ;
        if ( u.u_count!=0 ) {
                u.u_error=EFAULT ;
        }
        switch ( type ) {
        case 'D' :
                /* double density */
                type= 'I' ;
                command |= DDENS ;
        case 'I' :
                break ;
        default :
                u.u_error=EFAULT ;
        }
        if (!u.u_error) {
                DXADDR->rx2cs=command ;
                if ( !dxld(type) ) u.u_error=EIO ;
                while(!(DXADDR->rx2cs&DONE)) {
                        sleep((caddr_t)&dx11,PRIBIO) ;
                }
                if ( DXADDR->rx2cs<0 || dx11.x_erreg&ERINITD ) {
                        u.u_error=EIO ;
                }
        }
        dxfree() ;
}

dxstat(dev)
dev_t dev;
{
        /* read routine from character floppy device.
                does a maintenance read status.
                This is about the only way to detect the density.
        */

        register int command ;
        register char *ptr ;

        if ( !dxgrab(dev) ) return ;

        command= IENABLE|GO|RSTATUS ;
        if ( minor(dev)&1 ) command |= UNIT1 ;
        DXADDR->rx2cs= command ;
        while(!(DXADDR->rx2cs&DONE)) {
                sleep((caddr_t)&dx11,PRIBIO) ;
        }
        for ( ptr= &dx11.x_erreg ; ptr< &dx11.x_erreg+1 ; ptr++) {
                if ( passc(*ptr) < 0 ) break ;
        }
        dxfree() ;
}

int dxgrab(dev)
dev_t dev;
{
        if ( (minor(dev)&0377) > 1 ) {
                u.u_error=ENXIO ;
                return(FALSE) ;
        }
        spl5() ;
        while(dxtab.b_active != NULL ) {
                dx11.x_status |= XS_FMTW ;
                sleep((caddr_t)&dx11,PRIBIO) ;
        }
        dxtab.b_active=XFMT ;
        dx11.x_status &= ~XS_FMTW ;
        return(TRUE) ;
}

dxfree()
{
        /* this spl0() must be HERE because spurious interrupts
           now only cause a wakeup and not admin for the next i/o request
        */
        spl0() ;
        dxtab.b_active=NULL ;
        dxstart() ;
}
