#
/*
        Driver for (Nijmegen modified) DR11C
        connected back to back, to allow file shipping
        between machines.
        Single unit version.
        H.J. Thomassen, Nijmegen Univ. okt.1981

        Reader and writer are two separate devices, with
        separate special files each.
        System entries for read: drropen, drrclose, drread, (interrupt) drrint.
                      for write: drwopen, drwclose, drwrite, (interrupt) drwint.
        There is total symmetry between reader and writer at opposite sides;
        this driver may be tested with a round-loop cable on a single dr11c.
*/
#define ELECTR_1120 1

#ifdef VAX_INFORM
        /* the VAX link; Unibus 170530; vector 370/374 */
#define DRADDR 0170530
#endif VAX_INFORM
#ifdef ELECTR_1120
        /* the link to the electronica 11/20; Unibus 170600; vector 410/414 */
#define DRADDR 0170600
#endif ELECTR_1120
#ifdef M68000
        /* the link to the mc68000; Unibus 0170650; vector 510/514*/
#define DRADDR 0170650
#endif M68000
struct { int csr, out, in; }; /* hardware register template */
#include "../h/conf.h"
#include "../h/param.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/buf.h"

struct dr {
        int     state;
        char    *bptr; /* pointer to buffer header */
        union {
                int     *integ;
                char    *byt;
        }cptr; /* pointer into buffer */
        int     fcnt;  /* buffer fill count (words) */
        int     tcnt;  /* user transfer count */
        int     polcnt; /* polling loop acceptable delay */
} drw, drr;

#define WINTENAB          0100  /* csr register lay out */
#define WREQUEST          0200
#define RINTENAB           040
#define RREQUEST       0100000
#define RSTATUS             02  /* csr 1 bit */
#define WSTATUS             01  /* csr 0 bit */

#define DRPRI 040

/* following are internal state flag bits */
#define OPEN         01
#define CLOSED       00
#define T_FIRST 0100000 /* first packet of a transfer being xmitted */
#define T_DONE   040000 /* last packet just finished */
#define T_SYNCERR 02000 /* writer did something weird */
#define WAIT_TF      02 /* reader just started, and wants to synchronise
                             on next header word */
#define WAIT_TP      04 /* reader emptied a packet & waits for next header */
#define WAITS        06 /* mask for all wait flags */
#define GOT_TF      010 /* header of next packet recv'd with end of prev.one */

#define FLAGS   0176000 /* these flags are part of header words */
#define LENGTH  0001777 /* this is length field in header words */

/* if the writer sets bit WSTATUS in the csr, the reader sees RSTATUS and knows
   that the corresponding inputword is a data-word, as opposed to header.
   A transfer (equivalent to a user open-till-close term) 
   consists of subsequent user-writes. These writes are chopped in packets,
   length >0 and <= 512 bytes each. Every packet is preceded by one
   header word. Header of first packet of a transfer has T_FIRST flag on.
   Intermediate headers have no flag (only lengthfield, see below).
   After last packet, (i.e. at user-close) a header without subsequent data 
   is transferred, which has T_DONE bit on.
   But: reader may miss this one since writer may
   overwrite this with next transfer header with T_FIRST flag on; be aware.
   Lowest 10 bits of header are byte length of packet. Header length is
   not included in this count.
   Count may be odd, in which case high byte of last dataword is garbage.
*/

drwopen(dev, flag)
dev_t dev;
{
        if ((drw.state & OPEN) || (flag == 0)) {
                u.u_error = ENXIO;
                return;
        }
        drw.bptr = geteblk(); /* get internal buffer space */
        DRADDR->csr &= ~(WINTENAB|WSTATUS);
        drw.state = OPEN|T_FIRST;
/*      trace(0177700); trace(drw.state);*/
}

drwclose(dev, flag)
dev_t dev;
{
        spl7(); /* next two should be indivisible; race condition possible */
        DRADDR->csr &= ~(WINTENAB|WSTATUS);
        DRADDR->out =  T_DONE;
        spl0();
        if (drw.bptr != 0) {
                brelse(drw.bptr);
                drw.bptr = 0;
        }
        drw.state = CLOSED;
/*      trace(0177701); trace(drw.state);*/
}

drwrite(dev)
dev_t dev;
{
        register char *c;

        if (u.u_count == 0) return; /* allow empty write */
/*      trace(0177702); trace(u.u_count);*/
        while (c = min(512, u.u_count)) { /* split transfer in packets */
                drw.cptr.integ = drw.bptr->b_un.b_addr;
                /* move from user to kernel */
                iomove((caddr_t)drw.bptr->b_un.b_addr,  c, B_WRITE); 
                drw.fcnt = (c+1)>>1; /* transfer count in words */
                spl5();
                DRADDR->csr &= ~WSTATUS; /* signal a header word */
                DRADDR->out = (drw.state & FLAGS) | c; /* send the header */
/*              trace(0177703); trace((drw.state & FLAGS) | c);*/
                DRADDR->csr |= WINTENAB;
                sleep((caddr_t)&drw, DRPRI); /* wait till entire packet done */
/*              trace(0177703); trace(drw.state);*/
                drw.state &= ~T_FIRST;
                spl0();
        } /* while loop till all packets of this transfer done */
}

drwint()
{
/*      trace(0177704); trace(drw.fcnt);*/
nextword:
        if (drw.fcnt) {
                DRADDR->csr |= WSTATUS; /* data word to follow */
/*              trace(0177705); trace(*(drw.cptr.integ));*/
                DRADDR->out = *(drw.cptr.integ)++;
                drw.fcnt--;
                drw.polcnt = 10; /* tunable */
                do {
                        if (DRADDR->csr & WREQUEST) goto nextword;
                } while (--drw.polcnt);
/*              trace(0177706); trace(drw.polcnt);*/
        } else { /* last word of packet just got transferred */
                DRADDR->csr &= ~WINTENAB;
/*              trace(0177707);*/
                wakeup((caddr_t)&drw);
        }
/*      trace(0177710);*/
}

/***************************************************************************/

drropen(dev, flag)
dev_t dev;
{
        if ((drr.state & OPEN) || (flag != 0)) {
                u.u_error = ENXIO;
                return;
        }
        drr.bptr = geteblk(); /* get internal buffer space */
        if (drr.state & GOT_TF) {
                drr.state = OPEN;
        } else {
                drr.state = OPEN|WAIT_TF;
                drr.fcnt = 0;
        }
        drr.tcnt = 0;
}

drrclose(dev, flag)
dev_t dev;
{
        DRADDR->csr &= ~RINTENAB;
        if (drr.bptr != 0) {
                brelse(drr.bptr);
                drr.bptr = 0;
        }
        drr.tcnt = drr.fcnt = 0;
        drr.state &= GOT_TF; /* clear all bits, except GOT_TF (if on) */
}

drread()
{
        register char *c;
        register int n, m;

        /* note: the reader may chop up his read calls into pieces that do
                 not match the packet boundaries of the writer.
                 previous read calls may have received more input than
                 they passed through to the user. The leftover is then stored
                 in the buffer and must be given to the reader-user first.
                Before that buffer is empty, no new data will be let in.
        */
        while ( n = min(u.u_count, drr.tcnt )) { 
                /* give user the buffer contents, or even less if desired */
                m = drr.cptr.integ;
                m -= drr.bptr->b_un.b_addr;
                m -= drr.tcnt;
                iomove((caddr_t)drr.bptr->b_un.b_addr + m, n, B_READ);
                drr.tcnt -= n;
        }
        if (u.u_count == 0) return; /* implicitly allows empty reads as well */
        if (drr.state & T_DONE) { /* writer done with entire transfer */
                if (drr.state & T_SYNCERR)
                        u.u_error = EIO;
                return;
        }
        do {    /* read one packet after the other, till read resolved or
                   writer done or error  (error includes T_DONE)) */
                drr.cptr.integ = drr.bptr->b_un.b_addr;
                spl5();
                DRADDR->csr |= RINTENAB;
                sleep((caddr_t)&drr, DRPRI);
                spl0();
                if (n = min(u.u_count, drr.tcnt))
                        iomove((caddr_t)drr.bptr->b_un.b_addr,  n, B_READ);
                drr.tcnt -= n;
#ifndef VAX_INFORM
        } while (u.u_count && !(drr.state & (T_DONE|GOT_TF)));
#else
        } while (0);
#endif VAX_INFROM
}


#define POLLING 0
#define ALL_OK  1
#define SYNCERR 2
#define SHOULDNT 3
drrint()
{
        register int nstatus, ndata;
        register int nextcase;

        DRADDR->csr &= ~RINTENAB;
nextword:
        nstatus = DRADDR->csr;
        ndata   = DRADDR->in;
        switch (drr.state & WAITS) {
                /* switch according to what is being expected:
                   transfer header, packet header or dataword */
        case WAIT_TF:   /* transfer header expected */
                if (nstatus & RSTATUS) { /* dataword came in; wrong */
                        /* ignore, wait for transfer header */
                        nextcase = POLLING;
                        break;
                } else switch (ndata & (T_FIRST+T_DONE)) { /* control word came in */
                case T_FIRST:             /* transfer header, is correct */
                        ndata &= LENGTH;        /* take length field */
                        if ((ndata==0)||(ndata>512)) { /* should not happen */
                                nextcase = SHOULDNT;
                                break;
                        }
                        drr.fcnt = ndata;
                        drr.state &= ~(WAIT_TF+WAIT_TP);
                        /* fall through */
                case 0:                 /* packet header came in; wrong */
                                        /* ignore, wait for transfer header */
                case T_DONE:            /* close header of transfer; wrong */
                                        /* ignore, wait for transfer header */
                        nextcase = POLLING;
                        break;
                case T_FIRST+T_DONE:    /* should not happen */
                        nextcase = SHOULDNT;
                        break;
                }
                break;
        case WAIT_TP:           /* packet header expected */
                if (nstatus & RSTATUS) { /* dataword came in; wrong */
                        drr.state |= (T_SYNCERR|T_DONE);
                        nextcase = SYNCERR;
                        break;
                } else switch (ndata & (T_FIRST+T_DONE)) { /* control word came in */
                case T_FIRST:           /* transfer header came in */
                        /* writer must have restarted (or have done a close
                           plus subsequent open very fast) remember this
                           header as the first header of a next transfer */
                        ndata &= LENGTH;        /* take length field */
                        if ((ndata==0)||(ndata>512)) { /* should not happen */
                                nextcase = SHOULDNT;
                                break;
                        }
                        drr.fcnt = ndata;
                        drr.state &= ~(WAIT_TF+WAIT_TP);
                        drr.state |= GOT_TF;
                        nextcase = ALL_OK;
                        break;
                case 0:                 /* packet header came in, correct */
                        ndata &= LENGTH;        /* take length field */
                        if ((ndata==0)||(ndata>512)) { /* should not happen */
                                nextcase = SHOULDNT;
                                break;
                        }
                        drr.fcnt = ndata;
                        drr.state &= ~(WAIT_TF+WAIT_TP);
                        nextcase = POLLING;
                        break;
                case T_DONE:            /* close header came in, also correct */
                        drr.state &= ~WAIT_TP;
                        drr.state |= (T_DONE+WAIT_TF);
                        nextcase = ALL_OK; /* all done */
                        break;
                case T_FIRST+T_DONE:    /* should not happen */
                        nextcase = SHOULDNT;
                        break;
                }
                break;
        case WAIT_TF+WAIT_TP:   /* should not happen */
                nextcase = SHOULDNT;
                break;
        case 0:         /* data word expected */
                if (nstatus & RSTATUS) { /* dataword came in; correct */
                        *(drr.cptr.integ)++ = ndata;
                        drr.tcnt += 2;          /* buffer fill count */
                        drr.fcnt -= 2;          /* header length count */
                        if (drr.fcnt <= 0) { /* this packet's length expired */
                                if (drr.fcnt == -1) {
                                        drr.tcnt--; /* do not count odd byte at the end */
                                        drr.cptr.byt--; /* watch out */
                                }
                                drr.state |= WAIT_TP; /* expect packet header next */
                                nextcase = ALL_OK;
                                break;
                        }
                        nextcase = POLLING;
                        break;
                } else switch(ndata & (T_FIRST+T_DONE)) { /* control word came in; wrong */
                case T_FIRST:           /* transfer header came in */
                                        /* writer must have restarted; remember
                                           this header for the next open call */
                        ndata &= LENGTH;        /* take length field */
                        if ((ndata==0)||(ndata>512)) { /* should not happen */
                                nextcase = SHOULDNT;
                                break;
                        }
                        drr.fcnt = ndata;
                        drr.state &= ~(WAIT_TF+WAIT_TP);
                        drr.state |= (GOT_TF+T_SYNCERR);
                        nextcase = SYNCERR;
                        break;
                case 0:                 /* packet header came in */
                case T_DONE:            /* close header came in */
                        drr.state |= (T_SYNCERR|T_DONE);
                        nextcase = SYNCERR;
                        break;
                case T_FIRST+T_DONE:    /* should not happen */
                        nextcase = SHOULDNT;
                        break;
                }
                break;
        }
        switch(nextcase) {
        case POLLING:
                /* either await transfer header (to get sync), or
                        wait next word in a polling loop */
                drr.polcnt = 20; /* tunable constant */
                do {
                        if (DRADDR->csr & RREQUEST) goto nextword;
                } while (--drr.polcnt);
                DRADDR->csr |= RINTENAB;
                return;
        case ALL_OK:
                wakeup((caddr_t)&drr);
                return;
        case SYNCERR:
                wakeup((caddr_t)&drr);
                return;
        case SHOULDNT:
                /* severe error: leave interrupt enable off,
                   and do not wakeup; freezes program......*/
                return;
        }
}
