#ifndef lint
static char rcsid[]="$Header: proof.c,v 5.1 87/04/01 07:17:30 solomon Stab $";
static char cr[]="Copyright (c) 1986, Marvin Solomon";
#endif lint
/*	
 *	Previewer for X window package
 *
 * Copyright 1986 by Marvin Solomon, University of Wisconsin, Madison, WI
 * (solomon@wisc.edu)
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of
 * Wisconsin not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission.  Marvin Solomon and the University of Wisconsin make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 */

/*
output language from troff:
all numbers are character strings

sn	size in points
fn	font as number from 1-n
in	stipple `font' as number from 1-n
cx	ascii character x
Cxyz	funny char xyz. terminated by white space
Hn	go to absolute horizontal position n
Vn	go to absolute vertical position n (down is positive)
hn	go n units horizontally (relative)
vn	ditto vertically
nnc	move right nn, then print c (exactly 2 digits!)
		(this wart is an optimization that shrinks output file size
		 about 35% and run-time about 15% while preserving ascii-ness)
Dt ...\n	draw operation 't':
	Dt d		line thickness set to d
	Ds d		line style (coordinate bit map) set to d
	Dl x y		line from here by x,y
	Dc d		circle of diameter d with left side here
	De x y		ellipse of axes x,y with left side here
	Da x y r	arc counter-clockwise by x,y of radius r
	D~ x y x y ...	wiggly line by x,y then x,y ...
	Dg x y x y ...	gremlin spline by x,y then x,y ...
	Dp s x y ...	polygon filled with s by x,y then ...
	DP s x y ...	unbordered polygon filled with s by x,y then ...
nb a	end of line (information only -- no action needed)
	b = space before line, a = after
pn	new page begins -- set v to 0
#...\n	comment
x ...\n	device control functions:
	x i	init
	x T s	name of device is s
	x r n h v	resolution is n/inch
		h = min horizontal motion, v = min vert
	x p	pause (can restart)
	x s	stop -- done for ever
	x t	generate trailer
	x f n s	font position n contains font s
	x H n	set character height to n
	x S n	set slant to N

	Subcommands like "i" are often spelled out like "init".
*/
/* END OF COMMENTS */

#include	<stdio.h>
#include	<signal.h>
#include	<ctype.h>
#include	<X/Xlib.h>
#include	<X/XMenu.h>
#include	<sys/types.h>
#include	"dev.h"
#include	"proof.h"

char *malloc();
char *strcpy();

#define	FATAL	1
#define	BMASK	0377
#define	NFONT	60		/* maximum forever */

#ifndef FONTDIR
#define FONTDIR	"/usr/misc/lib/font";
#endif

/* default parameters */

int RES		= 300;	/* resolution (units/inch) of target device */
int	PPI		= 78;	/* pixels per inch on the preview device */
int PSCALE  = 8;	/* 10-point type displayed as PSCALE-point */
int MAXX	= 850;	/* 8-1/2 x 11 in 1/100 inch */
int MAXY	= 1100;

/* characteristics of the display (root window) */
WindowInfo DisplayInfo;
/* Characteristics of visible portion of the window (where .x and .y
 * are the position of visible portion relative to the corner
 * of the window)
 */
WindowInfo WinInfo;

int ReverseVideo;

/* alternate cursor, to indicate we're thinking */
#define hourglass_width 16
#define hourglass_height 16
static short hourglass_bits[] = {
   0x0000, 0x7ffc, 0x2008, 0x1010,
   0x0fe0, 0x07c0, 0x07c0, 0x0380,
   0x0380, 0x0380, 0x0540, 0x0920,
   0x1390, 0x27c8, 0x7ffc, 0x0000};
static short hourglass_mask_bits[] = {
   0xfffe, 0xfffe, 0xfffe, 0xfffe,
   0x7ffc, 0x3ff8, 0x1ff0, 0x1ff0,
   0x1ff0, 0x1ff0, 0x3ff8, 0x7ffc,
   0xfffe, 0xfffe, 0xfffe, 0xfffe};

Cursor wait_cursor;

/* "device" name.  We ignore the target device specified in the file
 * and use instead font description files for the preview device,
 * which had better be available in FONTDIR/devx
 */
char device[20] = "x";
/* Note on fonts:  It is assumed that the fonts used by X have names that
 * are a concatenation of the face name (e.g. R or S) with a point size
 * in the range 1-MAXSIZE.  Not all combinations may exist, so a search is
 * done until XOpenFont succeeds.  DEFAULT_FACE is a reasonable face name
 * to try if the desired face isn't found.
 */
#define MAXSIZE 36
#define DEFAULT_FACE "R"

int	pageno	= -1;	/* output page number */
int	nolist	= 0;	/* output page list if > 0 */
int	olist[20];		/* pairs of page numbers */

/* Solomon@wisc.edu, Feb 1987.  I decided these data structures need
 * some documentation.
 * 
 * A device is described by a "DESC" file (normally
 * ${FONTDIR}/dev${device}/DESC.out).  This file starts with a header of
 * 14 shorts ("struct dev" in dev.h) followed by three tables, which
 * describe various common characteristings of all fonts for this device.
 * The header, which contains such info as the size of the DESC file, the
 * resolution of the device, various dimensions, and the sizes of the
 * following tables, is read into the global data structure "dev" in
 * fileinit().  The fields dev.nfonts, dev.nsizes, dev.nchtab, and
 * dev.nstips are also copied into global variables nfonts, nsizes,
 * nchtab, and nstips, respectively.  The three tables following the
 * header are pstab, chtab, and chname.  Pstab consists of nsizes+1
 * shorts, indicating the available point sizes on the device.
 * Internally, point sizes are usually recorded as an index into this
 * table, rather than the "actual" point size.  Chtab and chname are used
 * to list the "extra characters" outside the regular ASCII printable set
 * (32-127), which are named in troff source by names like \c or \(cc.
 * Not all extra characters exist in all fonts; chname is simply a
 * concatenation of the names (null-terminated) of all extra characters
 * that appear in any font.  Chtab is an array of dev.nchtab shorts, which
 * are interpreted as indices into the chname array.  Chtab defines a
 * mapping from extra-character names to values in the range 128...
 * calcuated by put1s as follows:  Name n translates to code c+128 if
 * strcmp(&chname[chtab[c]],n)==0.
 * 
 * Each "font" (actually, face) with name N (e.g. "R", "CB", etc) is
 * described by a file (normally FONTDIR/devDEV/N.out).  Descriptions
 * for "common" faces are also concatenated on to the the end of the
 * DESC.out file.  A face description starts with a data structure of 4
 * one-byte fields and two 2-char strings (declared as "struct font" in
 * dev.h).  Of these, all but two fields are ignored by this program.  The
 * exceptions are nwfont (number of width entries) and namefont
 * (two-character, null-terminated) name.  Following the header are a
 * width table, a code table, and a font-inclusion table.  The width table
 * contains 2*nwfont bytes, and the other two contain nwfont bytes each.
 * Faces are "loaded" into a specified position by the 'x f' command in
 * the input file.  They are also loaded into positions 1..dev.nfonts from
 * the DESC.out file in fileinit().  When a face is loaded into position
 * n, all this information is read in to an malloc'ed buffer (unless it is
 * already loaded somewhere), and fontbase[n], widtab[n], codetab[n], and
 * fitab[n] are set to point to the four components of the file.
 * Fontname[n] is also set to the name of the face (by t_fp()).  The width
 * table contains the widths of all characters in a "standard size"
 * (dev.unitwidth) font; widths in other point sizes are assumed to be
 * scaled proportionally.  The width information is currently not used by
 * this program.  Fitab[n] maps character codes to indices in widtab[n]
 * and chartab[n].  Chartab[n] then maps the character to a "position".
 * Fitab[n] consists of 96+dev.nchtab one-byte integers.  A character is
 * first converted to a value >=32 (ASCII printables map to themselves,
 * while extra characters are mapped to values >=128 as described above).
 * Then 32 is subtracted, and the result is used to index the fitab[n]
 * array.  If (i=fitab[n][c-32])==0, the character is not present in this
 * face.  Otherwise, it is present in position codetab[n][i].  This
 * program interprets the resulting 8-bit "position" as a code point in an
 * X font.
 * 
 * Finally, the program maintains an array fontdata of structures to
 * represent currently-active fonts (really fonts this time, not faces).
 * Each member gives a "font" (actually face) number (used to index the
 * fontbase, fitab, and codetab arrays), a size (used to index the pstab
 * array) and a pointer to a fontInfo structure used by X to describe a font.
 * An unused member of the the array is indicated by fontdata[i].font==-1.
 * The global 'fs' points to the current member of fontdata.
 * Getfontdata(f,s) finds an element with font face f and size s, creating
 * one if necessary (including finding and opening a corresponding X
 * font), and sets fs to point to it.
 */
struct dev dev;
struct font *fontbase[NFONT+1];
short *	pstab;
int	nsizes = 1;
int	nfonts;
int	nstips;
int	nchtab;
char *	chname;
short *	chtab;
unsigned char *	fitab[NFONT+1];		/* legal characters for each font */
unsigned char *	widtab[NFONT+1];	/* width table for each font (not used) */
unsigned char *	codetab[NFONT+1];	/* device code translation */
char *	fontname[NFONT+1];		/* what font is on what position? */

char *	fontdir = FONTDIR;
FILE *	fp = stdin;		/* input file pointer */
char *	filename;		/* name of input file */

int	size = 1;
int	font = 1;
int	stip = 1;
int polyborder;		/* flag to turn off borders around a polygon */

typedef struct {
	int	font;
	int	size;
	FontInfo *fontInfo;
} fontset;

fontset	*fs;			/* A global pointer to the current face */
fontset fontdata[NFONT+1];	/* table of face data descripters */

int	lastsize	= -1;
int	lastfont	= -1;

int baseline; /* fs->fontInfo->baseline */

#ifdef NONDISP
int NoDisplay; /* for debugging, when there is no display */
#endif NONDISP

int debug = 0;

char *ProgName;   /* argv[0] */
/* Table of "seek" pointers for places in the input where pages start */
#define MAXPAGES	400
struct pte {
	long p_seek;
	int  p_number;
} PageTable [MAXPAGES];
int PageTableSize;

main(argc, argv)
char *argv[];
{
	char *mktemp();
	char *operand(), *option;

	ProgName = argv[0];
	if ((option = XGetDefault(ProgName, "ReverseVideo")) != NULL
		&& strcmp (option, "on") == 0)
		ReverseVideo = 1;

	while (--argc > 0 && **++argv == '-') {
		switch ((*argv)[1]) {
		case 'X': /* page width in 1/100's of an inch */
			MAXX = atoi(operand(&argc, &argv));
			break;
		case 'Y': /* page length in 1/100's of an inch */
			MAXY = atoi(operand(&argc, &argv));
			break;
		case 'S': /* scale*100 */
			PPI = atoi(operand(&argc, &argv));
			break;
		case 's': /* scale*100 */
			PSCALE = atoi(operand(&argc, &argv));
			break;
		case 'F':
			fontdir = operand(&argc, &argv);
			break;
		case 'o':
			outlist(operand(&argc, &argv));
			break;
		case 'r':
			ReverseVideo = 1 - ReverseVideo;
			break;
		case 'd':
			debug++;
			break;
		case 'D':
			strcpy(device, operand(&argc, &argv));
			break;
		default: goto usage;
		}
	}

	if (argc < 1) {
usage:	fprintf(stderr,
			"usage: %s [-X width] [-Y length] [-S scale] [-s pointscale]\n",
			ProgName);
		fprintf(stderr,"\t[-F fontdir] [-o pagelist] [-r] file [...]\n");
		exit(1);
	}
	else {
		if (!XOpenDisplay(0)) {
#ifdef NONDISP
			NoDisplay++;
#else  NONDISP
			fprintf(stderr,"Cannot open display\n");
			exit(1);
#endif NONDISP
		}
		/* Find out display parameters */
		XQueryWindow(RootWindow,&DisplayInfo);

		if (ReverseVideo) {
			Background = BlackPixel;
			Foreground = WhitePixel;
			Overstrike = GXor;
		}
		else {
			Foreground = BlackPixel;
			Background = WhitePixel;
			Overstrike = GXand;
		}
		wait_cursor = XCreateCursor(hourglass_width, hourglass_height,
							hourglass_bits, hourglass_mask_bits,
							8, 8, Foreground, Background,
							Overstrike);
		if (wait_cursor == NULL)  {
			fprintf(stderr,"Cannot create cursor\n");
			exit(1);
		}
		while (argc-- > 0) {
			if ((fp = fopen(*argv, "r")) == NULL)
				errorp(FATAL, *argv, "source file");
			filename = *argv;
			checkFormat();
			makePageTable();
			showfile();
			fclose(fp);
			argv++;
		}
	}
	t_wrapup();
	exit(0);
}


/*----------------------------------------------------------------------------*
 | Routine:	char  * operand (& argc,  & argv)
 |
 | Results:	returns address of the operand given with a command-line
 |		option.  It uses either "-Xoperand" or "-X operand", whichever
 |		is present.  The program is terminated if no option is present.
 |
 | Side Efct:	argc and argv are updated as necessary.
 *---------------------------------------------------------------------------*/

char *operand(argcp, argvp)
int * argcp;
char ***argvp;
{
	if ((**argvp)[2]) return(**argvp + 2); /* operand immediately follows */
	if ((--*argcp) <= 0) {			/* no operand */
	    error (FATAL, "command-line option operand missing.");
	}
	return(*(++(*argvp)));			/* operand next word */
}


outlist(s)	/* process list of page numbers to be printed */
register char *s;
{
	register int n1, n2;

	nolist = 0;
	while (*s) {
		n1 = 0;
		if (isdigit(*s))
			do
				n1 = 10 * n1 + *s++ - '0';
			while (isdigit(*s));
		else
			n1 = -9999;
		n2 = n1;
		if (*s == '-') {
			s++;
			n2 = 0;
			if (isdigit(*s))
				do
					n2 = 10 * n2 + *s++ - '0';
				while (isdigit(*s));
			else
				n2 = 9999;
		}
		olist[nolist++] = n1;
		olist[nolist++] = n2;
		if (*s != '\0')
			s++;
	}
	olist[nolist] = 0;
}


in_olist(n)	/* is n in olist? */
int n;
{
	int i;

	if (nolist == 0)
		return(1);	/* everything is included */
	for (i = 0; i < nolist; i += 2)
		if (n >= olist[i] && n <= olist[i+1])
			return(1);
	return(0);
}

/* check that fp is a ditroff output file */
checkFormat()
{
	int c;
	char str[100];

	(void) fseek(fp, 0L, 0);

	/* first line should be 'x T <typesettername>' */
	if ((c = getc(fp))!='x') {
		/* check for Adobe format, for a more informative error message */
		if (c=='%')
			error(FATAL,"%s seems to be in Adobe (laserwriter) format",
				filename);
		goto bad_format;
	}
	str[0] = '\0';
	(void) fscanf(fp, "%s", str);
	if (str[0] == 'T') return;
bad_format:
	error(FATAL,"%s is not in device-independent troff output format",filename);
}

makePageTable()
{
	int n;
	long ftell();
	register int c;
	register FILE *f = fp;

	(void) fseek(f, 0L, 0);

	PageTableSize = 0;

	/* make a table of seek pointers for starts of pages */

	/* finite automaton searching for EOF or "\np" */
	for (;;) {
		c = getc(f);
		for (;;) {
			if (c == EOF)  {
				if (PageTableSize==0)
					error(FATAL, "no pages in %s\n",filename);
				(void) fseek(f, 0L, 0);
				return;
			}
			if (c != '\n') break;
			c = getc(f);
			if (c == 'p') {
				/* success: found "\np" */
				fscanf(f, "%d", &n);
				if (in_olist(n)) {
					if (PageTableSize >= MAXPAGES)
						error(FATAL, "too many pages (%d max)\n",MAXPAGES);
					PageTable[PageTableSize].p_seek = ftell(f);
					PageTable[PageTableSize].p_number = n;
					PageTableSize++;
				}
				break;
			}
		}
	}
}

/* Fixed menu slots */
#define EXIT_SEL		0
#define CURPAGE_SEL		1
#define NEXTPAGE_SEL	2
#define PREVPAGE_SEL	3
#define RESET_SEL		4

#define SELECTSPERPANE 25

Pixmap WindowSave;

char curlabel[] = "CURRENT: 000";

/* Fill in a menu by creating all the necessary panes and selections
 * indicated in the global data structures PageTableSize and PageTable
 */
MakeMenu(m)
XMenu *m;
{
	static char label[MAXPAGES][9];
	static char plabel[(MAXPAGES+SELECTSPERPANE-1)/SELECTSPERPANE][15];
	int npanes, i, pane;

	npanes = (PageTableSize + SELECTSPERPANE - 1)/SELECTSPERPANE;

	/* add panes and basic selections */
	for (pane = 0;pane < npanes; pane++) {
		int last = pane * SELECTSPERPANE + SELECTSPERPANE - 1;

		if (last >= PageTableSize) last = PageTableSize - 1;

		/* heading */
		(void) sprintf(plabel[pane],"Pages %d-%d",
			PageTable[pane*SELECTSPERPANE].p_number,
			PageTable[last].p_number);
		if (XMenuAddPane(m,plabel[pane],1)==XM_FAILURE) merror("add pane");

		/* EXIT */
		if (XMenuAddSelection(m,pane,(caddr_t)0,"EXIT",1)!=EXIT_SEL)
			merror("add selection");
		/* CURRENT */
		(void) sprintf(curlabel,"CURRENT: %3d",0);
		if (XMenuAddSelection(m,pane,(caddr_t)0,curlabel,1)!=CURPAGE_SEL)
			merror("add selection");
		/* NEXT */
		if (XMenuAddSelection(m,pane,(caddr_t)0,"NEXT",PageTableSize>1)
			!=NEXTPAGE_SEL) merror("add selection");
		/* PREVIOUS */
		if (XMenuAddSelection(m,pane,(caddr_t)0,"PREVIOUS",0)!=PREVPAGE_SEL)
			merror("add selection");
		/* RESET */
		if (XMenuAddSelection(m,pane,(caddr_t)0,"RESET",1)!=RESET_SEL)
			merror("add selection");
	}
	/* add selections for individual pages  */
	for (i=0;i<PageTableSize;i++) {
		(void) sprintf(label[i], "Page %3d", PageTable[i].p_number);
		if (XMenuAddSelection(m,i/SELECTSPERPANE,(caddr_t)&PageTable[i],
				label[i],1)==XM_FAILURE) {
			printf("XMAddSelection(%d,%d,%d,%s,%d)\n",
				m,i%SELECTSPERPANE,(caddr_t)&PageTable[i],label[i],1);
			merror("add page");
		}
	}
	if (XMenuRecompute(m)!=XM_SUCCESS) merror("recompute");
}

/* Get rid of all the panes and selections in the menu (using the
 * global PageTableSize to calculate how many selections there are)
 */
ClearMenu(m)
XMenu *m;
{
	int npanes, i;

	npanes = (PageTableSize + SELECTSPERPANE - 1)/SELECTSPERPANE;
	for (i=npanes-1; i>=0; i--)
		XMenuDeletePane(m,i);
}

#define REFRESHWINDOW XPixmapPut(window, 0, 0, WinInfo.x, WinInfo.y, \
		WinInfo.width, WinInfo.height, \
		WindowSave, GXcopy, AllPlanes)

showfile()
{
	XMenu *m;
	int pane, select;
	struct pte *p;
	XEvent event;
	XButtonEvent *button_event = (XButtonEvent *)&event;
	struct pte *curpage;
	int i, npanes;

	npanes = (PageTableSize + SELECTSPERPANE - 1)/SELECTSPERPANE;

	op_next = op_space;
	op_last = &op_space[PAGEMAX-2]; /* leave a little extra space */
#ifdef NONDISP
	if (NoDisplay) {
		/* debugging version; no display present */

		/* prime the pump by interpreting up to the first 'p' command */
		(void) fseek(fp, 0L, 0);
		conv();
		/* Now "display" pages under user control.  Note that the user
		 * specifies pages by their sequence within the document rather
		 * than their actual page number.
		 */
		i = 0;
		for(;;) {
			p = &PageTable[i];
			op_next = op_space;
			t_page(p->p_number);
			(void) fseek(fp,p->p_seek,0);
			conv();
			fprintf(stderr,"Next page: ");
			scanf(" %d",&i);
			if (i<0) return;
		}
		/* end of debugging version */
	}
#endif NONDISP

	/* prime the pump by interpreting up to the first 'p' command */
	(void) fseek(fp, 0L, 0);
	conv();

	/* create the window (we do it as late as possible, to allow
	 * an 'x r' command help determine the window size).
	 */
	if (ReverseVideo)
		window = XCreateWindow(RootWindow, 0, 0,
					MAXX*PPI/100, MAXY*PPI/100, 2,
					WhitePixmap, BlackPixmap);
	else
		window = XCreateWindow(RootWindow, 0, 0,
					MAXX*PPI/100, MAXY*PPI/100, 2,
					BlackPixmap, WhitePixmap);
	XMapWindow(window);
	XDefineCursor(window, wait_cursor);

	/* create a menu */
	m = XMenuCreate(window,ProgName);
	if (m==NULL) merror("create menu");
	/* The current version (X10v3) of XMenu has a bug in the "freeze" option
	 * (which saves a pixmap of the region obscured by the menu):  If the menu
	 * extends past the edges of the display, XMenuActivate calls XPixmapSave
	 * with "illegal" (according to the server) arguments.  Therefore, we
	 * don't use this feature.
	 */
	if (XMenuSetFreeze(m,0)==XM_FAILURE) merror("set freeze");

	/* and fill it in with panes and selections */
	MakeMenu(m);

	/* then display the first page */
	curpage = PageTable;
	(void) sprintf(curlabel,"CURRENT: %3d",curpage->p_number);
	/* we really should call XMenuChangeSelection() here */
	t_page(curpage->p_number);
	(void) fseek(fp,curpage->p_seek,0);
	conv();
	dumppage();

	for (;;) {
		/* wait for a menu to be requested */
		XUndefineCursor(window, wait_cursor);
		XSelectInput(window,ButtonPressed | ExposeWindow | ExposeRegion);
		XWindowEvent(window,ButtonPressed | ExposeWindow | ExposeRegion,
				&event);
		XDefineCursor(window, wait_cursor);
		if (event.type != ButtonPressed) {
			REFRESHWINDOW;
			continue;
		}

		XSelectInput(window,0);

		/* pop up the window so that the menu is visible */
		XRaiseWindow(window);

		/* If the center or right button pushed,skip the menu business */
		if ((button_event->detail & 0xff)==MiddleButton) {
			if (curpage == &PageTable[0]) continue;
			select = PREVPAGE_SEL;
		}
		else if ((button_event->detail & 0xff)==RightButton) {
			if (curpage == &PageTable[PageTableSize-1]) continue;
			select = NEXTPAGE_SEL;
		}
		else {
			/* pop up a menu and wait for a command */
			/* make menu pop up at "EXIT" selection */
			select = EXIT_SEL;
			switch(
				XMenuActivate(m,&pane,&select,
					button_event->x,button_event->y,ButtonReleased,(char **)&p)
			) {
				case XM_FAILURE: merror("menu activate");
				case XM_IA_SELECT:
				case XM_NO_SELECT:
					/* Refresh the whole window because XMenuPost may have
					 * thrown away expose-region events generated by the above
					 * RaiseWindow.
					 * Also, the current version (X10v3) has a bug in the
					 * "freeze" feature, as described below (see call of
					 * XMenuFreeze).  Therefore, we don't use that
					 * feature.
					 */
					REFRESHWINDOW;
					continue;
			}
		}

		switch (select) {

		case EXIT_SEL:
			return;

		case CURPAGE_SEL:
			dumppage();
			continue;

		case RESET_SEL:
			/* reset everything */
			ClearMenu(m);
			fclose(fp);
			if ((fp = fopen(filename, "r")) == NULL)
				errorp(FATAL, filename, "source file");
			checkFormat();
			makePageTable();

			/* makePageTable() sets PageTableSize, which is used by
			 * ClearMenu() and MakeMenu().  Therefore, the calls to them must
			 * bracket the call to makePageTable().
			 */
			MakeMenu(m);

			/* prime the pump by interpreting up to the first 'p' command */
			(void) fseek(fp, 0L, 0);
			conv();

			/* then display the first page */
			p = curpage = PageTable;

			break;

		case NEXTPAGE_SEL:
			p = ++curpage;
			break;

		case PREVPAGE_SEL:
			p = --curpage;
			break;

		default: curpage = p;
		}

		/* goto page p */
		(void) sprintf(curlabel,"CURRENT: %3d",p->p_number);
		/* should do XMenuChangeSelection() calls here */

		/* activate or deactivate NEXT and PREVIOUS selections */
		/* (can't go back from first page or forward from last) */
		for (i=0; i<npanes; i++) {
			if (XMenuSetSelection(m, i, PREVPAGE_SEL,
				p > PageTable) == XM_FAILURE)
					merror("set selection");
			if (XMenuSetSelection(m, i, NEXTPAGE_SEL,
				p < &PageTable[PageTableSize-1]) == XM_FAILURE)
					merror("set selection");
		}

		op_next = op_space;
		t_page(p->p_number);
		(void) fseek(fp,p->p_seek,0);
		conv();
		dumppage();
	}
}

merror(s)
char *s;
{
	fprintf(stderr,"%s: %s",ProgName,s);
	if (_XMErrorCode != XME_NO_ERROR) fprintf(stderr," (%s)",XMenuError());
	fprintf(stderr,"\n");
	exit(1);
}
	
conv()
/* convert a page of intput into an internal form suitable for
 * fast dumping by dumppage().
 * NB:  thoughout this procedure, hpos and vpos are expressed in device
 * units.
 */
{
	register int c;
	register int k;
	int m, n, n1, m1;
	char str[100], buf[300];
	register FILE *f = fp;

	while ((c = getc(f)) != EOF) {
		switch (c) {
		case '\n':	/* when input is text */
		case ' ':
		case 0:		/* occasional noise creeps in */
			break;
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			/* two motion digits plus a character */
			hmot((c-'0')*10 + getc(f)-'0');
			SAVE(OPHPOS + hpos*PPI/RES)
			put1(getc(f));
			break;
		case 'c':	/* single ascii character */
			put1(getc(f));
			break;
		case 'C':
			fscanf(f, "%s", str);
			put1s(str);
			break;
		case 'D':	/* draw function */
			if (fgets(buf, sizeof(buf), f) == NULL)
			    error(FATAL, "unexpected end of input");
			switch (buf[0]) {
			case 'l':	/* draw a line */
				sscanf(buf+1, "%d %d", &n, &m);
				drawline(n, m);
				break;
			case 'c':	/* circle */
				sscanf(buf+1, "%d", &n);
				drawcirc(n);
				break;
			case 'e':	/* ellipse */
				sscanf(buf+1, "%d %d", &m, &n);
				drawellip(m, n);
				break;
			case 'a':	/* arc */
				sscanf(buf+1, "%d %d %d %d", &n, &m, &n1, &m1);
				drawarc(n, m, n1, m1);
				break;
			case 'P':
				polyborder = 0;		/* borderless polygon */
			case 'p':	/* polygon */
				sscanf(buf+1, "%d", &m);/* get stipple */
				n = 1;			/* number first */
				while (buf[++n] == ' ');
				while (isdigit(buf[n])) n++;
#ifdef STIPPLE
				setfill(m);		/* set up stipple */
#endif
				drawwig(buf+n, f, 2);	/* draw polygon */
				polyborder = 1;		/* assume polygons */
				break;			/*   all have borders */

			case 'g':	/* gremlin curve */
				drawwig(buf+1, f, 0);
				break;
			case '~':	/* wiggly line */
				drawwig(buf+1, f, 1);
				break;
			case 't':	/* line-thickness */
				sscanf(buf+1, "%d", &n);
				drawthick(n);
				break;
			case 's':	/* line-style */
				sscanf(buf+1, "%d", &n);
				drawstyle(n);
				break;
			default:
				error(FATAL, "unknown drawing function %s",buf);
				break;
			}
			break;
		case 's':
			fscanf(f, "%d", &n);	/* ignore fractional sizes */
			setsize(t_size(n));
			break;
		case 'f':
			fscanf(f, "%s", str);
			setfont(t_font(str));
			break;
#ifdef STIPPLE
		case 'i':
			fscanf(f, "%d", &n);
			setstip(n);
			break;
#endif STIPPLE
		case 'H':	/* absolute horizontal motion */
			/* fscanf(f, "%d", &n); */
			while ((c = getc(f)) == ' ')
				;
			k = 0;
			do {
				k = 10 * k + c - '0';
			} while (isdigit(c = getc(f)));
			ungetc(c, f);
			hgoto(k);
			SAVE(OPHPOS + hpos*PPI/RES)
			break;
		case 'h':	/* relative horizontal motion */
			/* fscanf(f, "%d", &n); */
			while ((c = getc(f)) == ' ')
				;
			k = 0;
			do {
				k = 10 * k + c - '0';
			} while (isdigit(c = getc(f)));
			ungetc(c, f);
			hmot(k);
			SAVE(OPHPOS + hpos*PPI/RES)
			break;
		case 'w':	/* word space */
			break;
		case 'V':
			fscanf(f, "%d", &n);
			vgoto(n);
			SAVE(OPVPOS + vpos*PPI/RES)
			break;
		case 'v':
			fscanf(f, "%d", &n);
			vmot(n);
			SAVE(OPVPOS + vpos*PPI/RES)
			break;
		case 'p':	/* new page */
			/* throw away the argument */
			fscanf(f, "%d", &n);
			return;
		case 'n':	/* end of line */
			hpos = 0;

		case '#':	/* comment */
			do
				c = getc(f);
			while (c != '\n' && c != EOF);
			break;
		case 'x':	/* device control */
			if (devcntrl()) return;
			break;
		default:
			error(FATAL, "unknown input character %o %c", c, c);
		}
	}
}


int devcntrl()	/* interpret device control functions */
				/* returns -1 upon "stop" command */
{
        char str[20], str1[50], buf[50];
	int c, n;

	fscanf(fp, "%s", str);
	switch (str[0]) {	/* crude for now */
	case 'i':	/* initialize */
		fileinit();
		t_init();
		break;
	case 'T':	/* device name */
		/* ignored */
		break;
	case 't':	/* trailer */
	case 'p':	/* pause -- can restart */
		return -1;
	case 's':	/* stop */
		return -1;
	case 'r':	/* resolution assumed when prepared */
		fscanf(fp, "%d", &n);
		RES = n;
		break;
	case 'f':	/* font used */
		fscanf(fp, "%d %s", &n, str);
		fgets(buf, sizeof buf, fp);	/* in case there's a filename */
		ungetc('\n', fp);	/* fgets goes too far */
		str1[0] = 0;	/* in case there's nothing to come in */
		sscanf(buf, "%s", str1);
		loadfont(n, str, str1);
		break;
	case 'H':	/* char height */
		fscanf(fp, "%d", &n);
		t_charht(n);
		break;
	case 'S':	/* slant */
		fscanf(fp, "%d", &n);
		t_slant(n);
		break;
	}
	while ((c = getc(fp)) != '\n')	/* skip rest of input line */
		if (c == EOF)
			return -1;
	return 0;
}


fileinit()	/* read in font and code files, etc. */
{
	register int i;
	register int fin;
	register int nw;
	short *filebase;
	register unsigned char *p;
	char temp[100];

		/* open table for device,
		 * read in resolution, size info, font info, etc.
		 * and set params
		 */

	sprintf(temp, "%s/dev%s/DESC.out", fontdir, device);
	if ((fin = open(temp, 0)) < 0)
		errorp(FATAL, temp, "opening font description tables");
	if (read(fin, (char *)&dev, sizeof(struct dev)) != sizeof(struct dev)) 
		errorp(FATAL, temp, "reading header");
	nfonts = dev.nfonts;
	nstips = dev.nstips;
	nsizes = dev.nsizes;
	nchtab = dev.nchtab;
	filebase = (short *)
		malloc((unsigned)dev.filesize);	/* enough room for whole file */

	/* all at once */
	if (read(fin, (char *)filebase, (int) dev.filesize) != dev.filesize)
		errorp(FATAL, temp, "reading whole file");

	pstab = (short *) filebase;
	chtab = pstab + nsizes + 1;
	chname = (char *) (chtab + dev.nchtab);
	p = (unsigned char *) chname + dev.lchname;
	for (i = 1; i <= nfonts; i++) {
		fontbase[i] = (struct font *) p;
		nw = *p & BMASK;		/* 1st thing is width count */
		p += sizeof(struct font);
		widtab[i] = p;			/* then width table */
		codetab[i] = p + 2 * nw;	/* then code conversion table */
		fitab[i] = p + 3 * nw;		/* then font inclusion table */
		p += 3 * nw + dev.nchtab + 128 - 32;
		t_fp(i, fontbase[i]->namefont, fontbase[i]->intname);
	}
	for (i = 1; i <= nstips; i++) {		/* add in Stipple "filenames" */
		if (nfonts + i <= NFONT)
		    t_fp(nfonts + i, (char *)p, (char *)0);
		p += strlen((char *)p) + 1;
	}
	fontbase[0] = NULL;
	close(fin);				/* no fonts loaded yet */
	for (i = 0; i <= NFONT; i++) fontdata[i].font = fontdata[i].size = -1;
}

loadfont(n, s, s1)	/* load font info for font s on position n (0...) */
int n;
char *s, *s1;
{
	char temp[60];
	int fin, nw;

	if (n < 0 || n > NFONT)
		error(FATAL, "illegal fp command %d %s", n, s);
	if (fontbase[n] != NULL && strcmp(s, fontbase[n]->namefont) == 0)
		return;

	for (fin = 1; fin <= NFONT; fin++)	/* first check to see if the */
	    if (strcmp(s, fontbase[fin]->namefont) == 0) {  /* font is loaded */
		register unsigned char *c;		    /* somewhere else */

#define ptrswap(x, y) { c = (unsigned char*) (x); x = y; y = c; }
#define ptrfswap(x, y) { c=(unsigned char*)(x); x = y; y = (struct font *) c; }

		ptrfswap(fontbase[n], fontbase[fin]);
		ptrswap(codetab[n], codetab[fin]);
		ptrswap(widtab[n], widtab[fin]);
		ptrswap(fitab[n], fitab[fin]);
		t_fp(n, fontbase[n]->namefont, fontbase[n]->intname);
		t_fp(fin, fontbase[fin]->namefont, fontbase[fin]->intname);
		return;
	    }

	if (s1 == NULL || s1[0] == '\0')
		sprintf(temp, "%s/dev%s/%s.out", fontdir, device, s);
	else
		sprintf(temp, "%s/%s.out", s1, s);
	if ((fin = open(temp, 0)) < 0) {
		/* This face isn't defined for the preview device, although it
		 * was (presumably) defined for the target device.  Try to find
		 * a reasonable substitute.
		 */
		char c = s[strlen(s)-1];
		char temp1[60];

		if (debug)
			fprintf(stderr,"Can't find font description for '%s'",s);
		if (c=='B') s = "B";
		else if (c=='I') s = "I";
		else s = "R";
		if (s1 == NULL || s1[0] == '\0')
			sprintf(temp1, "%s/dev%s/%s.out", fontdir, device, s);
		else
			sprintf(temp1, "%s/%s.out", s1, s);
		if (debug)
			fprintf(stderr," trying %s\n",temp1);
		if ((fin = open(temp1, 0)) < 0) {
			errorp(FATAL, temp, "font description file");
			return;
		}
	}
	if (fontbase[n] != NULL)
		free((char *)fontbase[n]);
	fontbase[n] = (struct font *) malloc((unsigned) (3*255 + dev.nchtab +
				(128-32) + sizeof(struct font)));
	if (fontbase[n] == NULL)
		error(FATAL, "Out of space in loadfont %s", s);
	read(fin, (char *)fontbase[n], 3*255 + nchtab+128-32 + sizeof(struct font));
	close(fin);
	nw = fontbase[n]->nwfont & BMASK;
	widtab[n] = (unsigned char *) fontbase[n] + sizeof(struct font);
	codetab[n] = (unsigned char *) widtab[n] + 2 * nw;
	fitab[n] = (unsigned char *) widtab[n] + 3 * nw;
	t_fp(n, fontbase[n]->namefont, fontbase[n]->intname);
}


/*VARARGS2*/
error(f, s, a1, a2, a3, a4, a5, a6, a7)
int f;
char *s;
{
	fprintf(stderr, "%s: ",ProgName);
	fprintf(stderr, s, a1, a2, a3, a4, a5, a6, a7);
	fprintf(stderr, "\n");
	if (f)
		exit(2);
}


t_init()	/* initialize device */
{
#ifdef NONDISP
	if (!NoDisplay)
#endif NONDISP
		XFlush();
	drawthick(3);		/* set the line thickness parameter */
	hpos = vpos = 0;
	setsize(t_size(10));	/* start somewhere */
}


/*----------------------------------------------------------------------------*
 | Routine:	t_page ( page_number )
 |
 | Results:	mark this page done for printing.
 *---------------------------------------------------------------------------*/

t_page(pg)	/* do whatever new page functions */
{
	hpos = vpos = 0;
	pageno = pg;
}


t_size(n)	/* convert integer to internal size number*/
int n;
{
	int i;

	if (n <= pstab[0])
		return(0);
	else if (n >= pstab[nsizes-1])
		return(nsizes-1);
	for (i = 0; n > pstab[i]; i++)
		;
	return(i);
}


/* ARGSUSED */
t_charht(n)	/* set character height to n */
int n;
{
	/* punt for now */
}


/* ARGSUSED */
t_slant(n)	/* set slant to n */
int n;
{
	/* punt for now */
}


t_font(s)	/* convert string to internal font number */
char *s;
{
	int n;

	n = atoi(s);
	if (n < 0 || n > nfonts)
		n = 1;
	return(n);
}


t_wrapup()
{
}

put1s(s)	/* s is a funny char name */
register char *s;
{
	static int i = 0;

	if (strcmp(s, &chname[chtab[i]]) != 0)
		for (i = 0; i < nchtab; i++)
			if (strcmp(&chname[chtab[i]], s) == 0)
				break;
	if (i < nchtab)
		put1(i + 128);
	else
		i = 0;
}


put1(c)	/* output char c */
register int c;
{
	register unsigned char *p;
	register int i;
	register int j;
	register int k;
	int ofont, code;

	c -= 32;
	if (c <= 0) {
		return;
	}
	ofont = font;
	i = fitab[font][c];
	if (i != 0) {	/* it's on this font */
		p = codetab[font];
	} else {		/* on another font */
		k = font;	/* start with current, then run down the list */
		for (j=0; j++ <= nfonts; k = (k+1) % (nfonts+1))
			if (fontbase[k] != NULL && (i = fitab[k][c]) != 0) {
				p = codetab[k];
				setfont(k);
				break;
			}
	}
	if (debug>1 && (ofont!=font || i==0)) {
		char cname[3];
		if (c<96) {
			cname[0] = c+32;
			cname[1] = 0;
		}
		else
			sprintf(cname,"%s",&chname[chtab[c-96]]);
		fprintf(stderr,"character 0%o(%s) not found in font %s;",
			c+32, cname,fontname[ofont]);
		if (i==0) fprintf(stderr," or anywhere\n");
		else fprintf(stderr," use font %s code 0%o instead\n", fontname[font],
			p[i]&BMASK);
	}
	code = p[i] & BMASK;
	if (i == 0) {
		return;
	}
	xychar(code);
	if (font != ofont)
		setfont(ofont);
}


setsize(n)	/* set point size to n (internal) */
int n;
{
	size = n;
	SAVE(OPSIZE + n)
}


/*----------------------------------------------------------------------------*
 | Routine:	t_fp ( number, string, string_internal )
 |
 | Results:	font position number now contains font 'string', internal
 |		font name (number) is ignored.
 |
 | Side Efct:	any fonts loaded into fontdata with this font number are
 |		removed.  And, to make sure they're not accessed, if lastfont
 |		equals number, it is "disabled" by setting lastfont to -1.
 *---------------------------------------------------------------------------*/

/* ARGSUSED */
t_fp(n, s, si)
int n;
char *s, *si;
{
	register int i;

	fontname[n] = s;
	for (i = 0; i <= NFONT; i++)		/* release any font files */
		if (fontdata[i].font == n) {	/* for this font */
			XCloseFont(fontdata[i].fontInfo);
			fontdata[i].font = -1;
		}
	if (n == lastfont) lastfont = -1;
}

setfont(n)	/* set font to n */
int n;
{
	if (n < 0 || n > nfonts)
		error(FATAL, "illegal font %d", n);
	font = n;
	SAVE(OPFONT + n)
}

#ifdef STIPPLE
setstip(n)	/* set stipple "font" to n */
int n;
{
	if (n < 1 || n > nstips)
		error(FATAL, "illegal stipple %d", n);
	stip = n;
}
#endif STIPPLE

/*----------------------------------------------------------------------------*
 | Routine:	getfontdata ( font, size )
 |
 | Results:	The font information pointer, fs, is set to point to data for
 |		"font" at point size "size".  If no information for that font is
 |		available, the info is read in from the appropriate font file.
 |		The table "fontdata" holds all the fonts, and it is cleared
 |		of a random font/size if necessary.
 *--------------------------------------------------------------------------*/

getfontdata(f, s)
int f;
int s;
{
	register int fam;
	int i, points;
	FontInfo *fontInfo = (FontInfo *) NULL;
	FontInfo *findfont();

#ifdef NONDISP
	FontInfo *OpenFont();
#endif NONDISP

	if (debug) fprintf(stderr,"getfontdata %s %d\n",fontname[f],pstab[s]);
	/* first check if it's here already */
	for (fam = 0; fam <= NFONT; fam++)
	    if (fontdata[fam].font == f && fontdata[fam].size == s) {
			fs = &fontdata[fam];
			baseline = fs->fontInfo->baseline;
			return;
	    }

	/* find an empty slot */
	for (fam = 0; fam < NFONT && fontdata[fam].font != -1; fam++);
	fs = &fontdata[fam];
	if (fs->font != -1) {		/* clear a slot if not empty */
		/* dumb version - always take the last one to replace */
		if (debug)
			fprintf(stderr,"No free slots for %s%d; dump font %s\n",
				fontname[f], pstab[s]*PSCALE/10, fontname[fs->font]);
#ifdef NONDISP
		if (NoDisplay)
			free(fs->fontInfo);
		else
#endif NONDISP
			XCloseFont(fs->fontInfo);
	}

	/* open font file */
	/* scale the point size */
	points = pstab[s]*PSCALE/10;

	if (debug) {
	    fprintf(stderr, "Looking for font %s%d: trying", 
			fontname[f], points);
	}
	/* Look for a suitable X font.  Our first choice is the smallest size
	 * of face 'fontname[f]' that is >= 'points'.  Second choice is any other
	 * size of this face.  If this fails we repeat the search in DEFAULT_FACE.
	 */
	for (i=points; i<=MAXSIZE; i++)
		if (fontInfo = findfont(fontname[f],i))
			goto font_found;
	for (i=points-1; i>=0; i--)
		if (fontInfo = findfont(fontname[f],i))
			goto font_found;
	for (i=points; i<=MAXSIZE; i++)
		if (fontInfo = findfont(DEFAULT_FACE,i))
			goto font_found;
	for (i=points-1; i>=0; i--)
		if (fontInfo = findfont(DEFAULT_FACE,i))
			goto font_found;
	if (debug) {
		fprintf(stderr, " -- nothing found.\n");
	}
	{
		char name[100];
		(void) sprintf(name,"%s%d",fontname[f],points);
		errorp(FATAL, "No suitable fonts found", name);
	}

font_found:
	fs->fontInfo = fontInfo;
	fs->size = s;
	fs->font = f;
	baseline = fs->fontInfo->baseline;
}

FontInfo *
findfont(face, pointsize)
char *face;
{
	char name[100];
	FontInfo *fontInfo;

	sprintf(name,"%s%d",face,pointsize);
	if (debug) {
		fprintf(stderr, " %s", name);
	}

#ifdef NONDISP
	if (NoDisplay) 
		fontInfo = OpenFont(name); 
	else
#endif NONDISP
		fontInfo = XOpenFont(name);

	if (fontInfo) {
		if (debug) {
			fprintf(stderr, ". baseline = %d\n",
			fontInfo->baseline);
		}
	}
	return fontInfo;
}

#ifdef NONDISP
/* temporary debugging kludge to get font data when there is no display */
FontInfo *OpenFont(name)
char *name;
{
	char fname[1000];
	int fi;
	FontInfo *r;
	struct {
		short bitmapOffset, dummy1, width, height, bitsPerPixel;
		short firstChar, lastChar, widthOffset, dummy2, baseline, spaceIndex;
		short fixedWidth;
	} h;
	sprintf(fname,"/usr/new/lib/X/font/%s.onx",name);
	fi = open(fname,0);
	if (fi<0) {
		perror(fname);
		exit(99);
	}
	if (read(fi,(char *)&h,sizeof h)!=sizeof h) {
		perror(fname);
		exit(98);
	}
	close(fi);
	r = (FontInfo *) malloc((unsigned)sizeof (FontInfo));
	r->id = (Font) r;
	r->height = h.height;
	r->width = h.width;
	r->baseline = h.baseline;
	r->fixedwidth = h.fixedWidth;
	r->firstchar = h.firstChar;
	r->lastchar = h.lastChar;
	r->widths = 0;
	return r;
}
#endif NONDISP

#ifdef STIPPLE
THIS ROUTINE NEEDS TO BE FIXED UP BEFORE IT WILL WORK CORRECTLY
/*----------------------------------------------------------------------------*
 | Routine:	setfill(stipple_number)
 |
 | Results:	sends the appropriate command to set the fill-pattern
 |		for a particular stipple.  Sends the glyph if necessary,
 |		and does nothing if the pattern is the same.  Takes stipple
 |		font from current "stip" number.
 *---------------------------------------------------------------------------*/

setfill(number)
register int number;
{
	register int fam;
	register int gsize;
	register glyph_dir *par;
	register unsigned char *p;
	register fontset *savefs;

	if (stip == laststip && number == laststipmem)
		return;

	savefs = fs;			/* getfontdata sets fs, so we have to */
					/* save it before calling getfontdata */
	fam = getfontdata(nfonts + stip, nsizes);
	laststip = stip;
	laststipmem = number;		/* must be set before call to polygon */

	if (!number || number < fs->first || number > fs->last) {
nostipbits:
		fs = savefs;		/* forget it if it's out of range */
		laststipmem = 0;	/* force NO stipple */
		return;
	}
	if (fs->chused[number] == 0) {		/* stipple not down-loaded */
		par = &(fs->glyph[number]);
		if (!par->g_bitp)
		    goto nostipbits;
		totglyph += glspace(par);
		putc(ABGLY, tf);
		putint((fam << 7) | number, tf);
 		putint(par->g_pwidth, tf);
		putint(par->g_width, tf);
		putint(par->g_left, tf);
		putint(par->g_height, tf);
		putint(par->g_up, tf);
		gsize = ((par->g_width + 7)/8) * par->g_height;
		p = fs->cdp + par->g_bitp;
		while (gsize--)
			putc(*p++, tf);
	}
						/* mark that it's been used */
	if (fs->chused[number] != BMASK)
		fs->chused[number]++;
	putc(ASTEXTURE, tf);			/* set the texture */
	putint((fam << 7) | number, tf);
	fs = savefs;				/* return fs to proper spot */
}
#endif STIPPLE

xychar(c)
int c;
{
	if (font != lastfont || size != lastsize) {
		getfontdata(font, size);
		lastsize = size;
		lastfont = font;
	}
	if (c < fs->fontInfo->firstchar || c > fs->fontInfo->lastchar) {
		return;
	}

	SAVE(OPCHAR + c)
}

dumppage()
/*
 * Display the buffered up characters for a page.
 * NB: Throughout this procedure, hpos and vpos are measured in screen
 * units (pixels).
 */ 
{
	register short *p;
	int dh, dv, newh, newv;
	short *dumpwig();

	hpos = 0; vpos = 0;
	font = 1; size = 1; lastfont = -1; lastsize = -1;

	/*
	 * We grab the server, raise our window to the top, display the
	 * current page in the window, save it away in the pixmap
	 * WindowSave, and finally release the server.  That's the only
	 * way I could think of to make sure XPixmapSave gets a clean
	 * copy of the page.
	 */
	XGrabServer();
	XRaiseWindow(window);
	XClear(window);
	for (p=op_space; p<op_next; p++) {
		switch(*p & OPERATOR) {
		case OPCHAR:  /* a character to display */
			/* We should really try harder to send more than one character
			 * at a time to the X server
			 */
			if (font != lastfont || size != lastsize) {
				getfontdata(font, size);
				lastsize = size;
				lastfont = font;
			}
			/* 
			 * WARNING: The following takes advantage of byte ordering:
			 * We assume that (char *)p points to the low byte of *p for
			 * a VAX, otherwise it points to the high byte of *p.
			 */
			XTextPad(
				window,
				hpos, vpos-baseline,               /* x,y */
#ifdef vax
				(char *)p,			/* characater */
#else
				((char *)p)+1,
#endif
				1, fs->fontInfo->id,  	/* len, font */
				0,0,                      /* charpad, spacepad */
				Foreground, Background,
				Overstrike, AllPlanes          /* func, planes */
				);
			continue;
		case OPHPOS:  /* absolute horiz position in pixels */
			hpos = *p & OPERAND;
			continue;
		case OPVPOS:  /* absolute vertical postion in pixels */
			vpos = *p & OPERAND;
			continue;
		case OPFONT:  /* font number */
			font = *p & OPERAND;
			continue;
		case OPSIZE:  /* font size */
			size = *p & OPERAND;
			continue;
		case OPLINE:  /* dump a line */
			newh = *++p;
			newv = *++p;
			dumpline(newh, newv);
			continue;
		case OPCIRC: /* dump a circle */
			dumpcirc(*p & OPERAND);
			continue;
		case OPELLIP: /* dump an ellipse */
			newh = *++p;
			dv = *++p;
			dumpellip(newh, dv);
			continue;
		case OPARC:	/* dump an arc */
			dh = *++p;
			dv = *++p;
			newh = *++p;
			newv = *++p;
			dumparc(dh,dv,newh,newv);
			continue;
		case OPWIG: /* polygon, spline, or 'pic curve' */
			p = dumpwig(p);
			continue;
		case OPTHICK:
			Thickness = *p & OPERAND;
			continue;
		case OPSTYLE:
			Style = *p & OPERAND;
			continue;
		}
	}
	XFlush();

	/* check to find out size of window, in case it changed */
	XQueryWindow(window,&WinInfo);

	/* clip to fit inside phyiscal display (XPixmapSave bombs otherwise) */
	{
		/* Calculate corners of the visible portion of the window in screen
		 * coordinates */
		int xmin = WinInfo.x, ymin = WinInfo.y;
		int xmax = xmin+WinInfo.width, ymax = ymin + WinInfo.height;

		/* clip them */
		if (xmin < 0) xmin = 0;
		if (ymin < 0) ymin = 0;
		if (xmax > DisplayInfo.width-3) xmax = DisplayInfo.width-3;
		if (ymax > DisplayInfo.height-3) ymax = DisplayInfo.height-3;
		/* The "-3" seems to be necessary to keep XPixmapSave from
		 * bombing.  I don't know why.  Perhaps for the border?
		 */

		/* now convert to window-relative coordinates */
		WinInfo.width = xmax-xmin;
		WinInfo.height = ymax-ymin;
		WinInfo.x = xmin - WinInfo.x;
		WinInfo.y = ymin - WinInfo.y;
	}

	if (WindowSave) XFreePixmap(WindowSave);
	WindowSave = XPixmapSave(window, WinInfo.x, WinInfo.y,
		WinInfo.width, WinInfo.height);
	XUngrabServer();
}
	
errorp(f, s, t)
int f;
char *s, *t;
{
	extern int errno, sys_nerr;
	extern char *sys_errlist[];
	int err = errno;

	fprintf(stderr, "%s: %s",ProgName,s);
	if (t) fprintf(stderr, "(%s)", t);
	if (err > 0 && err < sys_nerr) fprintf(stderr,"%s",sys_errlist[err]);
	fprintf(stderr, "\n");
	if (f)
		exit(2);
}
