/*
 * @COPYRIGHT@
 *
 * Scout Version 1.0
 * 
 * Copyright 1998 Arizona Board of Regents, on behalf of
 *         The University of Arizona
 *         All Rights Reserved
 *
 * @COPYRIGHT@
 *
 * $\RCSfile: wimp_wintree.c,v $
 *
 * HISTORY
 * $\Log: wimp_wintree.c,v $
 * Revision 1.5  1998/01/31 18:43:18  mjk
 * replaced copyright
 *
 * Revision 1.4  1998/01/28 18:23:56  mjk
 * copyright
 *
 * Revision 1.3  1997/08/27 18:09:11  acb
 * Fixed a couple of bugs:
 *   * no longer tries to create window inside null border bitmap
 *   * active window information is not lost on move/resize
 *
 * Revision 1.2  1997/06/09 22:50:46  acb
 * Compensated for title bar when creating bitmaps; fixed a couple of
 * minor bugs.
 *
 * Revision 1.1  1997/03/25 22:19:39  acb
 * Initial revision
 *
 *
 */
#include <oskit/wimpi.h>

#include <bitblit.h>
#include <border.h>
#include <clip.h>
#include <intersect.h>
#include <subs.h>
#include <update.h>

#include <wimp_wintree.h>

#define MY_WINDOW 0

static inline void 
wimpi_window_in_bitmap (WINDOW *win, BITMAP *map)
{
    int bwid = win->borderwid;

    if (win->window) 
	bit_destroy (win->window); 

    if (map) {
	win->window = bit_create(map, bwid, bwid + win->title_high, 
				 map->wide - (bwid<<1), 
				 map->high - (bwid<<1) - win->title_high);
    } else {
	win->window = (void *) 0;
    }
}


WINDOW *
wimpi_get_window (wimpiToplevelWindow ms, Window winId)
{
    WINDOW *win = NULL;

    if (winId == MY_WINDOW)
	return ms->win;

    if (winmapResolve(ms->window_map, winId, &win) != 0) {
      return NULL;
    }

    return win;
}

void
wimpi_make_top_sibling (WINDOW *win)
{
    WINDOW *top, *parent = win->parent;

    if ((top = parent->top_child) == NULL) {
	win->prev_sibling = win->next_sibling = win;
    } else {
	win->prev_sibling = top->prev_sibling;
	win->prev_sibling->next_sibling = win;
	win->next_sibling = top;
	win->next_sibling->prev_sibling = win;
    }
    parent->top_child = win;
}

void
wimpi_make_bottom_sibling (WINDOW *win)
{
    wimpi_make_top_sibling (win);
    if (win->next_sibling != win) {
	win->parent->top_child = win->next_sibling;
    }
}

void
wimpi_remove_sibling (WINDOW *win)
{
    if (win->prev_sibling != win) {
	win->prev_sibling->next_sibling = win->next_sibling;
	win->next_sibling->prev_sibling = win->prev_sibling;
	if (win->parent->top_child == win) {
	    win->parent->top_child = win->next_sibling;
	}
    } else {
	/* No other siblings */
	win->parent->top_child = NULL;
    }
}

void
wimpi_build_list_over_win (WINDOW *win, int ignore_kids) 
{
    WINDOW *tmp, *prev;
    int done;

    /*
     * Build a list of windows that can possibly overlap win.  These are:
     *   - all children
     *   - siblings that are higher than win in sibling list
     *   - siblings of an ancestor of win, that are higher than
     *     that ancestor in sibling list
     */

    prev = win;
    if (! ignore_kids && win->top_child != NULL) {
	for (done = 0, tmp = win->top_child; !done; tmp = tmp->next_sibling) {
	    prev->over = tmp;
	    prev = tmp;
	    if (tmp->next_sibling == win->top_child) done++;
	}
    }

    for (; !ROOT_WIN(win); win = win->parent) {
	for (tmp = win->parent->top_child; tmp != win; 
	     tmp = tmp->next_sibling) 
	{
	    prev->over = tmp;
	    prev = tmp;
	}
    }
    prev->over = NULL;
}

void
wimpi_make_tree_invisible (WINDOW *win)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;
    WINDOW *child;

    if (!win->is_mapped || win->outside_parent)
	return;

    win->is_visible = FALSE;

    if (win->flags & W_ACTIVE) {
	save_win(win);
	wimpi_window_in_bitmap (win, win->save);
	win->flags &= ~W_ACTIVE;
    }

    for (child = win->top_child; child; child = child->next_sibling)
    {
	wimpi_make_tree_invisible (child);
	if (child->next_sibling == win->top_child)
	    break;
    }
}

void
wimpi_make_tree_visible (WINDOW *win)
{
    WINDOW *child;

    if (!win->is_mapped || win->outside_parent || !win->border)
	return;

    win->is_visible = TRUE;
    win->flags &= ~W_CLIPDONE;		/* To be safe */

    for (child = win->top_child; child; child = child->next_sibling)
    {
	wimpi_make_tree_visible (child);
	if (child->next_sibling == win->top_child)
	    break;
    }
}

int
wimpi_window_is_obscured (WINDOW *win) 
{
    WINDOW *tmp;

    wimpi_build_list_over_win (win, FALSE);
    for (tmp = win->over; tmp; tmp = tmp->over) {
	if (tmp->is_visible && intersect (tmp, win)) {
	    return 1;
	}
    }
    return 0;
}

void 
wimpi_expose_win (WINDOW *win)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    /* Do two things:
     *  - expose win
     *  - invalidate clip list for win (since something above it changed)
     */

    if (!win->is_mapped) {
	return;
    }

    win->is_visible = TRUE;
    win->flags &= ~W_CLIPDONE;
    win->exposed = TRUE;

    if (win->save) {
	if (Do_clip())
	{
	    Set_all();
	}
	update(mr, win, &clip);
    } else {
	wimpi_send_expose_event (win);
    }
}

static void
wimpi_expose_tree_under (WINDOW *win, WINDOW *root)
{
    WINDOW *child;
    int obscured = 0;

    /* Do two things:
     *  - expose windows covered by win
     *  - see if exposed windows are no longer obscured
     */

    /* Expose all windows in tree based at root that intersect win. */

    /* Is root obscured by win? */
    if (root->is_visible && intersect(win, root)) {
	wimpi_expose_win (root);
	for (child = root->top_child; child; child = child->next_sibling) {
	    wimpi_expose_tree_under(win, child);
	    if (child->is_visible)
		obscured++;
	    if (child->next_sibling == root->top_child)
		break;
	}

	if (! obscured) {
	    obscured = wimpi_window_is_obscured(root);
	}

	if (! obscured) {
/*	    assert (! (root->flags & W_ACTIVE));*/
	    root->flags |= W_ACTIVE;

	    /* Save window has already been restored (by wimpi_expose_win)
	     * by this point */
	    if (root->save) {
		bit_destroy (root->save);
		root->save = NULL;
		wimpi_window_in_bitmap (root, root->border);
	    }
	}
    }
}

void 
wimpi_expose_lower_windows (WINDOW *win, bool expose_parent) 
{
    WINDOW *sib;

/*    assert (! win->is_visible);*/

    for (sib = win->next_sibling; sib != win->parent->top_child; 
	 sib = sib->next_sibling) 
    {
	wimpi_expose_tree_under (win, sib);
    }

    if (expose_parent && !ROOT_WIN(win->parent)) {
	wimpi_expose_win (win->parent);
	if (! wimpi_window_is_obscured (win->parent)) {
	    win->parent->flags |= W_ACTIVE;
	    bit_destroy (win->parent->save);
	    win->parent->save = NULL;
	    wimpi_window_in_bitmap (win->parent, win->parent->border);
	}
    }
}

static void
wimpi_obscure_tree_under (WINDOW *win, WINDOW *root)
{
    WINDOW *child;

    root->flags &= ~W_CLIPDONE;

    if (root->is_visible && intersect(win, root)) {
	for (child = root->top_child; child; child = child->next_sibling) {
	    wimpi_obscure_tree_under(win, child);

	    if (child->next_sibling == root->top_child)
		break;
	}

	if (root->flags & W_ACTIVE) {
	    save_win (root);
	    wimpi_window_in_bitmap (root, root->save);
	    root->flags &= ~W_ACTIVE;
	}
    }
}

void
wimpi_obscure_lower_windows (WINDOW *win)
{
    WINDOW *sib;

    /*
     * At this point, win should be at proper position in hierarchy and
     * visible, but not yet drawn!
     */
    for (sib = win->next_sibling; sib != win->parent->top_child; 
	 sib = sib->next_sibling) 
    {
	wimpi_obscure_tree_under (win, sib);
    }

    if (!ROOT_WIN(win->parent)) {
	if (win->parent->flags & W_ACTIVE) {
	    save_win (win->parent);
	    wimpi_window_in_bitmap (win->parent, win->parent->save);
	    win->parent->flags &= ~W_ACTIVE;
	}
    }
}

static WINDOW *
wimpi_in_child_window (WINDOW *win, int x, int y)
{
    WINDOW *child;

    for (child = win->top_child; child; child = child->next_sibling) {
	if (child->is_visible && mousein (x, y, child, 1))
	{
	    return wimpi_in_child_window(child, x, y);
	}
	if (child->next_sibling == win->top_child)
	    return win;
    }
    
    return win;
}

WINDOW *
wimpi_window_under_mouse (wimpiSession mr)
{
    WINDOW *win;

    win = wimpi_in_child_window (mr->root_win, mr->mousex, mr->mousey);

    if (win == mr->root_win)
	return NULL;
    
    return win;
}


static bool
wimpi_window_covered (WINDOW *win) 
{
    WINDOW *child, *tmp;
    int covered = 0;

    /* A window is covered if:
     * - it has visible children
     * - it has visible overlapping siblings
     * - an ancestor has visible overlapping siblings that intersect win
     * Otherwise, it is active.
     */

    for (child = win->top_child; child; child = child->next_sibling) {
	if (child->is_visible) {
	    wimpi_window_covered (child);
	    covered++;
	}
	if (child->next_sibling == win->top_child)
	    break;
    }

    if (! covered) {
	/* Win has no children or all are invisible */
	wimpi_build_list_over_win (win, TRUE);
	for (tmp = win->over; tmp; tmp = tmp->over) 
	{
	    if (tmp->is_visible && intersect (tmp, win)) {
		covered++;
		break;
	    }
	}
    }

    if (! covered) {
	if (!(win->flags & W_ACTIVE)) {
	    win->flags |= W_ACTIVE;
	    if (!win->window)
		wimpi_window_in_bitmap (win, win->border);
	}
	return FALSE;
    } else {
	if (win->flags & W_ACTIVE) {
	    win->flags &= ~W_ACTIVE;
	    if (win->exposed) {
		save_win(win);
		wimpi_window_in_bitmap (win, win->save);
	    }
	}
	return TRUE;
    }
}

void
wimpi_expose_tree (WINDOW *win)
{
    WINDOW *child;
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    if (!win->border)
	return;

    /* Don't clip if exposing a whole tree.  If windows are exposed in the
     * right order, get clipping for free.
     */
    win->exposed = TRUE;

    if (!ROOT_WIN(win)) {
	if (win->save) {
	    restore_win(win);
	    if (win->flags & W_ACTIVE) {
		bit_destroy (win->save);
		win->save = NULL;
		wimpi_window_in_bitmap (win, win->border);
	    }
	    
	} else {
	    CLEAR(win->border,PUTOP(BIT_CLR,win->style));
	    border_in_bitmap (win, win->border, BORDER_THIN);
	    wimpi_send_expose_event(win);
	}
    }

    /* Expose all mapped descendants of win, bottom-up */
    if (win->top_child) {
	for (child = win->top_child->prev_sibling; TRUE; 
	     child = child->prev_sibling) 
	{
	    if (child->is_mapped && !child->outside_parent) {
		child->is_visible = TRUE;
		wimpi_expose_tree (child);
	    }
	    if (child == win->top_child) 
		break;
	}
    }
}

void 
wimpi_set_up_window_bitmaps (WINDOW *win, BITMAP *screen)
{
    WINDOW *par = win->parent;
    int wide, high, max_right, max_bottom;
    int bwid = win->borderwid;
    bool changed = FALSE;

    /* Don't worry about creating bitmaps now if parent window doesn't
     * have them:  they will get created when parent's are.
     */
    if (!par->border) {
	win->is_visible = FALSE;
	return;
    }

    max_right = par->x0 + par->border->wide - par->borderwid;
    max_bottom = par->y0 + par->border->high - par->borderwid;

    /* 
     * If window lies outside of parent's visible area, punt.
     */
    if (win->x0 >= max_right || win->y0 >= max_bottom) {
	win->is_visible = FALSE;
	win->outside_parent = TRUE;
    } else {
	win->outside_parent = FALSE;

	/* 
	 * Size of bitmap for window is max_wide X max_high, bounded by
	 * border of parent. 
	 */
	
	wide = win->max_wide;
	high = win->max_high;
	
	if (win->x0 + wide > max_right) {
	    wide = max_right - win->x0;
	}
	
	if (win->y0 + high > max_bottom) {
	    high = max_bottom - win->y0;
	}
	
	    
	if (win->border) {
	    if (wide != win->border->wide - (bwid<<1) || 
		high != win->border->high - (bwid<<1) - win->title_high)
		changed = TRUE;
	    bit_destroy (win->border);
	}
	
	win->border = bit_create(screen,win->x0,win->y0,wide+(bwid<<1),
				 high+(bwid<<1)+win->title_high);

	if (changed && win->save) {
	    if (win->no_expose_events) {
		/* That darn rlogin window again... */
		BITMAP *tmp;
		
		/* Adjust size of save window */
		tmp = bit_alloc(BIT_WIDE(win->border),BIT_HIGH(win->border),
				(DATA*)0,BIT_DEPTH(win->border));
		CLEAR(tmp,PUTOP(BIT_CLR,win->style));
		bit_blit(tmp,bwid,bwid+win->title_high,BIT_WIDE(win->window),
			 BIT_HIGH(win->window),
			 BIT_SRC,win->window,0,0);
		
		border_in_bitmap (win, tmp, BORDER_THIN);
		bit_destroy (win->save);
		win->save = tmp;
		wimpi_window_in_bitmap (win, win->save);
	    } else {
		/* Safest thing to do is to send expose event. 
		 * Throw away save window */
		bit_destroy (win->save);
		bit_destroy (win->window);
		win->save = NULL;
		win->window = NULL;
	    }
	} else {
	    if (win->flags & W_ACTIVE) {
		wimpi_window_in_bitmap (win, win->border);
	    }
	}
    }
}

void
wimpi_fix_up_moved_tree (WINDOW *win)
{
    wimpi_make_tree_visible (win);
    wimpi_window_covered (win);
}

void 
wimpi_move_and_resize_tree (WINDOW *win, int dx, int dy)
{
    BITMAP *screen = win->wimpi->session->screen;
    WINDOW *child;
    
    win->x0 += dx;
    win->y0 += dy;

    wimpi_set_up_window_bitmaps (win, screen);

    for (child = win->top_child; child; child = child->next_sibling) {
	wimpi_move_and_resize_tree (child, dx, dy);
	if (child->next_sibling == win->top_child)
	    break;
    }
}

void
wimpi_print_node (WINDOW *win, int indent)
{
    int i, done;
    WINDOW *child;
    char *str1, *str2;

    for (i = 0; i < indent; i++) printf ("\t");

    if (win->is_visible)
	str1 = "visible";
    else
	str1 = "invisible";

    if (win->flags & W_ACTIVE)
	str2 = "active";
    else
	str2 = "obscured";

    printf ("Window %d: %s, %s\n", win->winId, str1, str2);
    if (win->top_child) {
	for (done = 0, child = win->top_child; !done; ) {
	    wimpi_print_node (child, indent+1);
	    child = child->next_sibling;
	    if (child == win->top_child) done++;
	}
    }
}

void
wimpi_dump_winlist () 
{
    wimpiSession mr;

    /*
      XXX
    mr = (wimpiSession) routerLookupByName ("wimp");

    wimpi_print_node (mr->root_win, 0);
    */
}


