/*
 * Copyright (c) 1996, 1998, 1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Exit code.
 */
#include <threads/pthread_internal.h>
#include "pthread_mutex.h"

static pthread_lock_t	reaper_lock;
static queue_head_t	reaper_queue;

/*
 * Just a dummy lock to satisfy the switch code.
 */
pthread_lock_t		threads_exit_lock    = PTHREAD_LOCK_INITIALIZER;

extern int		threads_alive;

void
pthread_exit(void *status)
{
	pthread_thread_t  *pthread = CURPTHREAD();

	assert_interrupts_enabled();
	assert_preemption_enabled();
	disable_preemption();
	
	pthread_lock(&pthread->lock);

	pthread_exit_locked(status);

	/*
	 * Never returns.
	 */
}

/*
 * Internal version. Thread lock is locked.
 */
void
pthread_exit_locked(void *status)
{
	pthread_thread_t  *pthread = CURPTHREAD();

	DPRINTF("(%d): %p(%p) %p\n", THISCPU, pthread, pthread->tid, status);

	if (pthread->flags & THREAD_EXITING) {
		panic("pthread_exit: Recursive call: 0x%x(%p)\n",
		      (int) pthread, pthread->tid);
	}

	pthread->flags  |= THREAD_EXITING;
	pthread->exitval = (oskit_u32_t) status;
	pthread_unlock(&pthread->lock);

	/*
	 * Enable preemption and interrupts while the cleanups and
	 * destructors are run, in case they misbehave.
	 */
	enable_interrupts();
	enable_preemption();

	/*
	 * Call the cancelation handlers.
	 */
	pthread_call_cleanup_handlers();
		
	/*
	 * Call the key destructors.
	 */
	pthread_call_key_destructors();

#ifdef  PRI_INHERIT
	/*
	 * If this thread is inheriting from a thread (which means a thread
	 * or threads is waiting on it), its probably a bad thing cause
	 * some resource is locked up.
	 */
	if (! queue_empty(&pthread->waiters))
		printf("PTHREAD_EXIT: TID(%p) is exiting, but other "
		       "threads are waiting for it. DEADLOCK?\n",
		       pthread->tid);
#endif
	if (--threads_alive == 0)
		exit(0);

	/*
	 * Let the idle thread clean up the thread.
	 */
	disable_preemption();
	pthread_lock(&reaper_lock);
	queue_enter(&reaper_queue, pthread, pthread_thread_t *, chain);
	pthread_sched_reschedule(RESCHED_BLOCK, &reaper_lock);
	
	/*
	 * Never returns.
	 */
}

/*
 * Reaper support. Let the idle thread clean up dead threads by calling
 * this each time through the idle loop.
 */
void
pthread_reap_thread(pthread_thread_t *pthread)
{
	assert_preemption_disabled();
	assert(pthread);
	
	pthread_lock(&pthread->lock);
		
	if (pthread->flags & THREAD_DETACHED) {
		pthread_destroy_internal(pthread);
		return;
	}
	pthread_unlock(&pthread->lock);

	/*
	 * Not detached, but perhaps someone is waiting for it. We use a
	 * mutex here, but since this is the idle thread it needs to be the
	 * spinning kind. That's okay since the lockers of this mutex are
	 * known to not keep it very long.
	 */
	fast_mutex_spinlock(&pthread->mutex);
	pthread->dead = 1;
	pthread_cond_signal(&pthread->cond);
	fast_mutex_unlock(&pthread->mutex);
}

/*
 * Called from the idle loop each time through to look for dead threads.
 */
void
pthread_reap_threads(void)
{
	pthread_thread_t	*pthread;
	
	pthread_lock(&reaper_lock);
	while (! queue_empty(&reaper_queue)) {
		queue_remove_first(&reaper_queue,
				   pthread, pthread_thread_t *, chain);
		pthread_reap_thread(pthread);
	}
	pthread_unlock(&reaper_lock);
}

void
pthread_init_exit()
{
	pthread_lock_init(&reaper_lock);
	queue_init(&reaper_queue);
}
