/*
 * Copyright (c) 1997-1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * start_conf_network.c
 *
 * Start up to 'maxif' interfaces.  Interfaces are configured first
 * via BOOTP, then any of the appropriate environment variables are
 * checked to supply values, finally for required info, the user is
 * prompted.
 *
 * Environment variables override BOOTP-provided information.  The
 * first BOOTP server to provide a host-level value overrides
 * subsequent BOOTP values.  The host-level values are: hostname,
 * gateway, nameserver(s), and domain.  The per-interface information
 * is address and netmask.
 *
 * Several overridable functions are invoked from this function:
 *    start_conf_network_init(): initialize the network
 *    start_conf_network_open_eif(): open a specific ethernet device
 *    start_conf_network_eifname(): name a specific ethernet device
 *    start_conf_network_ifconfig(): configure a specific ethernet device
 *    start_conf_network_host(): configure host-level network state
 *
 * Checked environment variables include:
 *  start_hostname
 *  start_domainname
 *  start_gateway
 *  start_nameservers
 *  start_<name>_addr  (i.e, start_de0_addr)
 *  start_<name>_mask
 */
#include <oskit/startup.h>
#include <oskit/dev/error.h>
#include <oskit/dev/ethernet.h>
#include <oskit/dev/freebsd.h>
#include <oskit/dev/net.h>
#include <oskit/dev/linux.h>
#include <oskit/net/socket.h>
#include <oskit/net/bootp.h>
#include <oskit/misc/sysconf.h>
#include <oskit/com/services.h>

#include <oskit/c/arpa/inet.h>
#include <oskit/c/assert.h>
#include <oskit/c/errno.h>
#include <oskit/c/fcntl.h>
#include <oskit/c/fd.h>
#include <oskit/c/stdarg.h>
#include <oskit/c/stdio.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/string.h>
#include <oskit/c/unistd.h>

#ifdef	PTHREADS
#include <oskit/threads/pthread.h>

#define start_conf_network	start_conf_network_pthreads
#endif

#define VERBOSITY 4  /* XXX gross */

extern long secondswest; /* from start_clock.c */

/*
 * These are global so release_network() can see them.
 */
static struct oskit_freebsd_net_ether_if **eif;
static int eif_count = 0;

/*
 * Exit hook to shut down the network device. Going to sleep 1 second
 * to allow the network to settle. 
 */
static void
release_network()
{
	int i;
#if VERBOSITY > 0
	printf("Shutting down network ... ");
#endif
	sleep(1);
	
	/* close etherdev and release net_io devices */
	for (i = 0; i < eif_count; i++) {
		start_conf_network_close_eif(eif[i]);
	}
	osenv_mem_free(eif, 0, sizeof(*eif) * eif_count);

#if VERBOSITY > 0
	printf("Done!\n");
#endif
}

/*
 * Local functions defined at the end of this file.
 */

static char*
query_user_netaddr(char *result, int maxlen,
		   const char *fmt, ...);

static char*
query_user_string(char *result, int maxlen,
		  const char *fmt, ...);

/*
 * Start up to maxif interfaces on this machine.
 *
 * If maxif < 0, then all available devices will be probed.
 */
void
start_conf_network(int maxif)
{
	oskit_etherdev_t **etherdev;
	oskit_socket_factory_t *fsc;
	int   i, ndev, err;
	char hostname[256];
	char domainname[64];
	char gateway[64];
	char** nameservers = NULL;

	/* Make host vars obviously bad so we can check them in the loop. */
	hostname[0] = '\0';
	domainname[0] = '\0';
	gateway[0] = '\0';

	start_net_devices();

        /*
         * Find all the Ethernet device nodes.
         */
        ndev = osenv_device_lookup(&oskit_etherdev_iid, (void***)&etherdev);
        if (ndev <= 0)
                panic("No Ethernet adaptors found!");

	err = start_conf_network_init(start_osenv(), &fsc);
	assert(!err);

	/*
	 * Trim the number of probed devices to maxif, if that
	 * value is positive
	 */
	eif_count = ndev;
	if ((maxif >= 0)
	    && (maxif < eif_count))
		eif_count = maxif;

	assert(eif_count);
	eif = osenv_mem_alloc(sizeof(*eif) * eif_count, 0, 0);
	assert(eif);
	
#ifdef  PTHREADS
	pthread_init_socketfactory(fsc);
#else
	oskit_register(&oskit_socket_factory_iid, (void *) fsc);
#endif
	
	for (i = 0; i < eif_count; i++) {
		struct bootp_net_info bootpinfo;
		char start_eif_addr_var[32];
		char start_eif_mask_var[32];
		const char *eif_name;
		char *eif_addr;
		char *eif_mask;
		
		eif_name = start_conf_network_eifname(i);
		eif_addr = NULL;
		eif_mask = NULL;

		snprintf(start_eif_addr_var, sizeof start_eif_addr_var,
			 "start_%s_addr", eif_name);
		snprintf(start_eif_mask_var, sizeof start_eif_mask_var,
			 "start_%s_mask", eif_name);
			
#if VERBOSITY > 1
		printf("Configuring ether dev %d as %s\n", i, eif_name);
#endif
		
		err = bootp(etherdev[i], &bootpinfo);
		if (!err)
		{
			/* Grab the interface address */
			if (bootpinfo.flags & BOOTP_NET_IP)
				eif_addr = strdup(inet_ntoa(bootpinfo.ip));

			/* Grab the interface network mask. */
			if (bootpinfo.flags & BOOTP_NET_NETMASK)
				eif_mask = strdup(inet_ntoa(bootpinfo.netmask));
			else
				eif_mask = strdup(inet_ntoa(
					    bootp_default_netmask(bootpinfo.ip)));

			/* Grab the physical location of the machine. */
			if (bootpinfo.flags & BOOTP_NET_TIME_OFFSET)
				secondswest = bootpinfo.time_offset;

			/*
			 * For the host-level data, only record it if
			 * it isn't already known.
			 */
			if ((hostname[0] == '\0')
			    && (bootpinfo.flags & BOOTP_NET_HOSTNAME))
				strncpy(hostname, bootpinfo.hostname,
					sizeof hostname);

			if ((domainname[0] == '\0')
			    && (bootpinfo.flags & BOOTP_NET_DOMAINNAME))
				strncpy(domainname, bootpinfo.domainname,
					sizeof domainname);
			
			/* XXX only record one gateway from the N provided */
			if ((gateway[0] == '\0')
			    && (bootpinfo.flags & BOOTP_NET_GATEWAY))
				strncpy(gateway, inet_ntoa(bootpinfo.gateway.addr[0]),
					sizeof gateway);

			if ((nameservers == NULL)
			    && (bootpinfo.flags & BOOTP_NET_DNS_SERVER))
			{
				struct bootp_addr_array ns;
				int x;

				ns = bootpinfo.dns_server;
				nameservers = malloc((sizeof(char*) * ns.len) + 1);
				for (x = 0; x < ns.len; x++)
					nameservers[x] = strdup(inet_ntoa(ns.addr[x]));
				nameservers[ns.len] = NULL;
			}

			bootp_free(&bootpinfo);
		}

		/*
		 * Open the underlying ether device.
		 */
		// err = oskit_freebsd_net_open_ether_if(etherdev[i], &eif[i]);
		err = start_conf_network_open_eif(etherdev[i], &eif[i]);
		assert(!err); /* XXX */

		/*
		 * Override per-interface BOOTP info with environment
		 * variables, if they're set.
		 */
		if (getenv(start_eif_addr_var))
			eif_addr = strdup(getenv(start_eif_addr_var));
		
		if (getenv(start_eif_mask_var))
			eif_mask = strdup(getenv(start_eif_mask_var));
		
		/*
		 * If either the address or netmask are unavailable,
		 * query the user for them.
		 */
		if (eif_addr == NULL)
		{
			eif_addr = malloc(sizeof(char) * 64);
			query_user_netaddr(eif_addr, 64,
					   "IP address for interface %s (eif# %d): ",
					   eif_name, i);
		}
		
		if (eif_mask == NULL)
		{
			eif_mask = malloc(sizeof(char) * 64);
			query_user_netaddr(eif_mask, 64,
					   "Netmask for interface %s (eif %d): ",
					   eif_name, i);
		}
		
#if VERBOSITY > 2
		printf("  ifconfig %s as %s (mask %s)\n",
		       eif_name, eif_addr, eif_mask);
#endif

		err = start_conf_network_eifconfig(eif[i],
						   eif_name,
						   eif_addr,
						   eif_mask);

#if VERBOSITY > 2
		if (err)
			printf("  ifconfig FAILED.  err=%d\n", err);
#endif

		assert(eif_addr);
		assert(eif_mask);
		free(eif_addr);
		free(eif_mask);
	}

	/*
	 * Override various bits of host-level configuration with
	 * environment variables.
	 */
	if (getenv("start_hostname"))
		strncpy(hostname, getenv("start_hostname"), sizeof hostname);

	if (getenv("start_domainname"))
		strncpy(domainname, getenv("start_domainname"), sizeof domainname);

	if (getenv("start_gateway"))
		strncpy(gateway, getenv("start_gateway"), sizeof gateway);

	if (getenv("start_nameservers"))
		panic("not built to handle a list(?) of nameservers yet.  Please fix me.");

	/*
	 * Ask the user for host-level configuration data if
	 * none is provided.
	 */
	if (hostname[0] == '\0')
		query_user_string(hostname, sizeof hostname,
				  "Hostname for this machine: ");

	if (domainname[0] == '\0')
		query_user_string(domainname, sizeof domainname,
				  "Domain name for this machine: ");

	if (gateway[0] == '\0')
		query_user_string(gateway, sizeof gateway,
				  "Gateway machine for this machine: ");

	if (nameservers[0] == '\0')
	{
		/* XXX warning */
		printf("WARNING: No nameserver is defined.  You'll just have to live with that.");
	}


	/*
	 * Now do the host-level configuration.
	 */
#if VERBOSITY > 1
	printf("Configuring this host as %s (gw=%s) (domain=%s)\n",
	       hostname, gateway, domainname);
#endif

	start_conf_network_host(hostname, gateway, domainname, nameservers);
	
	/*
	 * Cleanup various messes.
	 */
	startup_atexit(release_network, NULL);

	if (nameservers != NULL)
	{
		char **ns = nameservers;
		while ((*ns) != NULL)
		{
			free(*ns);
			ns++;
		}
		free(nameservers);
	}

#if VERBOSITY > 1
	printf("Network configuration of %s complete.", hostname);
#endif
}


/*
 * Ask the user (via the console) for configuration information
 */
static char*
query_user_netaddr(char *result, int maxlen,
		   const char *fmt, ...)
{
	struct in_addr ip;
	va_list args;
		
        va_start(args, fmt);
	do
	{
		vprintf(fmt, args);
		fgets(result, maxlen, stdin);
		result[strlen(result)-1] = '\0'; /* Nuke trailing \n */
	}
	while(inet_aton(result, &ip) == 0);
        va_end(args);

	return result;
}

static char*
query_user_string(char *result, int maxlen,
		  const char *fmt, ...)
{
	va_list args;
		
        va_start(args, fmt);
	vprintf(fmt, args);
        va_end(args);

	fgets(result, maxlen, stdin);
	result[strlen(result)-1] = '\0'; /* Nuke trailing \n */

	return result;
}
