/*
 * Copyright (c) 1997,1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#ifndef _OSKIT_AVC_AVC_H
#define _OSKIT_AVC_AVC_H 1

#include <oskit/com.h>
#include <oskit/avc/avcomcallback.h>
#include <flask/security.h>
#include <flask/av_permissions.h>

/*
 * COM access vector cache interface.
 * IID 4aa7dfe1-7c74-11cf-b500-08000953adc2
 *
 * Obtain access rulings and register access revocation callbacks.
 */
struct oskit_avc {
        struct oskit_avc_ops *ops;
};
typedef struct oskit_avc oskit_avc_t;

struct oskit_avc_ops {

	/*** COM-specified IUnknown interface operations ***/
	OSKIT_COMDECL	(*query)(oskit_avc_t *s,
				 const struct oskit_guid *iid,
				 void **out_ihandle);
	OSKIT_COMDECL_U	(*addref)(oskit_avc_t *s);
	OSKIT_COMDECL_U	(*release)(oskit_avc_t *s);

	/*** Operations specific to the oskit_avc interface ***/

	/*
	 * Determine whether the requested permissions
	 * are granted between (sclass, ssid) and (tclass,tsid).
	 *
         * Returns OSKIT_ENOMEM if there is insufficient memory,
	 *         OSKIT_EAGAIN if the caller must retry the operation,
	 *         OSKIT_EACCES if permission is denied, or
	 *         0 if permission is granted.
	 */
	OSKIT_COMDECL	(*has_permission)(oskit_avc_t *a,
					  security_class_t sclass,
                                          security_id_t ssid,
                                          security_class_t tclass,
                                          security_id_t tsid,
                                          access_vector_t requested);

        /*
         * This method is similar to the has_permission method,
	 * except that rather than returning a single decision
	 * over the entire set of requested permissions, it
	 * returns individual decisions over each of the requested
	 * permissions, and may also return decisions for other
	 * permissions beyond the requested set.  The absence of an 
	 * unrequested permission from the returned vector cannot be 
	 * interpreted as a denial, only as an undecided condition.
	 *
	 * Returns OSKIT_ENOMEM if there is insufficient memory,
	 *         OSKIT_EAGAIN if the caller must retry the operation, or
	 *         0 if an access ruling was found for the query.
	 */
	OSKIT_COMDECL	(*compute_access_vector)(oskit_avc_t *a,
					  security_class_t sclass,
                                          security_id_t ssid,
                                          security_class_t tclass,
                                          security_id_t tsid,
                                          access_vector_t requested,
					  access_vector_t *out_allowed);

        /*
	 * Add a callback from the avc to the object manager
	 * for a given avc event, where the event is identified
	 * by an event type and a set of class, SID and permission
	 * parameters.  The class and SID parameters may be wildcarded.
	 * The permissions value matches any event with a nonnull
	 * intersection.
	 */
	OSKIT_COMDECL	(*add_callback)(oskit_avc_t *a,				
				        struct oskit_avcomcallback *c,
					oskit_u32_t etype,
 					security_class_t sclass,
                                        security_id_t ssid,
                                        security_class_t tclass,
                                        security_id_t tsid,
				        access_vector_t permissions);

        /*
	 * Remove a previously registered callback object.
	 */
	OSKIT_COMDECL	(*remove_callback)(oskit_avc_t *a,
				           struct oskit_avcomcallback *c);

};


#define oskit_avc_query(a,iid,out_ihandle) ((a)->ops->query((a),(iid),(out_ihandle)))
#define oskit_avc_addref(a) ((a)->ops->addref((a)))
#define oskit_avc_release(a) ((a)->ops->release((a)))
#define oskit_avc_has_permission(a,sclass,ssid,tclass,tsid,requested) ((a)->ops->has_permission((a),(sclass),(ssid),(tclass),(tsid),(requested)))
#define oskit_avc_add_callback(a,c,t,sc,ss,tc,ts,p) ((a)->ops->add_callback((a),(c),(t),(sc),(ss),(tc),(ts),(p)))
#define oskit_avc_remove_callback(a, c) ((a)->ops->remove_callback((a),(c)))


#define COMCHECK(_avc,_sclass,_ssid,_tclass,_tsid,_perm,_rcp) \
    *(_rcp) = oskit_avc_has_permission(_avc, SECCLASS_##_sclass, \
				             _ssid, \
					     SECCLASS_##_tclass, \
					     _tsid, \
					     ##_sclass##_##_tclass##__##_perm)

#define COMCHECK2(_avc,_sclass,_ssid,_tclass,_tsid,_perm1,_perm2,_rcp) \
    *(_rcp) = oskit_avc_has_permission(_avc, SECCLASS_##_sclass, \
				             _ssid, \
					     SECCLASS_##_tclass, \
					     _tsid, \
					     ##_sclass##_##_tclass##__##_perm1 |\
					     ##_sclass##_##_tclass##__##_perm2)

#define COMCHECK3(_avc,_sclass,_ssid,_tclass,_tsid,_perm1,_perm2,_perm3,_rcp) \
    *(_rcp) = oskit_avc_has_permission(_avc, SECCLASS_##_sclass, \
				             _ssid, \
					     SECCLASS_##_tclass, \
					     _tsid, \
					     ##_sclass##_##_tclass##__##_perm1 |\
					     ##_sclass##_##_tclass##__##_perm2 |\
					     ##_sclass##_##_tclass##__##_perm3)


/* GUID for oskit_avc interface */
extern const struct oskit_guid oskit_avc_iid;
#define OSKIT_AVC_IID OSKIT_GUID(0x4aa7dfe1, 0x7c74, 0x11cf, \
				0xb5, 0x00, 0x08, 0x00, 0x09, 0x53, 0xad, 0xc2)

#endif
