/*
 * Copyright (c) 1996-1999 The University of Utah and the Flux Group.
 * 
 * This file is part of the OSKit Linux Glue Libraries, which are free
 * software, also known as "open source;" you can redistribute them and/or
 * modify them under the terms of the GNU General Public License (GPL),
 * version 2, as published by the Free Software Foundation (FSF).
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Linux scheduler support.
 *
 * NOTE: These routines must carefully preserve the
 * caller's interrupt flag.
 */

#include <linux/sched.h>
#include <asm/atomic.h>
#include "osenv.h"

/*
 * Semaphores are implemented using a two-way counter:
 * The "count" variable is decremented for each process
 * that tries to sleep, while the "waiting" variable is
 * incremented _while_ the process is sleeping on that
 * semaphore.
 *
 * Notably, the inline "up()" and "down()" functions can
 * efficiently test if they need to do any extra work (up
 * needs to do something only if count was negative before
 * the increment operation.
 */
static void
normalize_semaphore(struct semaphore *sem)
{
        atomic_add(xchg(&sem->waiting,0), &sem->count);
}


void
__down(struct semaphore *sem)
{
	struct task_struct *cur = current;
	struct wait_queue wait = { cur, NULL };
	unsigned long flags;

	save_flags(flags);
	cli();

	__add_wait_queue(&sem->wait, &wait);
	atomic_inc(&sem->waiting);

	if (sem->count + sem->waiting <= 0) {
		do {
			osenv_sleep_init(&cur->sleeprec);
			osenv_sleep(&cur->sleeprec);
		} while (sem->count < 0);
	}

	__remove_wait_queue(&sem->wait, &wait);
	normalize_semaphore(sem);

	current = cur;
        restore_flags(flags);
}


void
__up(struct semaphore *sem)
{
	normalize_semaphore(sem);
	wake_up(&sem->wait);
}


static void
__sleep_on(struct wait_queue **q, int interruptible)
{
	struct task_struct *cur = current;
	struct wait_queue wait = { cur, NULL };

	if (!q)
		return;

	osenv_sleep_init(&cur->sleeprec);
	add_wait_queue(q, &wait);

	osenv_sleep(&cur->sleeprec);
	current = cur;

	remove_wait_queue(q, &wait);

	current = cur;
}


void
sleep_on(struct wait_queue **q)
{
	__sleep_on(q, 0);
}


void
interruptible_sleep_on(struct wait_queue **q)
{
	osenv_log(OSENV_LOG_ALERT, "INTERRUPTIBLE WAIT!\n");
	__sleep_on(q, 1);
}


void
wake_up(struct wait_queue **q)
{
	struct wait_queue *next;
	struct wait_queue *head;

	if (!q || !(next = *q))
		return; 
	head = WAIT_QUEUE_HEAD(q);
	while (next != head) {
		struct task_struct *p = next->task;
		next = next->next;
		if (p != NULL) {
			osenv_wakeup(&p->sleeprec, OSENV_SLEEP_WAKEUP);
		}
		if (!next)
			panic("wait_queue is bad");
	}
}


void
__wait_on_buffer(struct buffer_head *bh)
{
	struct task_struct *cur = current;
	struct wait_queue wait = { cur, NULL };
	unsigned flags;

	flags = linux_save_flags();
	linux_cli();

	__add_wait_queue(&bh->b_wait, &wait);

	while (buffer_locked(bh)) {
		osenv_sleep_init(&cur->sleeprec);
		osenv_sleep(&cur->sleeprec);
	}
	current = cur;
	
	__remove_wait_queue(&bh->b_wait, &wait);

	linux_restore_flags(flags);

	current = cur;
}


void
unlock_buffer(struct buffer_head *bh)
{
	clear_bit (BH_Lock, &bh->b_state);
	wake_up(&bh->b_wait);
}


/*
 * schedule() should never be called in the oskit since we have
 * no resource contention.
 */
void
schedule()
{
	panic("SCHEDULE CALLED!\n");
}

