/*
 * Copyright (c) 1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Wrappers for the socket calls that involve a struct sockaddr.
 *
 * We need to wrap them because a sockaddr isn't the same in the 
 * oskit and linux.  The linux sockaddr looks like:
 *
 * struct {			vs. 	struct {
 *	short 	sa_family;			u_char	sa_len;
 *	char 	sa_data[14];			u_char 	sa_family;
 * }						char	sa_data[14];
 *					}
 *
 * This causes the sa_family field to appear byte swapped on little
 * endian machines, and makes anything put in the sa_len field part of
 * sa_family.
 */
#include <sys/types.h>
#include <sys/socket.h>

typedef struct {
	u_char  sa_len;
        u_char  sa_family;
    	char 	sa_data[14];
} oskit_sockaddr;


#define MUNGE_SOCKADDR(x) {						\
	if (x) 								\
 		(x)->sa_family = ((oskit_sockaddr *)(x))->sa_family; 	\
}

#if (BYTE_ORDER == LITTLE_ENDIAN)
#define UNMUNGE_SOCKADDR(x) { 						\
	if (x) { 							\
	    	((oskit_sockaddr *)(x))->sa_family = (x)->sa_family; 	\
	    	((oskit_sockaddr *)(x))->sa_len = 0; 			\
	} 								\
}
#else
#define UNMUNGE_SOCKADDR(x)
#endif


int
native_accept(int s, struct sockaddr *addr, int *addrlen) 
{
    	int rc;
    
	rc = native___accept(s, addr, addrlen);
	UNMUNGE_SOCKADDR(addr);
    
	return rc;
}

int
native_bind(int s, /* const */ struct sockaddr *name, int namelen) 
{
	int rc;

	MUNGE_SOCKADDR(name);
	rc = native___bind(s, name, namelen);
	UNMUNGE_SOCKADDR(name);

	return rc;
}

int
native_connect(int s, /* const */ struct sockaddr *name, int namelen) 
{
    	int rc;

	MUNGE_SOCKADDR(name);
	rc = native___connect(s, name, namelen);
	UNMUNGE_SOCKADDR(name);

	return rc;
}


int
native_getpeername(int s, struct sockaddr *name, int *namelen)
{
    	int rc;

	rc = native___getpeername(s, name, namelen);
	UNMUNGE_SOCKADDR(name);

	return rc;
}

int
native_getsockname(int s, struct sockaddr *name, int *namelen)
{
    	int rc;

	rc = native___getsockname(s, name, namelen);
	UNMUNGE_SOCKADDR(name);

	return rc;
}


ssize_t
native_recvfrom(int s, void *buf, size_t len, int flags, struct sockaddr *from,
	 int *fromlen)
{
    	int rc;

	rc = native___recvfrom(s, buf, len, flags, from, fromlen);
	UNMUNGE_SOCKADDR(from);

	return rc;
}

ssize_t
native_sendto(int s, const void *msg, size_t len, int flags,
	      /* const */ struct sockaddr *to, int tolen)
{
    	int rc;

	MUNGE_SOCKADDR(to);
        rc = native___sendto(s, msg, len, flags, to, tolen);
	UNMUNGE_SOCKADDR(to);

	return rc;
}
