/*
 * @COPYRIGHT@
 *
 * Scout Version 1.0
 * 
 * Copyright 1998 Arizona Board of Regents, on behalf of
 *         The University of Arizona
 *         All Rights Reserved
 *
 * @COPYRIGHT@
 *
 * $\RCSfile: wimp_internal.c,v $
 *
 * HISTORY
 * $\Log: wimp_internal.c,v $
 * Revision 1.9  1998/01/31 18:43:18  mjk
 * replaced copyright
 *
 * Revision 1.8  1998/01/29 15:40:54  acb
 * Fixed crash due to resizing a window in funny ways.
 *
 * Revision 1.7  1998/01/28 18:23:56  mjk
 * copyright
 *
 * Revision 1.6  1997/12/12 16:59:06  acb
 * Fixed bug in wimp_post_blit
 *
 * Revision 1.5  1997/08/28 18:31:18  acb
 * fixed bug having to do with deletion of child window which is the
 * terminus of a path.
 *
 * Revision 1.4  1997/08/27 20:15:12  acb
 * Fixed a bug and got rid of a superfluous function.
 *
 * Revision 1.3  1997/08/27 18:05:16  acb
 * Cannot do create_window on a single-window path.
 *
 * Revision 1.2  1997/06/09 22:48:08  acb
 * Cleaned up border drawing, changed coordinates of moved window to
 * account for title bar and edge of screen.
 *
 * Revision 1.1  1997/05/30 21:41:24  acb
 * Initial revision
 *
 * Revision 1.1  1997/04/15 18:15:49  acb
 * Initial revision
 *
 * Revision 1.1  1997/04/09 21:33:31  acb
 * Initial revision
 *
 * Revision 1.1  1997/03/25 22:19:39  acb
 * Initial revision
 *
 *
 */
#include <oskit/wimpi.h>

#include <bitblit.h>
#include <border.h>
#include <font.h>
#include <font_subs.h>
#include <clip.h>
#include <destroy.h>
#include <erase_win.h>
#include <intersect.h>
#include <new_window.h>
#include <subs.h>
#include <update.h>

#include <wimp_internal.h>
#include <wimp_wintree.h>

#include <string.h>

#define BITMAP(win) (win->flags & W_ACTIVE ? win->window : win->save)

extern int traceLevelWIMP;

WINDOW *
wimp_internal_create_window (wimpiSession mr, wimpiToplevelWindow ms, WINDOW *parent, 
			      int x, int y, int wide, int high, 
			     bool has_border)
{
    Window winId;
    WINDOW *win;
    
    winId = mr->next_winId++;
    win = (WINDOW*)malloc(sizeof(WINDOW));
    memset(win, 0, sizeof(WINDOW));

    ms->num_windows++;
    win->wimpi = ms;
    win->winId = winId;
    win->is_mapped = win->is_visible = FALSE;
    strcpy (win->title, "WiMP");

    if (!parent)
	parent = mr->root_win;

    win->has_border = has_border;

    win->parent = parent;
    wimpi_make_bottom_sibling (win);
    
    x += parent->x0 + parent->borderwid;
    y += parent->y0 + parent->borderwid + parent->title_high;

    if (init_window (mr, win, x, y, wide, high, 0, (char **) 0 ) < 0) {
	fprintf(stderr, "wimpi_create_window: Couldn't create a window \n");
	return NULL;
    }

    return win;
}

static void
wimpi_destroy_tree (wimpiSession mr, WINDOW *win) 
{
    if (win->wimpi->window_map) {
	winmapUnbind(win->wimpi->window_map, win->winId);
    }

    for (;win->top_child != NULL;) {
	wimpi_destroy_tree (mr, win->top_child);
    }

    wimpi_remove_sibling (win);

    if (win == mr->active) 
	mr->active = NULL;

    win->wimpi->num_windows--;
    if (win->wimpi->win == win) {
	/* Handle path terminating in child window of another window
	 * (e.g., MPEG child of NETTV) */
	win->wimpi->win = NULL;
    }

    unlink_win(win, 0);

    free (win);
}

void
wimp_internal_destroy_window (WINDOW *win)
{
    wimpiSession mr = win->wimpi->session;

    /* Only need to erase this window, since all children are
       contained within it. */
    if (win->is_visible) {
      wimp_internal_unmap_window (win);
    }

    wimpi_destroy_tree (mr, win);
}

/* 
 * Steps for routines that alter the window tree.  Some are omitted when
 * not necessary:
 *  - mark subtree to be raised, lowered, moved, etc. invisible.  This 
 *    does not affect the screen.
 *  - expose newly uncovered windows (e.g., those under a window that was
 *    made invisible above)
 *  - modify window tree
 *  - mark subtree visible
 *  - obscure newly covered windows
 *  - expose subtree
 */

void 
wimp_internal_map_window (WINDOW *win) {
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    /* In almost all of these routines, turning cursor and mouse off
     * here is conservative.  We could be smarter here (i.e., right
     * before saving a window and only if the mouse is in the window
     * do we turn off the mouse) 
     */

    cursor_off(mr);
    MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);

    win->is_mapped = TRUE;
    
    if (! IS_TOP_SIBLING(win)) {
	wimpi_remove_sibling (win);
	wimpi_make_top_sibling (win);
    }

    mr->active = win;

    wimpi_fix_up_moved_tree (win);
    wimpi_obscure_lower_windows (win);

    /* if (win->has_border)
     * border_in_bitmap (win, win->border, BORDER_FAT );                     
     */

    /* 
     * if (win->no_expose_events && !win->save) {
     *	wimp_internal_fill_rectangle (win, 0, 0, win->window->wide, 
     *				      win->window->high, 
     *				      GETBCOLOR(win->style));
     *    }
     */

    wimpi_expose_tree (win);

    SETMOUSEICON(mr, DEFAULT_MOUSE_CURSOR(mr));    
    
    cursor_on(mr);
    MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
}

void 
wimp_internal_unmap_window (WINDOW *win)
{
    wimpiSession mr = win->wimpi->session;

    /* 
     * Erases the window from the screen and moves it to the bottom of
     * its sibling list.  
     */

    cursor_off(mr);
    MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);
    
    wimpi_make_tree_invisible (win);

    win->is_mapped = FALSE;

    if (ROOT_WIN(win->parent)) {
	erase_visible (mr, win);
    }

    wimpi_expose_lower_windows (win, TRUE);

    wimpi_remove_sibling (win);
    wimpi_make_bottom_sibling (win);

    mr->active = mr->root_win->top_child;

    cursor_on(mr);
    MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
}

void
wimp_internal_raise_window (WINDOW *win)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    cursor_off(mr);
    MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);

    if (! IS_TOP_SIBLING(win)) {
	wimpi_remove_sibling (win);
	wimpi_make_top_sibling (win);
    }

    mr->active = win;

    /* If win is already active (i.e., on top), then screen doesn't
     * change */

    if (!(win->flags & W_ACTIVE)) {
	wimpi_fix_up_moved_tree (win);
	wimpi_obscure_lower_windows (win);
	wimpi_expose_tree (win);
	/* border_in_bitmap (win, win->border, BORDER_FAT ); */
    } else {
	/* border(win, BORDER_FAT ); */
    }

    SETMOUSEICON(mr, DEFAULT_MOUSE_CURSOR(mr));    
    
    cursor_on(mr);
    MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
}

void
wimp_internal_lower_window (WINDOW *win)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    /* Moves window to the bottom of its sibling list but does not
     * erase it from the screen */

    if (win->next_sibling == win)
	return;

    cursor_off(mr);
    MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);
    
    wimpi_make_tree_invisible (win);
    wimpi_expose_lower_windows (win, FALSE);  /* don't expose parent */

    wimpi_remove_sibling (win);
    wimpi_make_bottom_sibling (win);

    wimpi_fix_up_moved_tree (win);

    cursor_on(mr);
    MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
}

void 
wimp_internal_move_resize_window (WINDOW *win, int x, int y, int wide,
				  int high)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;
    int update_screen = win->is_visible;
    
    if (wide < 0) {
	wide = -(wide);
	x -= wide;
    }

    if (high < 0) {
	high = -(high);
	y -= high;
    }

    if (update_screen) {
	cursor_off(mr);
	MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);
	
	wimpi_make_tree_invisible (win);
	
	if (ROOT_WIN(win->parent)) {
	    erase_visible (mr, win);
	}

	wimpi_expose_lower_windows (win, FALSE);  /* don't expose parent */
    }

    if (wide) {
	win->max_wide = wide;
    }

    if (high) {
	win->max_high = high;
    }

    x += win->parent->x0 + win->parent->borderwid;
    y += win->parent->y0 + win->parent->borderwid + win->parent->title_high;

    /* Adjust x and y on outer frame to fit border, title on screen */
    if (win->has_border) {
	int right, bottom;
	int bwid = win->borderwid;

	right = x + win->max_wide + (bwid<<1);
	bottom = y + win->max_high + (bwid<<1) + win->title_high;

	if (right > mr->screen->wide)
	    x -= right - mr->screen->wide;

	if (bottom > mr->screen->high)
	    y -= bottom - mr->screen->high;
    }

    if (x < 0) 
	x = 0;
	
    if (y < 0) 
	y = 0;

    wimpi_move_and_resize_tree (win, x - win->x0, y - win->y0);

    if (update_screen) {
	wimpi_remove_sibling (win);
	wimpi_make_top_sibling (win);
	
	wimpi_fix_up_moved_tree (win);
	wimpi_obscure_lower_windows (win);
	
	/* The way this is written now, window tree must be unobscured */
	/* Expose is unnecessary for AWT applications whose outer frames
	 * have been resized.  AWT erases and redraws all components after
	 * it receives a resize event.  Could be fixed.
	 */

	if (win->is_mapped && !win->outside_parent)
	    wimpi_expose_tree (win); 
	

	/* if (mr->active == win) {
	 *    border (win, BORDER_FAT);
	 *}
	 */
	
	cursor_on(mr);
	MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
    }
}

static int
wimpi_mouse_overlaps_rect (wimpiSession mr, WINDOW *win, int x, int y, int wide, 
			   int high)
{
    int mx, my;

    mx = mr->mousex;
    my = mr->mousey;

    x += win->x0 + win->borderwid;
    y += win->y0 + win->borderwid;

    return (mx + 16 >= x && mx <= x + wide && my + 16 >= y && my <= y + high);
}

void
wimpi_pre_blit (WINDOW *win, int x, int y, int wide, int high)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;

    if (mr->active == win) 
	cursor_off(mr);

    if (wimpi_mouse_overlaps_rect (mr, win, x, y, wide, high))
	MOUSE_OFF(mr, mr->screen, mr->mousex, mr->mousey);
}

void
wimpi_post_blit (WINDOW *win, int x, int y, int wide, int high)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;
    int bwid = win->borderwid;

    if (!(win->flags & W_ACTIVE)) {
	if (Do_clip())
	{
	    x += bwid;
	    y += bwid + win->title_high;
	    Set_clipall();
	    Set_cliplow (x, y);
	    Set_cliphigh (x+wide+1, y+high+1);
	}
	update(mr, win, &clip); 
    }

    cursor_on(mr);
    if (wimpi_mouse_overlaps_rect (mr, win, x, y, wide, high))
	MOUSE_ON(mr, mr->screen, mr->mousex, mr->mousey);
}

void
wimp_internal_fill_rectangle (WINDOW *win, int x, int y, int wide, 
			      int high, int color) 
{
    if (!win->is_visible) return;

    wimpi_pre_blit (win, x, y, wide, high);

    bit_blit(win->window, x, y, wide, high, 
	     BUILDOP(0,0,color), 
	     0, 0, 0);

    wimpi_post_blit (win, x, y, wide, high);
}

void
wimp_internal_set_window_title (WINDOW *win, char *title)
{
    wimpiSession mr = (wimpiSession) win->wimpi->session;
    int length;

    length = strlen (title);
    if (length > 79) 
	length = 79;

    strcpy (win->title, title);

    if (win->is_visible) {
	border (win, 0);
	if (!(win->flags & W_ACTIVE)) {
	    if (Do_clip())
	    {
		int x, y;
		x = y = win->borderwid;
		Set_clipall();
		Set_cliplow (x, y);
		Set_cliphigh (x + win->window->wide, y+win->font->head.high);
	    }
	    update(mr, win, &clip); 
	}
    }
}
