/*
 * Copyright (c) 1994-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Threaded file descriptor support that puts the current thread to
 * sleep if input/output is not available.
 */

#include <oskit/dev/dev.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include "native.h"
#include "support.h"

/*
 * The callback routine.
 */
static void
threaded_callback(void *arg)
{
	osenv_sleeprec_t	*sleeprec = (osenv_sleeprec_t *) arg;

	osenv_wakeup(sleeprec, OSENV_SLEEP_WAKEUP);
}

/*
 * Wait for I/O.
 */
int
threaded_fd_wait(int fd, unsigned iotype)
{
	int			rcode;
	osenv_sleeprec_t	sleeprec;

	osenv_sleep_init(&sleeprec);
	register_async_fd(fd, iotype, threaded_callback, &sleeprec);

	/*
	 * If interrupted, then clear the async wait.
	 */
	if ((rcode = osenv_sleep(&sleeprec)) != OSENV_SLEEP_WAKEUP)
		unregister_async_fd(fd);

	return rcode;
}

/*
 * Non-blocking read from an fd.
 */
int
threaded_read(int fd, void* buf, size_t len)
{
	int		count;
	int		enabled = osenv_intr_enabled();

	if (enabled)
		osenv_intr_disable();

	for (;;) {
		count = NATIVEOS(read)(fd, buf, len);
		if (count >= 0 || !(errno == EAGAIN || errno == EINTR))
			break;

		if (threaded_fd_wait(fd, IOTYPE_READ)) {
			errno = EINTR;
			count = -1;
			break;
		}
	}
	
	if (enabled)
		osenv_intr_enable();
	return count;
}

/*
 * Non-blocking write to an fd.
 */
int
threaded_write(int fd, const void* buf, size_t len)
{
	int		count = 1;
	const void*	ptr;
	int		enabled = osenv_intr_enabled();

	ptr = buf;
	if (enabled)
		osenv_intr_disable();

	while (len > 0 && count > 0) {
		count = NATIVEOS(write)(fd, ptr, len);
		if (count >= 0) {
			ptr += count;
			len -= count;
			count = ptr - buf;
			continue;
		}
		if (!(errno == EAGAIN || errno == EINTR))
			break;

		if (threaded_fd_wait(fd, IOTYPE_WRITE)) {
			errno = EINTR;
			count = -1;
			break;
		}
		count = 1;
	}
	if (enabled)
		osenv_intr_enable();
	return count;
}

/*
 * Non-blocking socket connect.
 */
int
threaded_connect(int fd, struct sockaddr* addr, size_t len)
{
	int		rc;
	int		enabled = osenv_intr_enabled();

	if (enabled)
		osenv_intr_disable();

	printf("threaded_connect: fd = %d\n", fd);
	
	for (;;) {
		rc = NATIVEOS(connect)(fd, addr, len);
		if (rc == 0 || !(errno == EINPROGRESS ||
				 errno == EALREADY || errno == EINTR)) 
		      break;

		if (threaded_fd_wait(fd, IOTYPE_WRITE)) {
			errno = EINTR;
			rc    = -1;
			break;
		}
	}
	/* annul EISCONN error */
	if (rc < 0 && errno == EISCONN)
		rc = 0;

	if (enabled)
		osenv_intr_enable();
	return rc;
}

/*
 * Non-blocking socket accept.
 */
int
threaded_accept(int fd, struct sockaddr* addr, size_t* len)
{
	int		newfd;
	int		enabled = osenv_intr_enabled();

	if (enabled)
		osenv_intr_disable();
	for (;;) {
		newfd = NATIVEOS(accept)(fd, addr, len);
		if (newfd >= 0 || !(errno == EAGAIN || errno == EINTR))
			break;
		
		if (threaded_fd_wait(fd, IOTYPE_READ)) {
			errno = EINTR;
			newfd = -1;
			break;
		}
	}

	if (enabled)
		osenv_intr_enable();
	return newfd;
}

/*
 * Non-blocking socket recvfrom 
 */
int
threaded_recvfrom(int fd, void* buf, size_t len, int flags, 
	struct sockaddr* from, int* fromlen)
{
	int		count;
	int		enabled = osenv_intr_enabled();
 
	if (enabled)
		osenv_intr_disable();
	for (;;) {
		count = NATIVEOS(recvfrom)(fd, buf, len, flags, from, fromlen);
		if (count >= 0 || !(errno == EAGAIN || errno == EINTR))
			break;
		
		if (threaded_fd_wait(fd, IOTYPE_READ)) {
			errno = EINTR;
			count = -1;
			break;
		}
	}
	
	if (enabled)
		osenv_intr_enable();
	return count;
}

/*
 * Non-blocking socket sendto
 */
int
threaded_sendto(int fd, const void* buf, size_t len, int flags, 
	struct sockaddr* to, int tolen)
{
	int		total = 0, count;
	int		enabled = osenv_intr_enabled();
 
	if (enabled)
		osenv_intr_disable();
	while (len > 0) {
		count = NATIVEOS(sendto)(fd, buf, len, flags, to, tolen);
		if (count >= 0) {
			buf   += count;
			len   -= count;
			total += count;
			continue;
		}

		if (!(errno == EAGAIN || errno == EINTR))
			break;

		if (threaded_fd_wait(fd, IOTYPE_WRITE)) {
			errno = EINTR;
			total = -1;
			break;
		}
		count = 1;
	}
	
	if (enabled)
		osenv_intr_enable();
	return total;
}

