/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Notes:
 *
 * What about locking and other issues of execution environments.
 *
 * I'm not using superpages at all. That complicates things 'cause anytime
 * an application wants to modify a page in a superpage, the entire superpage
 * has to be broken up into invidual pages. Too nasty for a first draft.
 *
 * Unused memory cannot be protected since the lmm requires write
 * access to the pages to maintain the free lists. Besides, there is
 * no telling what memory the kernel will access. This means the applicaton
 * can write anything it wants to, except non-existent memory.
 *
 * I am allowing just READ and READ/WRITE prots.  PROT_NONE is not
 * allowed.
 *
 * Should morecore be redefined so that memory shortages in the malloc pool
 * are handled properly?
 *
 * Pages are read and written to/from the swap area through the virtual
 * address.
 */

/*
 * Things to possibly do:
 *
 * 1) External pager interface. Add export of policy for selecting pages
 * to aid in discardable page.
 *
 * 2) Provide support for specifying discardable pages (SML paper) and
 * a way to flush disardable pages from the system.
 *
 * 3) In the threads world, use a thread for a simple pageout daemon.
 */

#include "svm_internal.h"

/*
 * Use the AMM to track virtual mappings.
 */
amm_t		svm_amm;

/*
 * Application level SEGV handler for page faults.
 */
int		(*svm_segv_handler)(oskit_addr_t la, int rw);

/*
 * Enabled?
 */
int		svm_enabled = 0;

/*
 * Thread safe locking.
 */
oskit_lock_t	*svm_lock = 0;

extern int	enable_gdb;	/* XXX */

/*
 * The VM region starts after the end of physical memory, and extends
 * as far as possible.
 */
void
svm_init(oskit_absio_t *pager_absio)
{
	extern char		_start[], end[], start_of_data[];
	oskit_addr_t		addr, start_addr;
	oskit_lock_mgr_t	*lock_mgr;

	if (svm_enabled)
		return;

	if (ptab_alloc(&base_pdir_pa))
		panic("Can't allocate kernel page directory");

	start_addr = (unsigned) _start & ~(PAGE_SIZE - 1);

	/*
	 * Map lower part of memory (ROM and I/O).
	 */
#ifdef  DEBUG_SVM
	printf("Mapping 0x%x to 0x%x\n", 0, (int) start_addr);
#endif
	if (pdir_map_range(base_pdir_pa, 0, 0, (oskit_addr_t) start_addr,
		map_modebits(PTE_MBITS_VALID|PTE_MBITS_URW)))
		panic("Can't direct-map lower physical memory");

	/*
	 * Map the text segment as read-only.
	 */
#ifdef  DEBUG_SVM
	printf("Mapping 0x%x to 0x%x\n", (int) start_addr, (int)start_of_data);
#endif	
	if (pdir_map_range(base_pdir_pa, (oskit_addr_t) start_addr,
		(oskit_addr_t) start_addr,
		(oskit_addr_t) start_of_data - (oskit_addr_t) start_addr,
		(enable_gdb ? map_modebits(PTE_MBITS_VALID|PTE_MBITS_URW) :
		              map_modebits(PTE_MBITS_VALID|PTE_MBITS_UR))))
		panic("Can't direct-map text memory");

	/*
	 * Map data segment as read/write.
	 */
#ifdef  DEBUG_SVM
	printf("Mapping 0x%x to 0x%x\n", (int) start_of_data, round_page(end));
#endif
	if (pdir_map_range(base_pdir_pa, (oskit_addr_t) start_of_data,
		(oskit_addr_t) start_of_data,
	        (oskit_addr_t) round_page(end) - (oskit_addr_t) start_of_data,
		map_modebits(PTE_MBITS_VALID|PTE_MBITS_URW)))
		panic("Can't direct-map data memory");

	/*
	 * Map the bottom page of the stack as read-only. It was part of
	 * the data segment, so must unmap it first.
	 */
	pdir_unmap_page(base_pdir_pa, (oskit_addr_t) base_stack_start);

#ifdef  DEBUG_SVM
	printf("Mapping 0x%x to 0x%x\n", (int) base_stack_start,
	       (int) base_stack_start + PAGE_SIZE);
#endif	
	if (pdir_map_range(base_pdir_pa, (oskit_addr_t) base_stack_start,
		(oskit_addr_t) base_stack_start, PAGE_SIZE,
		map_modebits(PTE_MBITS_VALID|PTE_MBITS_UR)))
		panic("Can't direct-map stack redzone");

	/*
	 * Map the rest of phys memory as not-present so that page faults 
	 * are generated. Without mapping entries, the CPU allows access.
	 *
	 * XXX Do this in a page loop to avoid superpage creation.
	 */
#ifdef  DEBUG_SVM
	printf("Mapping 0x%x to 0x%x\n", round_page(end),
	       round_page(phys_mem_max));
#endif
	addr = (oskit_addr_t) round_page(end);
	while (addr < round_page(phys_mem_max)) {
		if (pdir_map_range(base_pdir_pa, addr, addr, PAGE_SIZE,
			map_modebits(PTE_MBITS_VALID|PTE_MBITS_URW)))
			panic("Can't direct-map rest of physical memory");

		addr += PAGE_SIZE;
	}

	/*
	 * Set the WP bit in CR0. This bit turns on enforcement of read-only
	 * access when a user-mode page is accessed in supervisor mode.
	 */
	set_cr0(get_cr0() | CR0_WP);

	/*
	 * This turns on paging!
	 */
	base_paging_load();

	/*
	 * Initialize the stuff for catching stack overflow.
	 */
	svm_redzone_init();

	/*
	 * Allocate a memory map for the virtual range.
	 */
	amm_init(&svm_amm, round_page(phys_mem_max), AMM_MAXADDR);

	/*
	 * See if thread-safe locks are required.
	 */
	if (oskit_lookup_first(&oskit_lock_mgr_iid, (void *) &lock_mgr))
		panic("svm_init: oskit_lookup_first");

	if (lock_mgr) {
		if (oskit_lock_mgr_allocate_lock(lock_mgr, &svm_lock))
			panic("svm_init: oskit_lock_mgr_allocate_lock");
	}

	/*
	 * XXX: Must turn this on before starting the pager. See mem.c
	 */
	svm_enabled = 1;

	/*
	 * Initialize the pager if requested.
	 */
	if (pager_absio) {
		if (svm_pager_init(pager_absio))
			panic("svm_init: Could init the pager!\n");
	}

	/*
	 * Hook in a new page fault handler.
	 */
	base_trap_handlers[T_PAGE_FAULT] = svm_page_fault_handler;

	/*
	 * and initialize the application level handler to null so
	 * that the default is to send a SIGSEGV to the application.
	 */
	svm_segv_handler = 0;
}

void
svm_shutdown()
{
	extern int	svm_pagein_count, svm_pageout_count;
	extern int	svm_pagein_time, svm_pageout_time;
	extern int	svm_swapread_time, svm_swapwrite_time;
	extern int	svm_swapwrite_count, svm_swapread_count;

	printf("svm_shutdown: swapread  count: %d pages. time %d\n"
	       "              swapwrite count: %d pages. time %d\n",
	       svm_swapread_count, svm_swapread_time,
	       svm_swapwrite_count, svm_swapwrite_time);

	printf("svm_shutdown: pagein  count: %d pages. time %d\n"
	       "              pageout count: %d pages. time %d\n",
	       svm_pagein_count, svm_pagein_time,
	       svm_pageout_count, svm_pageout_time);
}
