/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include "svm_internal.h"
#include <oskit/c/strings.h>

/*
 * Page fault handler. If the fault is for a mapped page that is paged
 * out or unallocated, take care of it.
 *
 * If its a protection or otherwise unallocated reference, pass to the
 * SEGV handler if it is defined. Otherwise panic.
 */
int
svm_fault(oskit_addr_t vaddr, int eflags)
{
	pt_entry_t	pte;
	oskit_addr_t	vpage;
	oskit_addr_t	paddr;
	int		mbits, nbits;

	SVM_LOCK();

#ifdef  DEBUG_SVM
	printf("svm_fault: 0x%x 0x%x\n", vaddr, eflags);
#endif

	vpage = trunc_page(vaddr);

	/*
	 * If address not mapped, go right to SEGV handler.
	 */
	if (svm_mapped(vpage, PAGE_SIZE))
		goto segv;

	/*
	 * Address is mapped. Is the page not even allocated (zero-fill)?
	 * If so, find a page and map it. Just return.
	 */
	pte   = pdir_get_pte(base_pdir_pa, vpage);
	paddr = pte_to_pa(pte);
	mbits = unmap_modebits(pte_to_modebits(pte));

	if (pte == NULL)
		panic("svm_fault: Invalid pte!");

	if (mbits & PTE_MBITS_ZFILL) {
		/*
		 * Zero fill. Allocate a page and map it.
		 */
		paddr = svm_alloc_physpage();

#ifdef  DEBUG_SVM
		printf("svm_fault: "
		       "Mapping vaddr 0x%x to paddr 0x%x\n", vpage, paddr);
#endif
		nbits = (mbits & ~PTE_MBITS_ZFILL) | PTE_MBITS_VALID;
		pte   = create_pte(paddr, map_modebits(nbits));

		if (pdir_map_page(base_pdir_pa, vpage, pte))
			panic("svm_alloc: Can't map physical memory");
		ftlbentry(vpage);

		/*
		 * Plug into ptov table.
		 */
		svm_ptov[btop(paddr)] = vpage;

		/*
		 * Zero the page through the virtual address.
		 */
		bzero((void *)vpage, PAGE_SIZE);

		goto done;
	}

	/*
	 * If there is a paddr, it might be a disk block number. Call the
	 * pagein code to bring it in.
	 */
	if (mbits & PTE_MBITS_PAGED) {
		svm_pagein_page(vpage);
		goto done;
	}

	/*
	 * Region is mapped and has a valid physical translation. Must
	 * be a protection violation.
	 */
   segv:
	SVM_UNLOCK();
	if (svm_segv_handler)
		return (*svm_segv_handler)(vaddr, eflags);

	return 1;

   done:
	SVM_UNLOCK();
	return 0;
}
