/*
 * Copyright (c) 1997,1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#ifndef _OSKIT_FS_FILEPSID_H
#define _OSKIT_FS_FILEPSID_H 1

#include <oskit/com.h>
#include <oskit/fs/file.h>
#include <flask/security.h>

/*
 * COM file persistent SID table interface. 
 * IID 4aa7dfe5-7c74-11cf-b500-08000953adc2
 *
 * Get or set the SID of a file within a persistent SID table.
 *
 * The kserver and file server supply a oskit_filepsid 	
 * interface to the fs security COM layer during 
 * initialization; the fs security COM layer then
 * uses this interface whenever:	
 * 	a) a file or dir COM object is created (get)
 * 	b) a new file is created (create, set)
 *	c) chown is invoked on an existing file (chown, set)
 *	d) chsid is invoked on an existing file (set)
 *	e) a file is destroyed (unset)
 */
struct oskit_filepsid {
        struct oskit_filepsid_ops *ops;
};
typedef struct oskit_filepsid oskit_filepsid_t;


struct oskit_filepsid_ops {

	/*** COM-specified IUnknown interface operations ***/
	OSKIT_COMDECL	(*query)(oskit_filepsid_t *p,
				 const struct oskit_guid *iid,
				 void **out_ihandle);
	OSKIT_COMDECL_U	(*addref)(oskit_filepsid_t *p);
	OSKIT_COMDECL_U	(*release)(oskit_filepsid_t *p);

	/*** Operations specific to the oskit_filepsid interface ***/

	/*
	 * Return a reference to the SID bound to the specified
	 * file in this file persistent SID table. 
	 */
	OSKIT_COMDECL	(*get)(oskit_filepsid_t *p,
			       oskit_file_t *file,
			       security_id_t *outsid);

	/*
	 * Bind the specified SID to the specified file 
	 * in this file persistent SID table. 
	 */
	OSKIT_COMDECL	(*set)(oskit_filepsid_t *p,
			       oskit_file_t *file,
			       security_id_t newsid);

	/*
	 * Destroy the binding for the specified inode
	 * in this file persistent SID table. 
	 */
	OSKIT_COMDECL	(*unset)(oskit_filepsid_t *p,
				 oskit_ino_t ino);
				

	/*
	 * Return a reference to the default SID to use
	 * for a file to be created, based on the provided
	 * client SID, parent directory SID, and file security
	 * class.   XXX doesn't belong here.
	 */
	OSKIT_COMDECL	(*create)(oskit_filepsid_t *p,
				  security_id_t csid,
				  security_id_t dsid,
				  security_class_t fclass,
				  security_id_t *outsid);

	/*
	 * Return a reference to the SID that a file
	 * with the specified SID would have if its uid/gid
	 * were changed to the specified values. 
	 * XXX doesn't belong here
	 */  
	OSKIT_COMDECL	(*chown)(oskit_filepsid_t *p,
				 security_id_t fsid,
				 oskit_u32_t mask,
				 const oskit_stat_t *stats,
				 security_id_t *outsid);
};


#define oskit_filepsid_query(p,iid,out_ihandle) ((p)->ops->query((p),(iid),(out_ihandle)))
#define oskit_filepsid_addref(p) ((p)->ops->addref((p)))
#define oskit_filepsid_release(p) ((p)->ops->release((p)))
#define oskit_filepsid_get(p, f, outsid) ((p)->ops->get((p),(f),(outsid)))
#define oskit_filepsid_set(p, f, newsid) ((p)->ops->set((p),(f),(newsid)))
#define oskit_filepsid_unset(p, i) ((p)->ops->unset((p),(i)))
#define oskit_filepsid_create(p, c, d, f, outsid) ((p)->ops->create((p),(c),(d),(f),(outsid)))
#define oskit_filepsid_chown(p, f, mask, stats, outsid) ((p)->ops->chown((p),(f),(mask),(stats),(outsid)))

/* GUID for oskit_filepsid interface */
extern const struct oskit_guid oskit_filepsid_iid;
#define OSKIT_FILEPSID_IID OSKIT_GUID(0x4aa7dfe5, 0x7c74, 0x11cf, \
				0xb5, 0x00, 0x08, 0x00, 0x09, 0x53, 0xad, 0xc2)

#endif
