/*
 * Copyright (c) 1996-1999 The University of Utah and the Flux Group.
 * 
 * This file is part of the OSKit Linux Glue Libraries, which are free
 * software, also known as "open source;" you can redistribute them and/or
 * modify them under the terms of the GNU General Public License (GPL),
 * version 2, as published by the Free Software Foundation (FSF).
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Linux kernel memory allocation.
 */

#ifndef OSKIT
#define OSKIT
#endif

#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/delay.h>

#include "shared.h"
#include "osenv.h"

/* XXX are all these flags really necessary? */
#define KMALLOC_FLAGS	(OSENV_AUTO_SIZE | OSENV_PHYS_WIRED |	\
			 OSENV_VIRT_EQ_PHYS | OSENV_PHYS_CONTIG)

/*
 * Invoke the OS's memory allocation services,
 * saving the current pointer around the possibly-blocking call.
 */
void *
oskit_linux_mem_alloc(oskit_size_t size, osenv_memflags_t flags, unsigned align)
{
	struct task_struct *cur = current;
	void *mem;
	mem = osenv_mem_alloc(size, flags, align);
	current = cur;
	return mem;
}

void
oskit_linux_mem_free(void *block, osenv_memflags_t flags, oskit_size_t size)
{
	struct task_struct *cur = current;
	osenv_mem_free(block, flags, size);
	current = cur;
}

/*
 * Allocate wired down kernel memory.
 */
void *
kmalloc(unsigned int size, int priority)
{
	int flags;
	void *p;

	flags = KMALLOC_FLAGS;
	if (priority & GFP_ATOMIC)
		flags |= OSENV_NONBLOCKING;
	if (priority & GFP_DMA)
		flags |= OSENV_ISADMA_MEM;
	p = oskit_linux_mem_alloc(size, flags, 0);
	return (p);
}

/*
 * Free kernel memory.
 * XXX this should probably pass the same flags as kmalloc.  It doesn't matter
 *     right now since osenv_mem_free only listens to OSENV_AUTO_FLAGS.
 */
void
kfree(void *p)
{
	oskit_linux_mem_free(p, KMALLOC_FLAGS, 0);
}

/*
 * Allocate physically contiguous pages.
 */
unsigned long
__get_free_pages(int priority, unsigned long order, int dma)
{
	int flags;
	void *p;

	flags = OSENV_PHYS_WIRED|OSENV_VIRT_EQ_PHYS|OSENV_PHYS_CONTIG;
	if (priority & GFP_ATOMIC)
		flags |= OSENV_NONBLOCKING;
	if ((priority & GFP_DMA) || dma)
		flags |= OSENV_ISADMA_MEM;
	p = oskit_linux_mem_alloc(PAGE_SIZE << order, flags, PAGE_SIZE);
	return ((unsigned long)p);
}

/*
 * Free pages.
 */
void
free_pages(unsigned long addr, unsigned long order)
{
	oskit_linux_mem_free((void *)addr, 0, PAGE_SIZE << order);
}

/*
 * Allocate virtually contiguous memory.
 * The underlying physical pages do not need to be contiguous.
 */
void *
vmalloc(unsigned long size)
{
	int order;

	for (order = 0; (PAGE_SIZE << order) < size; order++)
		;
	return ((void *)__get_free_pages(0, order, 0));
}
