/*
 * Copyright (c) 1997,1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <oskit/fs/soa.h>
#include <oskit/c/assert.h>
#include <oskit/c/stddef.h>
#include <oskit/c/malloc.h>
#include <oskit/boolean.h>
#include "sfs.h"

static struct oskit_file_ops file_ops;
static struct oskit_absio_ops file_absio_ops;
static struct oskit_dir_ops dir_ops;
static struct oskit_comsid_ops sid_ops;
static struct oskit_file_secure_ops sfile_ops;
static struct oskit_dir_secure_ops sdir_ops;

#define COMCHECKFILEDIR(_csid,_com,_perm,_rcp) \
        if (_com##->filei.ops == &file_ops) \
                COMCHECK(_com##->sfs->avc,SUBJECT,_csid,FILE,_com##->sid,_perm,_rcp); \
	else \
		COMCHECK(_com##->sfs->avc,SUBJECT,_csid,DIR,_com##->sid,_perm,_rcp)

oskit_error_t sfiledir_create(struct sfilesystem *sfs, 
			     oskit_file_t *file,
			     struct sfiledir **out_sfiledir)
{
    oskit_dir_t *d;
    struct sfiledir *sfiledir;
    oskit_stat_t stats;
    char pathbuf[32]; /* arbitrary, doesn't matter */
    oskit_u32_t linksz;
    oskit_error_t rc;


    rc = oskit_file_stat(file, &stats);
    if (rc)
	    return rc;
    
    sfiledir = (struct sfiledir *) 
	    hashtab_search(sfs->ino2sf, (hashtab_key_t) stats.ino);

    if (sfiledir)
    {
	oskit_file_addref(&sfiledir->filei);
	*out_sfiledir = sfiledir;
	return 0;
    }
    
    sfiledir = malloc(sizeof(struct sfiledir));
    if (!sfiledir)
	    return OSKIT_ENOMEM;
    
    rc = oskit_file_query(file, &oskit_dir_iid, (void**)&d);
    if (rc)
    {
	sfiledir->filei.ops = &file_ops;
        rc = oskit_file_readlink(file, pathbuf, sizeof pathbuf, &linksz);
        if (rc == OSKIT_E_NOTIMPL)
		sfiledir->symlink = FALSE;
	else 
		sfiledir->symlink = TRUE;

    }
    else
    {
	oskit_dir_release(d);
	sfiledir->filei.ops = (struct oskit_file_ops *)&dir_ops;
	sfiledir->sdiri.ops = &sdir_ops;
	sfiledir->symlink = FALSE;
    }

    sfiledir->sfilei.ops = &sfile_ops;
    sfiledir->absioi.ops = &file_absio_ops;
    sfiledir->sidi.ops = &sid_ops;
    sfiledir->count = 1;

    rc = oskit_filepsid_get(sfs->psid, file, &sfiledir->sid);
    if (rc)
    {
	free(sfiledir);
	return rc;
    }

    rc = hashtab_insert(sfs->ino2sf, 
			(hashtab_key_t) stats.ino,
			(hashtab_datum_t) &sfiledir->filei);
    if (rc)
    {
	free(sfiledir);
	return rc;
    }
    
    sfiledir->sfs = sfs;
    oskit_filesystem_addref(&sfs->fsi);

    sfiledir->mountedhere = NULL;
    sfiledir->virtual = FALSE;
    sfiledir->parent = NULL;

    sfiledir->file = file;
    oskit_file_addref(file);

    rc = oskit_file_query(file, &oskit_absio_iid, (void**)&sfiledir->absio);
    if (rc)
	    sfiledir->absio = NULL;

    /*
     * initialize sopenfile_list
     */
    sfiledir->sopenfile_list = NULL;

    *out_sfiledir = sfiledir;
    return 0;
}    



/*
 * oskit_file methods
 */

static OSKIT_COMDECL file_query(oskit_file_t *f,
			       const struct oskit_guid *iid,
			       void **out_ihandle)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*)f;
    assert(sfile);
    assert(sfile->count);
    
    if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
	memcmp(iid, &oskit_posixio_iid, sizeof(*iid)) == 0 ||
	memcmp(iid, &oskit_file_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sfile->filei;
	++sfile->count;
	return 0;
    }

    if (memcmp(iid, &oskit_comsid_iid, sizeof(*iid)) == 0) 
    {
	*out_ihandle = &sfile->sidi;
	++sfile->count;
	return 0;
    }

    if (memcmp(iid, &oskit_file_secure_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sfile->sfilei;
	++sfile->count;
	return 0;
    }

    if (sfile->absio && memcmp(iid, &oskit_absio_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sfile->absioi;
	++sfile->count;
	return 0;
    }

    *out_ihandle = NULL;
    return OSKIT_E_NOINTERFACE;    
}


static OSKIT_COMDECL_U file_addref(oskit_file_t *f)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*)f;
    assert(sfile);
    assert(sfile->count);

    return ++sfile->count;
}


static OSKIT_COMDECL_U file_release(oskit_file_t *f)
{
    struct sfiledir *sfile;
    oskit_stat_t stats;
    oskit_error_t rc;
    unsigned newcount;
    
    sfile = (struct sfiledir*)f; 
    assert(sfile);
    assert(sfile->count);

    newcount = --sfile->count;
    if (newcount == 0)
    {
	rc = oskit_file_stat(sfile->file, &stats);
	if (!rc)
	{
	    (void)hashtab_remove(sfile->sfs->ino2sf, 
				 (hashtab_key_t)stats.ino, 0, 0);
	}
	oskit_filesystem_release(&sfile->sfs->fsi);
	if (sfile->mountedhere)
		oskit_file_release(sfile->mountedhere);
	if (sfile->parent)
		oskit_dir_release(sfile->parent);
	oskit_file_release(sfile->file);
	if (sfile->absio)
		oskit_absio_release(sfile->absio);
	free(sfile);
    }

    return newcount;
}


static OSKIT_COMDECL file_stat(oskit_file_t *f,
			      struct oskit_stat *out_stats)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,GETATTR,&rc);
    
    if (rc)
	    return rc;
    
    return oskit_file_stat(sfile->file,out_stats);
}


static OSKIT_COMDECL file_setstat(oskit_file_t *f, oskit_u32_t mask,
				const struct oskit_stat *stats)
{
    struct sfiledir *sfile;
    oskit_error_t rc;
    security_id_t csid, fsid;
    

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid, sfile, SETATTR, &rc);
    if (rc)
	    return rc;

    if (mask & (OSKIT_STAT_UID | OSKIT_STAT_GID))
    {
	COMCHECKFILEDIR(csid, sfile, CHSIDFROM, &rc);
	if (rc)
		return rc;
	
	rc = oskit_filepsid_chown(sfile->sfs->psid, sfile->sid,
				 mask, stats, &fsid);
	if (rc)
		return rc;

	if (sfile->filei.ops == &file_ops)
		COMCHECK(sfile->sfs->avc,SUBJECT, csid, FILE, fsid, CHSIDTO, &rc);
	else
		COMCHECK(sfile->sfs->avc,SUBJECT, csid, DIR, fsid, CHSIDTO, &rc);
	if (rc)
		return rc;

	if (sfile->filei.ops == &file_ops)
		COMCHECK(sfile->sfs->avc,SUBJECT, fsid, FILE, sfile->sid, FILE_INHERIT, &rc);
	else
		COMCHECK(sfile->sfs->avc,SUBJECT, fsid, DIR, sfile->sid, FILE_INHERIT, &rc);
	if (rc)
		return rc;

	if (sfile->filei.ops == &file_ops)
		COMCHECK(sfile->sfs->avc,SUBJECT, fsid, FILESYSTEM, sfile->sfs->sid, FILE_ASSOCIATE, &rc);
	else
		COMCHECK(sfile->sfs->avc,SUBJECT, fsid, FILESYSTEM, sfile->sfs->sid, DIR_ASSOCIATE, &rc);
	if (rc)
		return rc;
    }

    rc = oskit_file_setstat(sfile->file,mask,stats);

    if (!rc && (mask & (OSKIT_STAT_UID | OSKIT_STAT_GID)))
    {
	rc = oskit_filepsid_set(sfile->sfs->psid,sfile->file,fsid);
	if (rc)
		return rc;
    }

    return rc;
}


static OSKIT_COMDECL file_pathconf(oskit_file_t *f, oskit_s32_t option,
				 oskit_s32_t *out_val)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,PATHCONF,&rc);
    
    if (rc)
	    return rc;

    return oskit_file_pathconf(sfile->file,option,out_val);
}


static OSKIT_COMDECL file_sync(oskit_file_t *f, oskit_bool_t wait)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;
 
    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid, sfile, SYNC, &rc);
    
    if (rc)
	    return rc;
    
    return oskit_file_sync(sfile->file,wait);
}


static OSKIT_COMDECL file_access(oskit_file_t *f, oskit_amode_t mask)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid, sfile, ACCESS, &rc);
    if (rc)
	    return rc;

    if (mask & OSKIT_R_OK)	
    {
	COMCHECKFILEDIR(csid,sfile,READ,&rc);
	if (rc)
		return rc;
    }

    if (mask & OSKIT_W_OK)
    {
	if (sfile->filei.ops == &file_ops)
		COMCHECK(sfile->sfs->avc,SUBJECT, csid, FILE, sfile->sid, WRITE, &rc);
	else
		COMCHECK3(sfile->sfs->avc, SUBJECT, csid, DIR, sfile->sid, 
			  ADD_NAME, REMOVE_NAME, REPARENT, &rc);	
	if (rc)
		return rc;
    }

    if (mask & OSKIT_X_OK)
    {
	if (sfile->filei.ops == &file_ops)	
		COMCHECK(sfile->sfs->avc,SUBJECT, csid, FILE, sfile->sid, EXECUTE, &rc);
	else
		COMCHECK(sfile->sfs->avc,SUBJECT, csid, DIR, sfile->sid, SEARCH, &rc);
	if (rc)
		return rc;
    }

    return oskit_file_access(sfile->file,mask);    
}


static OSKIT_COMDECL file_readlink(oskit_file_t *f,
				  char *buf, oskit_u32_t len,
				  oskit_u32_t *out_actual)
{
    struct sfiledir *sfile;
    oskit_error_t rc;
    security_id_t csid;

    sfile = (struct sfiledir *)f;
    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    if (f->ops != &file_ops)
	    return OSKIT_E_NOTIMPL;

    if (!sfile->symlink)
	    return OSKIT_E_NOTIMPL;

    CSID(&csid);
    COMCHECK(sfile->sfs->avc,SUBJECT, csid, FILE, sfile->sid, READ, &rc);
    
    if (rc)
	    return rc;

    return oskit_file_readlink(sfile->file,buf,len,out_actual);
}




static OSKIT_COMDECL file_open(oskit_file_t *f, oskit_oflags_t iflags,
			      struct oskit_openfile **out_openfile)
{
    struct sfiledir *sfile;
    oskit_error_t rc;
    security_id_t csid;

    sfile = (struct sfiledir *)f;
    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;
    
    CSID(&csid);
    rc = oskit_file_secure_open(&sfile->sfilei,iflags,csid,out_openfile);
    
    return rc;
}


#if 0

static OSKIT_COMDECL file_mount(oskit_file_t *f, oskit_file_t *sub)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,MOUNTON,&rc);
    
    if (rc)
	    return rc;

    if (sfile->mountedhere)
	    oskit_file_release(sfile->mountedhere);

    if (sub)
	    oskit_file_addref(sub);

    sfile->mountedhere = sub;
    return 0;
}

#endif 0

static OSKIT_COMDECL file_getfs(oskit_file_t *f, struct oskit_filesystem **out_fs)
{
    struct sfiledir *sfile;	
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir *)f;

    if (!sfile || !sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,GETFS,&rc);
    
    if (rc)
	    return rc;
    
    oskit_filesystem_addref(&sfile->sfs->fsi);
    *out_fs = &sfile->sfs->fsi;

    return 0;
}


static struct oskit_file_ops file_ops = {
    file_query,
    file_addref,
    file_release,
    file_stat,
    file_setstat,
    file_pathconf,
    file_sync,
    file_sync,
    file_access,
    file_readlink,
    file_open,
    file_getfs
};



/*
 * methods for absio interface of files
 */

static OSKIT_COMDECL afile_query(oskit_absio_t *io,
				const struct oskit_guid *iid,
				void **out_ihandle)
{
    struct sfiledir *sfile;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));
    if (!sfile->count)
	    return OSKIT_E_INVALIDARG;

    /* may be file_query or dir_query */
    return oskit_file_query(&sfile->filei, iid, out_ihandle);
}

static OSKIT_COMDECL_U afile_addref(oskit_absio_t *io)
{
    struct sfiledir *sfile;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));

    return file_addref(&sfile->filei);
}

static OSKIT_COMDECL_U afile_release(oskit_absio_t *io)
{
    struct sfiledir *sfile;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));

     return file_release(&sfile->filei);
}


static OSKIT_COMDECL afile_read(oskit_absio_t *io, void *buf,
			       oskit_off_t offset, oskit_size_t amount,
			       oskit_size_t *out_actual)
{
    struct sfiledir *sfile;	
    security_id_t csid;
    oskit_error_t rc;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));
    if (!sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,READ,&rc);
    
    if (rc)
	    return rc;

    return oskit_absio_read(sfile->absio,buf,offset,amount,out_actual);
}


static OSKIT_COMDECL afile_write(oskit_absio_t *io, const void *buf,
				oskit_off_t offset, oskit_size_t amount,
				oskit_size_t *out_actual)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));
    if (!sfile->count)
	    return OSKIT_E_INVALIDARG;

    if (sfile->filei.ops != &file_ops)
	    return OSKIT_EISDIR;

    CSID(&csid);
    COMCHECK(sfile->sfs->avc,SUBJECT,csid,FILE,sfile->sid,WRITE,&rc);
    
    if (rc)
	    return rc;

    return oskit_absio_write(sfile->absio,buf,offset,amount,out_actual);
}


static OSKIT_COMDECL afile_get_size(oskit_absio_t *io, oskit_off_t *out_size)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));
    if (!sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,GETATTR,&rc);
    
    if (rc)
	    return rc;

    return oskit_absio_getsize(sfile->absio,out_size);
}


static OSKIT_COMDECL afile_set_size(oskit_absio_t *io, oskit_off_t new_size)
{
    struct sfiledir *sfile;
    security_id_t csid;
    oskit_error_t rc;

    if (!io || io->ops != &file_absio_ops)
	    return OSKIT_E_INVALIDARG;
    
    sfile = (struct sfiledir *) ((char *) io - offsetof(struct sfiledir, absioi));
    if (!sfile->count)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECKFILEDIR(csid,sfile,SETATTR,&rc);
    
    if (rc)
	    return rc;

    return oskit_absio_setsize(sfile->absio,new_size);
}


static struct oskit_absio_ops file_absio_ops = {
    afile_query,
    afile_addref,
    afile_release,
    (void *)0,			/* slot reserved for getblocksize */
    afile_read,
    afile_write,
    afile_get_size,
    afile_set_size
};



/*
 * oskit_dir methods; we inherit all the file methods above except query
 */

static OSKIT_COMDECL dir_query(oskit_dir_t *d,
			      const struct oskit_guid *iid,
			      void **out_ihandle)
{
    struct sfiledir *sdir;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);
    
    if (memcmp(iid, &oskit_dir_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sdir->filei;
	++sdir->count;
	return 0;
    }

    if (memcmp(iid, &oskit_dir_secure_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sdir->sdiri;
	++sdir->count;
	return 0;
    }

    return file_query((oskit_file_t *) d, iid, out_ihandle);
}


OSKIT_INLINE oskit_error_t dir_lookup_helper(struct sfiledir *sdir,
					   const char *name,
					   struct sfiledir **out_sfile)
{
    oskit_file_t *file;
    oskit_error_t rc;

    rc = oskit_dir_lookup((oskit_dir_t*)sdir->file,name,&file);
    if (rc)
	    return rc;

    rc = sfiledir_create(sdir->sfs, file, out_sfile);
    oskit_file_release(file);
    return rc;
}


static OSKIT_COMDECL dir_lookup(oskit_dir_t *d, const char *name,
			       oskit_file_t **out_file)
{
    security_id_t csid;
    struct sfiledir *sdir, *sfile;
    oskit_error_t rc;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);
    COMCHECK(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, SEARCH, &rc);
    
    if (rc)
	    return rc;

    if (sdir->virtual && !strcmp(name,".."))
    {
	if (sdir->parent)
		*out_file = (oskit_file_t *) sdir->parent;
	else
		*out_file = &sdir->filei;

	oskit_file_addref(*out_file);
	return 0;
    }

    rc = dir_lookup_helper(sdir,name,&sfile);
    if (rc)
	    return rc;
    
    if (sfile->mountedhere)
    {
	*out_file = sfile->mountedhere;
	oskit_file_addref(sfile->mountedhere);
	oskit_file_release(&sfile->filei);
    }
    else
    {
	*out_file = &sfile->filei;
    }

    return 0;
}


static OSKIT_COMDECL dir_create(oskit_dir_t *d, const char *name,
			       oskit_bool_t excl, oskit_mode_t mode,
			       oskit_file_t **out_file)
{
    struct sfiledir *sdir;
    security_id_t csid, fsid;
    oskit_error_t rc;

    
    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);

    rc = oskit_filepsid_create(sdir->sfs->psid, 
			      csid, sdir->sid, SECCLASS_FILE,
			      &fsid);
    
    if (rc)
	    return rc;

    return oskit_dir_secure_create(&sdir->sdiri,name,excl,mode,fsid,out_file);
}



static OSKIT_COMDECL dir_link(oskit_dir_t *d, char *name,
			     oskit_file_t *f)
{
    struct sfiledir *sdir, *sfile;
    oskit_error_t rc;
    security_id_t csid;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    assert(f);
    if (f->ops != &file_ops && f->ops != (struct oskit_file_ops *) &dir_ops)
	    return OSKIT_EXDEV;
    
    sfile = (struct sfiledir*)f;
    assert(sfile->count);

    CSID(&csid);
    COMCHECK2(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, SEARCH, ADD_NAME, &rc);
    if (rc)
    {
	
	return rc;
    }

    COMCHECKFILEDIR(csid,sfile,LINK,&rc);
    
    if (rc)
	    return rc;

    return oskit_dir_link((oskit_dir_t*)sdir->file,name,sfile->file);
}



static OSKIT_COMDECL dir_unlink(oskit_dir_t *d, const char *name)
{
    struct sfiledir *sdir, *sfile;
    oskit_stat_t stats;
    security_id_t csid;
    oskit_error_t rc;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);

    COMCHECK2(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, SEARCH, REMOVE_NAME, &rc);
    if (rc)
    {
	
	return rc;
    }

    rc = dir_lookup_helper(sdir, name, &sfile);
    if (rc)
	    return rc;

    COMCHECKFILEDIR(csid, sfile, UNLINK, &rc);
    
    if (rc)
    {
	oskit_file_release(&sfile->filei);
	return rc;
    }

    rc = oskit_file_stat(sfile->file, &stats);
    oskit_file_release(&sfile->filei);
    if (rc)
	    return rc;

    rc = oskit_dir_unlink((oskit_dir_t*)sdir->file,name);
    if (rc)
	    return rc;

    if (stats.nlink == 1)
	    return oskit_filepsid_unset(sdir->sfs->psid, stats.ino);

    return 0;
}


static OSKIT_COMDECL dir_rename(oskit_dir_t *old_d, char *old_name,
			       oskit_dir_t *new_d, char *new_name)
{
    struct sfiledir *sdirf, *sdirt, *sfilef, *sfilet;
    oskit_dir_t *dir;
    security_id_t csid;
    oskit_stat_t stats;
    oskit_bool_t unset_filet = FALSE;    
    oskit_error_t rc;

    sdirf = (struct sfiledir*)old_d;
    assert(sdirf);
    assert(sdirf->count);

    assert(new_d);
    if (new_d->ops != &dir_ops)
	    return OSKIT_EXDEV;
    
    sdirt = (struct sfiledir*)new_d;
    assert(sdirt->count);

    CSID(&csid);

    COMCHECK2(sdirf->sfs->avc,SUBJECT, csid, DIR, sdirf->sid, SEARCH, REMOVE_NAME, &rc);
    if (!rc)
	    COMCHECK2(sdirt->sfs->avc,SUBJECT, csid, DIR, sdirt->sid, SEARCH, ADD_NAME, &rc);
    if (rc)
	    return rc;

    /*
     * This lookup might seem unnecessary unless sdirf != sdirt.
     * However, it _is_ important, in order to prevent
     * renaming of the security binding directory.
     */
    rc = dir_lookup_helper(sdirf, old_name, &sfilef);
    if (rc)
	    return rc;

    if (sdirf != sdirt)
    {
	rc = oskit_file_query(&sfilef->filei, &oskit_dir_iid, (void**)&dir);
	if (!rc)
	{
	    oskit_dir_release(dir);
	    COMCHECK(sfilef->sfs->avc,SUBJECT, csid, DIR, sfilef->sid, REPARENT, &rc);
	    if (rc)
	    {
		
		oskit_file_release(&sfilef->filei);
		return rc;
	    }
	}
    }

    oskit_file_release(&sfilef->filei);

    rc = dir_lookup_helper(sdirt, new_name, &sfilet);
    if (!rc)
    {
	COMCHECK(sdirt->sfs->avc,SUBJECT, csid, DIR, sdirt->sid, REMOVE_NAME, &rc);
	if (rc)
	{
	    oskit_file_release(&sfilet->filei);
	    return rc;
	}

	rc = oskit_file_query(&sfilet->filei, &oskit_dir_iid, (void**)&dir);
	if (rc)
	{
	    COMCHECK(sfilet->sfs->avc,SUBJECT, csid, FILE, sfilet->sid, UNLINK, &rc); 
	}
	else
	{
	    oskit_dir_release(dir);	    
	    COMCHECK(sfilet->sfs->avc,SUBJECT, csid, DIR, sfilet->sid, RMDIR, &rc); 
	}

	if (rc)
	{
	    oskit_file_release(&sfilet->filei);
	    return rc;
	}

	rc = oskit_file_stat(sfilet->file, &stats);
	oskit_file_release(&sfilet->filei);
	if (rc)
		return rc;

	if (stats.nlink == 1)
		unset_filet = TRUE;
    }
    else if (rc != OSKIT_ENOENT)	
	    return rc;
    
    rc = oskit_dir_rename((oskit_dir_t*)sdirf->file,
			 old_name,
			 (oskit_dir_t*)sdirt->file,
			 new_name);
    if (rc)
	    return rc;
    
    if (unset_filet)
	    return oskit_filepsid_unset(sdirt->sfs->psid, stats.ino);

    return 0;
}


OSKIT_INLINE oskit_error_t file_create_checks(security_id_t csid,
					    struct sfiledir *sdir,
					    oskit_bool_t isdir,
					    security_id_t new_fsid)
{
    oskit_error_t rc;

    COMCHECK(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, ADD_NAME, &rc);
    if (rc)
	    return rc;

    COMCHECK(sdir->sfs->avc,SUBJECT, csid, FILESYSTEM, sdir->sfs->sid, ADD_FILE, &rc);
    if (rc)
	    return rc;

    if (isdir)
	    COMCHECK(sdir->sfs->avc,SUBJECT, csid, DIR, new_fsid, CREATE, &rc);
    else
	    COMCHECK(sdir->sfs->avc,SUBJECT, csid, FILE, new_fsid, CREATE, &rc);
    if (rc)
	    return rc;

    if (isdir)
	    COMCHECK(sdir->sfs->avc,SUBJECT, new_fsid, FILESYSTEM, sdir->sfs->sid, DIR_ASSOCIATE, &rc);
    else
	    COMCHECK(sdir->sfs->avc,SUBJECT, new_fsid, FILESYSTEM, sdir->sfs->sid, FILE_ASSOCIATE, &rc);

    return rc;
}


/*
 * TBD  
 *
 * In DIR_MKCALL, after the file is created, compare
 * the acontext of 'fsid' against the uid/gid of the file.
 * If they are inconsistent, then adjust the uid/gid.
 *
 * This requires a Unix identity server to translate
 * between uid/gid pairs and acontexts.
 */ 

#define DIR_MKCALL(call, d, isdir, fsid, name, args...) { \
    struct sfiledir *sdir; \
    security_id_t csid; \
    oskit_file_t *file; \
    oskit_bool_t rfsid = FALSE; \
    oskit_error_t rc; \
\
\
    sdir = (struct sfiledir*)(d); \
    assert(sdir); \
    assert(sdir->count); \
\
    CSID(&csid); \
    COMCHECK(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, SEARCH, &rc); \
    if (rc) \
	    return rc; \
\
    if (!fsid) \
    { \
        rc = oskit_filepsid_create(sdir->sfs->psid,  \
	    		          csid, sdir->sid, \
				  (isdir) ? SECCLASS_DIR :SECCLASS_FILE, \
			          &(fsid)); \
        if (rc) \
	        return rc; \
        rfsid = TRUE; \
    } \
\
    rc = file_create_checks(csid, sdir, (isdir), (fsid)); \
     \
    if (rc) \
	    return rc; \
\
    rc = oskit_dir_##call##((oskit_dir_t*)sdir->file,(name),##args); \
    if (rc) \
	    return rc; \
\
    rc = oskit_dir_lookup((oskit_dir_t*)sdir->file,(name),&file); \
    if (rc) \
	    return rc; \
\
    rc = oskit_filepsid_set(sdir->sfs->psid, file, (fsid)); \
    oskit_file_release(file); \
    if (rc) \
    { \
        if (isdir) \
	      oskit_dir_rmdir((oskit_dir_t*)sdir->file,(name)); \
        else \
	      oskit_dir_unlink((oskit_dir_t*)sdir->file,(name)); \
    } \
    return rc; \
}						      

static OSKIT_COMDECL dir_mkdir(oskit_dir_t *d, const char *name,
			      oskit_mode_t mode)
{
    security_id_t fsid = SECSID_NULL;
    DIR_MKCALL(mkdir,d,TRUE,fsid,name,mode);
}


static OSKIT_COMDECL dir_rmdir(oskit_dir_t *d, const char *name)
{
    struct sfiledir *sdir, *sfile;
    security_id_t csid;
    oskit_stat_t stats;
    oskit_error_t rc;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);

    COMCHECK2(sdir->sfs->avc,SUBJECT, csid, DIR, sdir->sid, SEARCH, REMOVE_NAME, &rc);
    if (rc)
	    return rc;

    rc = dir_lookup_helper(sdir, name, &sfile);
    if (rc)
	    return rc;

    COMCHECK(sfile->sfs->avc,SUBJECT, csid, DIR, sfile->sid, RMDIR, &rc);
    
    if (rc)
    {
	oskit_file_release(&sfile->filei);
	return rc;
    }

    rc = oskit_file_stat(sfile->file, &stats);
    oskit_file_release(&sfile->filei);
    if (rc)
	    return rc;

    rc = oskit_dir_rmdir((oskit_dir_t*)sdir->file,name);
    if (rc)
	    return rc;

    return oskit_filepsid_unset(sdir->sfs->psid, stats.ino);
}


static OSKIT_COMDECL dir_getdirentries(oskit_dir_t *d, oskit_u32_t *inout_ofs,
				      oskit_u32_t nentries,
				      struct oskit_dirents *out_dirents)
{
    struct sfiledir *sdir;
    security_id_t csid;
    oskit_error_t rc = 0;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);

    COMCHECK(sdir->sfs->avc,SUBJECT,csid,DIR,sdir->sid,READ,&rc);
    if (rc)
	    return rc;

    return oskit_dir_getdirentries((oskit_dir_t*)sdir->file, inout_ofs, 
				   nentries, out_dirents);
}


static OSKIT_COMDECL dir_mknod(oskit_dir_t *d, char *name,
			      oskit_mode_t mode, oskit_dev_t dev)
{
    security_id_t fsid = SECSID_NULL;
    DIR_MKCALL(mknod,d,FALSE,fsid,name,mode,dev);
}


static OSKIT_COMDECL dir_symlink(oskit_dir_t *d, char *link_name,
				char *dest_name)
{
    security_id_t fsid = SECSID_NULL;
    DIR_MKCALL(symlink,d,FALSE,fsid,link_name,dest_name);
}


static OSKIT_COMDECL dir_reparent(oskit_dir_t *d, oskit_dir_t *parent,
				 oskit_dir_t **out_dir)
{
    struct sfiledir *sdir, *svdir;
    security_id_t csid;
    oskit_error_t rc;

    sdir = (struct sfiledir*)d;
    assert(sdir);
    assert(sdir->count);

    CSID(&csid);
    COMCHECK(sdir->sfs->avc,SUBJECT,csid,DIR,sdir->sid,VIRTUALIZE,&rc);
    
    if (rc)
	    return rc;

    svdir = malloc(sizeof(struct sfiledir));
    if (!svdir)
	    return OSKIT_ENOMEM;
    
    svdir->filei.ops = (struct oskit_file_ops*)&dir_ops;
    svdir->absioi.ops = &file_absio_ops;
    svdir->sidi.ops = &sid_ops;
    svdir->count = 1;
    svdir->sid = sdir->sid;
    svdir->sfs = sdir->sfs;
    oskit_filesystem_addref(&sdir->sfs->fsi);
    svdir->mountedhere = NULL;
    svdir->file = sdir->file;
    oskit_file_addref(sdir->file);
    svdir->absio = sdir->absio;
    if (sdir->absio)
	    oskit_absio_addref(svdir->absio);

    svdir->virtual = TRUE;
    svdir->parent = parent;
    if (parent)
	    oskit_dir_addref(parent);

    *out_dir = (oskit_dir_t*)&svdir->filei;
    return 0;
}


static struct oskit_dir_ops dir_ops = {
    dir_query,
    file_addref,
    file_release,
    file_stat,
    file_setstat,
    file_pathconf,
    file_sync,
    file_sync, 		
    file_access,
    file_readlink,
    file_open,
    file_getfs,
    dir_lookup,
    dir_create,
    dir_link,
    dir_unlink,
    dir_rename,
    dir_mkdir,
    dir_rmdir,
    dir_getdirentries,
    dir_mknod,
    dir_symlink,
    dir_reparent
};




/*
 * oskit_comsid methods
 */

static OSKIT_COMDECL sid_query(oskit_comsid_t *s,
			      const struct oskit_guid *iid,
			      void **out_ihandle)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) s - 
				 offsetof(struct sfiledir, sidi));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_query(&sfile->filei, iid, out_ihandle);
}


static OSKIT_COMDECL_U sid_addref(oskit_comsid_t *s)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) s - 
				 offsetof(struct sfiledir, sidi));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_addref(&sfile->filei);
}


static OSKIT_COMDECL_U sid_release(oskit_comsid_t *s)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) s - 
				 offsetof(struct sfiledir, sidi));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_release(&sfile->filei);
}


static OSKIT_COMDECL sid_get(oskit_comsid_t *s,
			    security_class_t *outclass,
			    security_id_t *outsid)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) s - 
				 offsetof(struct sfiledir, sidi));
    assert(sfile);
    assert(sfile->count);

    *outclass = ((sfile->filei.ops == &file_ops) ? 
		 SECCLASS_FILE : 
		 SECCLASS_DIR);
    *outsid = sfile->sid;
    return 0;
}

#define ALL_PERMS 0xffffffff

/*
 * Evaluate the open file with respect to the relabeled file it refers to.
 * Returns a set of flags to be revoked.
 */
static oskit_oflags_t file_open_evaluate(struct sfiledir *sfile,
					 struct sopenfile *sofile)
{
    oskit_error_t rc;
    security_id_t ofsid = sofile->sid;
    oskit_oflags_t revoke_flags = 0;
    
    if (sfile->filei.ops == &file_ops)
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, FILE, sfile->sid, OPEN_ASSOCIATE, &rc);
    else
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, DIR, sfile->sid, OPEN_ASSOCIATE, &rc);
    if (rc)
	    return ALL_PERMS;	/* revoke all perms */

    if (sofile->flags & OSKIT_O_RDONLY)
    {
	    COMCHECKFILEDIR(ofsid,sfile,READ,&rc);
	    if (rc)
	    {
		/* 
		 * retract OSKIT_O_RDONLY
		 */
		revoke_flags |= OSKIT_O_RDONLY;
	    }
    }

    if (sofile->flags & OSKIT_O_WRONLY)
    {
	assert(sfile->filei.ops == &file_ops);
	if (sofile->flags & OSKIT_O_APPEND)
	{
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, FILE, sfile->sid, APPEND, &rc);
	    if (rc)
	    {
		revoke_flags |= (OSKIT_O_WRONLY | OSKIT_O_APPEND);
	    }
	}
	else
	{
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, FILE, sfile->sid, WRITE, &rc);
	    if (rc)
            {
		revoke_flags |= (OSKIT_O_WRONLY | OSKIT_O_APPEND);
	    }
	}
    }

    return revoke_flags;
}

static OSKIT_COMDECL sid_set(oskit_comsid_t *s,
			    security_id_t newsid)
{
    struct sfiledir *sfile;
    struct sopenfile *sofile;
    oskit_oflags_t revoke_flags;
    oskit_error_t rc;

    sfile = (struct sfiledir*) ((char *) s - 
				 offsetof(struct sfiledir, sidi));
    assert(sfile);
    assert(sfile->count);

    rc = oskit_filepsid_set(sfile->sfs->psid, sfile->file, newsid);
    if (rc)
	return(rc);

    sfile->sid = newsid;

    /*
     * Evaluate all sopenfiles for this file wrt newsid
     */
    sofile = sfile->sopenfile_list;

    if (sofile == NULL)
        return 0;

    do 
    {
	revoke_flags = file_open_evaluate(sfile, sofile);
	sofile->flags = sofile->flags & (~ revoke_flags);
    	sofile = sofile->next;
    } while (sofile != sfile->sopenfile_list);

    return 0;
}


static struct oskit_comsid_ops sid_ops = {
    sid_query,
    sid_addref,
    sid_release,
    sid_get,
    sid_set
};




/*
 * oskit_dir_secure methods
 */

static OSKIT_COMDECL sdir_query(oskit_dir_secure_t *d,
			       const struct oskit_guid *iid,
			       void **out_ihandle)
{
    struct sfiledir *sdir;

    sdir = (struct sfiledir*) ((char *) d - 
				 offsetof(struct sfiledir, sdiri));
    assert(sdir);
    assert(sdir->count);

    return oskit_file_query(&sdir->filei, iid, out_ihandle);
}


static OSKIT_COMDECL_U sdir_addref(oskit_dir_secure_t *d)
{
    struct sfiledir *sdir;

    sdir = (struct sfiledir*) ((char *) d - 
			       offsetof(struct sfiledir, sdiri));
    assert(sdir);
    assert(sdir->count);

    return oskit_file_addref(&sdir->filei);
}


static OSKIT_COMDECL_U sdir_release(oskit_dir_secure_t *d)
{
    struct sfiledir *sdir;

    sdir = (struct sfiledir*) ((char *) d - 
			       offsetof(struct sfiledir, sdiri));
    assert(sdir);
    assert(sdir->count);

    return oskit_file_release(&sdir->filei);
}


static OSKIT_COMDECL sdir_create(oskit_dir_secure_t *d, 
				const char *name,
				oskit_bool_t excl, 
				oskit_mode_t mode,
				security_id_t fsid,
				oskit_file_t **out_file)
{
    struct sfiledir *sdir, *sfile;
    security_id_t csid;
    oskit_file_t *file;
    oskit_error_t rc;

    sdir = (struct sfiledir*) ((char *) d - 
			       offsetof(struct sfiledir, sdiri));
    assert(sdir);
    assert(sdir->count);

    rc = dir_lookup((oskit_dir_t*)&sdir->filei, name, out_file);
    if (!rc)
	    return 0;

    if (rc != OSKIT_ENOENT)	
	    return rc;

    CSID(&csid);

    rc = file_create_checks(csid, sdir, FALSE, fsid);
    
    if (rc)
	    return rc;

    rc = oskit_dir_create((oskit_dir_t*)sdir->file,name,excl,mode,&file);
    if (rc)
	    return rc;

    /*
     * TBD
     *
     * Compare the acontext of 'sid' against the uid/gid
     * bound to the file.  If they are inconsistent, then use
     * oskit_file_setstat to change the uid/gid to the appropriate 
     * values.
     *
     * This requires a Unix identity server to translate
     * between uid/gid pairs and acontexts.
     */

    rc = oskit_filepsid_set(sdir->sfs->psid, file, fsid);
    if (rc)
    {
	oskit_dir_unlink((oskit_dir_t*)sdir->file,name);
	oskit_file_release(file);
	return rc;
    }

    rc = sfiledir_create(sdir->sfs, file, &sfile);
    oskit_file_release(file);
    if (rc)
	    return rc;

    *out_file = &sfile->filei;
    return 0;
}


static OSKIT_COMDECL sdir_mkdir(oskit_dir_secure_t *sd, 
			       const char *name,
			       oskit_mode_t mode,
			       security_id_t fsid)
{
    oskit_dir_t *d;

    assert(sd);
    
    d = (oskit_dir_t *) ((char *) sd - 
			offsetof(struct sfiledir, sdiri));

    DIR_MKCALL(mkdir,d,TRUE,fsid,name,mode);
}


static struct oskit_dir_secure_ops sdir_ops = {
    sdir_query,
    sdir_addref,
    sdir_release,
    sdir_create,
    sdir_mkdir
};



/*
 * oskit_file_secure methods
 */

static OSKIT_COMDECL sfile_query(oskit_file_secure_t *f,
				const struct oskit_guid *iid,
				void **out_ihandle)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) f - 
				offsetof(struct sfiledir, sfilei));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_query(&sfile->filei, iid, out_ihandle);
}


static OSKIT_COMDECL_U sfile_addref(oskit_file_secure_t *f)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) f - 
			       offsetof(struct sfiledir, sfilei));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_addref(&sfile->filei);
}


static OSKIT_COMDECL_U sfile_release(oskit_file_secure_t *f)
{
    struct sfiledir *sfile;

    sfile = (struct sfiledir*) ((char *) f - 
			       offsetof(struct sfiledir, sfilei));
    assert(sfile);
    assert(sfile->count);

    return oskit_file_release(&sfile->filei);
}

OSKIT_INLINE oskit_error_t file_open_checks(security_id_t csid,
					  struct sfiledir *sfile,
					  security_id_t ofsid,
					  oskit_oflags_t oflags)
{
    oskit_error_t rc;

    
    if (sfile->filei.ops == &file_ops)
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, FILE, sfile->sid, OPEN_ASSOCIATE, &rc);
    else
	    COMCHECK(sfile->sfs->avc,SUBJECT, ofsid, DIR, sfile->sid, OPEN_ASSOCIATE, &rc);
    
    if (rc)
	    return rc;

    if (csid != ofsid)
    {
	COMCHECK(sfile->sfs->avc,SUBJECT,csid,OPENFILE,ofsid,CREATE,&rc);
	if (rc)
		return rc;
    }

    if (oflags & OSKIT_O_TRUNC)
    {
	COMCHECKFILEDIR(csid,sfile,SETATTR,&rc);
	if (rc)
		return rc;
    }
	
    if (oflags & OSKIT_O_RDONLY)
    {
	COMCHECKFILEDIR(ofsid,sfile,READ,&rc);
	if (rc)
		return rc;
    }

    if (oflags & OSKIT_O_WRONLY)
    {
	if (sfile->filei.ops != &file_ops)
		return OSKIT_EISDIR;
	    
	COMCHECK(sfile->sfs->avc,SUBJECT,ofsid,FILE,sfile->sid,WRITE,&rc);
	if (rc)
	{
	    if (oflags & OSKIT_O_APPEND)
	    {
		rc = 0;
		COMCHECK(sfile->sfs->avc,SUBJECT,ofsid,FILE,sfile->sid,APPEND,&rc);
		if (rc)
			return rc;
	    }
	    else
	    {
		return rc;
	    }
	}		
    }

    return 0;
}


static OSKIT_COMDECL sfile_open(oskit_file_secure_t *f,
			       oskit_oflags_t flags,
			       security_id_t ofsid,
			       struct oskit_openfile **out_openfile)
{
    struct sfiledir *sfile;
    oskit_openfile_t *ofile;
    struct sopenfile *sofile;
    security_id_t csid;
    oskit_error_t rc;

    sfile = (struct sfiledir*) ((char *) f - 
			       offsetof(struct sfiledir, sfilei));
    assert(sfile);
    assert(sfile->count);

    CSID(&csid);

    rc = file_open_checks(csid, sfile, ofsid, flags);
    if (rc)
	    return rc;

    rc = oskit_file_open(sfile->file, flags, &ofile);
    if (rc)
	    return rc;
    if (!ofile)
    {
	rc = oskit_soa_openfile_from_absio(sfile->file, NULL, flags, &ofile);
	if (rc)
	{
	    oskit_stream_t *stream;
	    rc = oskit_file_query(sfile->file, 
				  &oskit_stream_iid, (void **) &stream);
	    if (rc)
	    {
		rc = oskit_soa_openfile_from_stream(sfile->file, NULL, flags, 
						    &ofile);
	    }
	    else
	    {
		rc = oskit_soa_openfile_from_stream(sfile->file, stream, flags,
						    &ofile);		
		oskit_stream_release(stream);
	    }
	}

	if (rc)
		return rc;
    }

    rc = sopenfile_create(sfile, ofile, flags, 
			  (sfile->filei.ops == &file_ops) ? FALSE : TRUE, 
			  ofsid,
			  &sofile);
    oskit_openfile_release(ofile);
    if (rc)
	    return rc;
    
    *out_openfile = &sofile->ofilei;
    return 0;    
}


static OSKIT_COMDECL sfile_chsid(oskit_file_secure_t *f,
			         security_id_t newsid)
{
    struct sfiledir *sfile;
    security_id_t csid;
    security_id_t fsid;
    oskit_error_t rc;

    sfile = (struct sfiledir*) ((char *) f - 
			       offsetof(struct sfiledir, sfilei));
    fsid = sfile->sid;

    assert(sfile);
    assert(sfile->count);

    CSID(&csid);

    COMCHECKFILEDIR(csid,sfile,CHSIDFROM,&rc);
    if (rc)
	return rc;

    COMCHECKFILEDIR(newsid, sfile, FILE_INHERIT, &rc);
    if (rc)
	return rc;

    if (sfile->filei.ops == &file_ops)
	COMCHECK(sfile->sfs->avc,SUBJECT, csid, FILE, newsid, CHSIDTO, &rc);
    else
	COMCHECK(sfile->sfs->avc,SUBJECT, csid, DIR, newsid, CHSIDTO, &rc);
    if (rc)
	return rc;

    if (sfile->filei.ops == &file_ops)
	COMCHECK(sfile->sfs->avc,SUBJECT,newsid,FILESYSTEM,sfile->sfs->sid,FILE_ASSOCIATE,&rc);
    else
	COMCHECK(sfile->sfs->avc,SUBJECT,newsid,FILESYSTEM,sfile->sfs->sid,DIR_ASSOCIATE,&rc);
    if (rc)
	return rc;


    /*
     * Change to new sid
     */

    rc = oskit_comsid_set(&(sfile->sidi), newsid);

    return(rc);
}

static struct oskit_file_secure_ops sfile_ops = {
    sfile_query,
    sfile_addref,
    sfile_release,
    sfile_open,
    sfile_chsid
};

