/*
 * @COPYRIGHT@
 *
 * Scout Version 1.0
 * 
 * Copyright 1998 Arizona Board of Regents, on behalf of
 *         The University of Arizona
 *         All Rights Reserved
 *
 * @COPYRIGHT@
 *
 * $\RCSfile: wimp_interface.c,v $
 *
 * HISTORY
 * $\Log: wimp_interface.c,v $
 * Revision 1.7  1998/01/31 18:43:18  mjk
 * replaced copyright
 *
 * Revision 1.6  1998/01/28 18:23:56  mjk
 * copyright
 *
 * Revision 1.5  1997/09/02 21:27:30  acb
 * Circle drawing works now
 *
 * Revision 1.4  1997/08/27 20:15:36  acb
 * Fixed a bug and got rid of some warnings.
 *
 * Revision 1.3  1997/08/27 18:02:51  acb
 * Moved WinMgrIface initialization to wimp.c
 *
 * Revision 1.2  1997/06/09 22:45:11  acb
 * Added events, modified wimp_draw_string() to do blit of characters
 * with transparent background, added copy_area() and set_window_title()
 * functions.
 *
 * Revision 1.1  1997/05/30 21:41:24  acb
 * Initial revision
 *
 * Revision 1.1  1997/04/15 18:15:49  acb
 * Initial revision
 *
 * Revision 1.1  1997/04/09 21:33:31  acb
 * Initial revision
 *
 * Revision 1.1  1997/03/25 22:19:39  acb
 * Initial revision
 *
 *
 */
#include <oskit/wimpi.h>

#include <screen.h>
#include <bitblit.h>
#include <border.h>
#include <clip.h>
#include <destroy.h>
#include <erase_win.h>
#include <font.h>
#include <font_subs.h>
#include <graph_subs.h>
#include <intersect.h>
#include <new_window.h>
#include <subs.h>
#include <update.h>
#include <win_subs.h>

#include <wimp_wintree.h>
#include <wimp_internal.h>

#define FONT_ASCENT 12     /* BAD */

extern int traceLevelWIMP;

void
wimpi_blank_save_window (WINDOW *win)
{
    int bwid = win->borderwid, wide, high;
    BITMAP *tmp;
    
    tmp = bit_alloc(BIT_WIDE(win->border),BIT_HIGH(win->border),
		    (DATA*)0,BIT_DEPTH(win->border));
    
    CLEAR(tmp,PUTOP(BIT_CLR,win->style));

    border_in_bitmap (win, tmp, BORDER_THIN);
    
    win->save = tmp;
    
    wide = win->border->wide - (bwid<<1);
    high = win->border->high - (bwid<<1) - win->title_high;
    
    win->window = bit_create(win->save, bwid, bwid + win->title_high, 
			     wide, high);
}

void
wimpi_send_expose_event (WINDOW *win)
{
    wimpiEvent e;
    
    e = (wimpiEvent)malloc(sizeof(*e));
    e->type = EXPOSE_EVENT;
    e->winId = win->winId;

    /* Can assume that, if a window is exposed, its clip list is no
       longer valid. */
    win->flags &= ~(W_CLIPDONE);

    if (!(win->flags & W_ACTIVE)) {
	/*
	 * Ugly: Create blank save window for window that is obscured and
	 * is exposed for the first time.
	 */
	wimpi_blank_save_window (win);
    }
    
    wimpiEventDeliver (win->wimpi, e);
}

void
wimpi_send_mouse_event (WINDOW *win, wimpiEventType type, int button)
{
    wimpiSession mr = win->wimpi->session;
    wimpiEvent e;
    int bwid, x, y;

    e = (wimpiEvent)malloc(sizeof(*e));
    e->type = type;
    e->winId = win->winId;

    bwid = win->borderwid;

    x = mr->mousex - win->x0 - bwid;
    y = mr->mousey - win->y0 - bwid;

    x = BETWEEN (0, x, win->window->wide);
    y = BETWEEN (0, y, win->window->high);

    e->event.mouseEvent.x = x;
    e->event.mouseEvent.y = y;
    e->event.mouseEvent.button = button;

    wimpiEventDeliver (win->wimpi, e);
}

void
wimpi_send_move_resize_event (WINDOW *win) 
{
    wimpiEvent e;

    e = (wimpiEvent)malloc (sizeof(*e));
    e->type = WINDOW_MOVE_RESIZE_EVENT;
    e->winId = win->winId;

    e->event.moveResizeEvent.x = win->x0 + win->borderwid;
    e->event.moveResizeEvent.y = win->y0 + win->borderwid + win->title_high;
    e->event.moveResizeEvent.wide = win->window->wide;
    e->event.moveResizeEvent.high = win->window->high;

    wimpiEventDeliver (win->wimpi, e);
}

void
wimpi_send_destroy_event (WINDOW *win) 
{
    wimpiEvent e;

    e = (wimpiEvent)malloc (sizeof(*e));
    e->type = WINDOW_DESTROY_EVENT;
    e->winId = win->winId;

    wimpiEventDeliver (win->wimpi, e);
}

wimpiWindow
wimpi_create_window (wimpiWindow parent, int x, int y, 
		    int wide, int high, int color, int has_border)
{
    wimpiSession mr = parent->w->session;
    WINDOW *win, *parent_win = NULL;
    wimpiWindow newsw;
    
    if (parent->winId) {
	parent_win = wimpi_get_window (parent->w, parent->winId);
	x += parent_win->borderwid;
	y += parent_win->borderwid;
    }

    if (!parent->w->window_map) {
	/* Trying to create another window on a single window path.
	 * We've somewhat arbitrarily decided we won't allow this.
	 */
      /* XXX
	traceLog(WIMP, TR_ERRORS, "Cannot create another window on a single-window path");
	*/
	return NULL;
    }

    win = wimp_internal_create_window (mr, parent->w, parent_win, x, y, wide, high,
				       has_border);

    if (!win)
	return NULL;

    if (!winmapBind(parent->w->window_map, win->winId, win)) {
        fprintf(stderr, "wimpi_create_window: mapBind failed\n");
        return NULL;
    }

    win->style = PUTBCOLOR(win->style,color);
    win->op = PUTFCOLOR(win->style, 0);
    win->op = PUTFCOLOR(win->op, 0);

    newsw = (wimpiWindow)malloc(sizeof(struct wimpiWindow));
    newsw->w = parent->w;
    newsw->winId=win->winId;
    
    return newsw;
}

void
wimpi_destroy_window (wimpiWindow sw)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win)
	return;

    wimp_internal_destroy_window (win);
}

void
wimpi_map_window (wimpiWindow sw) 
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || win->is_mapped) 
	return;

    if (win->outside_parent) {
	win->is_mapped = TRUE;
	return;
    }

     /* If there is an ancestor of window unmapped, do not map this window. */
    if (!win->parent->is_visible) {
	win->is_mapped = TRUE;
	win->is_visible = FALSE;
	return;
    }

    wimp_internal_map_window (win);
}

void
wimpi_unmap_window (wimpiWindow sw)
{
    WINDOW *win;
    
    win = wimpi_get_window (sw->w, sw->winId);
    
    if (!win || !win->is_mapped) 
	return;

    if (!win->is_visible) {
	win->is_mapped = FALSE;
	return;
    }

    wimp_internal_unmap_window (win);
}

void
wimpi_raise_window (wimpiWindow sw) 
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible) 
	return;

    wimp_internal_map_window (win);
}

void
wimpi_lower_window (wimpiWindow sw) 
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible) 
	return;

    wimp_internal_lower_window (win);
}

void
wimpi_fill_rectangle (wimpiWindow sw, int x, int y, int wide, 
		     int high)
{
    WINDOW *win;
    int color;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    color = GETFCOLOR(win->op);

    wimp_internal_fill_rectangle (win, x, y, wide, high, color);
}

void
wimpi_set_window_background (wimpiWindow sw, int color)
{
    WINDOW *win;

    /* This also erases the contents of the window */
    win = wimpi_get_window (sw->w, sw->winId);

    if (!win)
	return;

    win->style = PUTBCOLOR(win->style,color);

    if (!win->is_visible)
	return;

    wimp_internal_fill_rectangle(win, 0, 0, win->window->wide, 
			win->window->high, color);
}

void
wimpi_clear_area (wimpiWindow sw, int x, int y, int wide, 
		 int high, bool exposures) 
{
    WINDOW *win;
    int color; 

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    color = GETBCOLOR (win->style); 

    if (wide == 0) {
	wide = win->window->wide - x;
    }

    if (high == 0) {
	high = win->window->high - y;
    }

    wimp_internal_fill_rectangle(win, x, y, wide, high, color);

    if (exposures) {
	/* Generate expose event */
    }
}

void
wimpi_move_resize_window (wimpiWindow sw, int x, int y, int wide, 
		 int high) 
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win)
	return;

    wimp_internal_move_resize_window (win, x, y, wide, high);
}

void
wimpi_make_child_window (wimpiToplevelWindow w, Window parent, wimpiToplevelWindow child) 
{
    WINDOW *childwin, *win;

    win = wimpi_get_window (w, parent);

    if (!win)
	return;

    childwin = child->win;

    wimpi_remove_sibling (childwin);

    childwin->parent = win;

    if (win->flags & W_ACTIVE) {
	win->flags &= ~W_ACTIVE;
	if (win->is_visible) {
	    save_win (win);
	} 
    }

    wimpi_make_top_sibling (childwin);

    /* Remove border -- bitmaps are fixed below */
    childwin->borderwid = 0;
    childwin->title_high = 0;
    childwin->has_border = FALSE;

    wimp_internal_move_resize_window (childwin, 10, 10, 0, 0);
}

void
wimpi_draw_string (wimpiWindow sw, int x, int y, char *string,
		   int length)
{
    WINDOW *win;
    int fsizehigh, fsizewide, i, off, op = 0;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    y -= FONT_ASCENT; 

    fsizehigh = win->font->head.high;
    fsizewide = win->font->head.wide;
    
    wimpi_pre_blit (win, x, y, x + (fsizewide * length), y + fsizehigh);

    op = PUTFCOLOR (op, BLACK);
    op = PUTBCOLOR (op, WHITE);

    for (i = 0, off = x; i < length; i++) {
	BITMAP *src_map;

	src_map = win->font->glyph[(unsigned char)*string];
	if (BIT_CACHE(src_map) && (0xffff&op)>>4 != BIT_CHCLR(src_map)) {
	    bit_destroy(BIT_CACHE(src_map));
	    BIT_CACHE(src_map) = NULL;
	}

	if (!BIT_CACHE(src_map)) {
	    BIT_CACHE(src_map) = bit_expand(src_map->primary,BLACK, WHITE);
	    BIT_CHCLR(src_map) = op>>4 & 0xffff;
	}

	BIT_CACHE(src_map)->type |= _TRANSPARENT_BG;

        bit_blit(win->window, off, y, fsizewide, fsizehigh,
                 win->op, BIT_CACHE(src_map), 
		 ((unsigned char)*string) * fsizewide, 0);

        string++;
        off += fsizewide;
    }

    wimpi_post_blit (win, x, y, x + (fsizewide * length), y + fsizehigh);
}

void
wimpi_draw_line (wimpiWindow sw, int x0, int y0, int x1, int y1)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    wimpi_pre_blit (win, x0, y0, x1-x0, y1-y0);

    bit_line (win->window, x0, y0, x1, y1, win->op);

    wimpi_post_blit (win, x0, y0, x1-x0, y1-y0);
}

void
wimpi_draw_rectangle (wimpiWindow sw, int x, int y, int wide, 
		      int high)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    wimpi_pre_blit (win, x, y, wide, high);

    bit_line (win->window, x, y, x+wide, y, win->op);
    bit_line (win->window, x, y, x, y+high, win->op);
    bit_line (win->window, x, y+high, x+wide, y+high, win->op);
    bit_line (win->window, x+wide, y, x+wide, y+high, win->op);

    wimpi_post_blit (win, x, y, wide, high);
}

void
wimpi_draw_arc (wimpiWindow sw, int x, int y, int wide, int high,
		    int angle1, int angle2)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    wimpi_pre_blit (win, x, y, wide, high);

    /* MGR's old arc-drawing routine doesn't draw the right kind of arc.
     * Need to lift arc drawing from X.
     */

    wimpi_post_blit (win, x, y, wide, high);
}

void
wimpi_draw_ellipse (wimpiWindow sw, int x, int y, int wide, int high)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    wimpi_pre_blit (win, x, y, wide, high);

    /* This doesn't work! */

    ellipse (win->window, x + (wide>>1), y + (high>>1), wide>>1, high>>1,
	     win->op);

    wimpi_post_blit (win, x, y, wide, high);
}

void
wimpi_set_foreground (wimpiWindow sw, int color)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win)
	return;

    win->op = PUTFCOLOR(win->op,color);
    win->style = PUTFCOLOR(win->style,color);
}

void
wimpi_put_image (wimpiWindow sw, int x, int y, void * data,
		 int wide, int high, unsigned depth)
{
    wimpiSession mr = sw->w->session;
    WINDOW *win;
    BITMAP *image;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    image = bit_alloc (wide, high, data, depth);

    wimpi_pre_blit (win, x, y, wide, high);

    /* Not sure if the op created by BUILDOP is right... */
    bit_blit (win->window, x, y, wide, high, 
	      BUILDOP (BIT_SRC, mr->color_map[ROOT_COLOR_FG], 
		       mr->color_map[ROOT_COLOR_BG]),
	      image, 0, 0);
    
    wimpi_post_blit (win, x, y, wide, high);

    free (image);

    return;
}

void
wimpi_copy_area (wimpiWindow sw, int src_x, int src_y, int width, 
		int height, int dst_x, int dst_y)
{
    WINDOW *win;
    
    win = wimpi_get_window (sw->w, sw->winId);

    if (!win || !win->is_visible)
	return;

    wimpi_pre_blit (win, src_x, src_y, width, height);

    bit_blit(win->window, dst_x, dst_y, width, height,
	     win->op, win->window, src_x, src_y);
    
    wimpi_post_blit (win, src_x, src_y, width, height);
}

void
wimpi_set_window_title (wimpiWindow sw, char *title)
{
    WINDOW *win;

    win = wimpi_get_window (sw->w, sw->winId);

    if (!win)
	return;

    wimp_internal_set_window_title (win, title);
}

