/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * tcpdump.c
 * 
 * this is not a real tcpdump, but simply something that inserts itself
 * between the TCP/IP stack and the networking driver
 */
#include <oskit/dev/error.h>
#include <oskit/dev/ethernet.h>
#include <oskit/dev/freebsd.h>
#include <oskit/dev/net.h>
#include <oskit/net/freebsd.h>
#include <oskit/net/socket.h>
#include <oskit/c/fd.h>

#include <oskit/c/stdio.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/string.h>
#include <oskit/c/assert.h>

#include "startup/bootp.h"

#define VERBOSITY 0

/* this name shouldn't matter */
#define IFNAME		"de0"

static char ipaddr[20];
static char gateway[20];
static char netmask[20];
extern long secondswest;

/* XXX move in header file */
oskit_netio_t * oskit_netio_link_create(
	void (*dump_f)(void *cookie, void *p, oskit_size_t len),
	void *cookie,
	oskit_netio_t *forwardto);

/*
 * global variables related to networking
 */
extern char hostname[256]; 
static struct oskit_freebsd_net_ether_if *eif;

void
start_network_with_dump(void (*dump_f)(void *cookie, void *p, unsigned len))
{
	oskit_etherdev_t **etherdev;
	oskit_netio_t	*send_nio, *recv_nio;
	oskit_socket_factory_t *fsc;
	int   ndev, err;

        /*
         * Find all the Ethernet device nodes.
         */
        ndev = osenv_device_lookup(&oskit_etherdev_iid, (void***)&etherdev);
        if (ndev <= 0)
                panic("no Ethernet adaptors found!");

        /* get ip addr via bootp or console */
        get_ipinfo(etherdev[0], ipaddr, gateway, netmask, 
		hostname, sizeof hostname, &secondswest);
 
	err = oskit_freebsd_net_init(&fsc);
	assert(!err);
	oskit_register(&oskit_socket_factory_iid, (void *) fsc);

	/* this sets eif.recv_nio */
	err = oskit_freebsd_net_prepare_ether_if(&eif);
	assert(!err);

	/* we must set eif->haddr and eif->send_nio */
	oskit_etherdev_getaddr(etherdev[0], eif->haddr);

        /* Open this adaptor, exchanging net_io devices */
	recv_nio = oskit_netio_link_create(dump_f, 
		(void *)0 /* incoming */, eif->recv_nio);
	oskit_netio_release(eif->recv_nio);

        err = oskit_etherdev_open(etherdev[0], 0, recv_nio, &send_nio);
        if (err) {
                osenv_log(OSENV_LOG_ERR, "Error opening ethercard");
                return;
        }
	eif->send_nio = oskit_netio_link_create(dump_f, 
		(void *)1 /* outgoing */, send_nio);
	oskit_netio_release(send_nio);

#if VERBOSITY > 0
	printf("hostname = `%s'\n", hostname);
	printf("gateway = `%s'\n", gateway);
	printf("ipaddr = `%s'\n", ipaddr);
	printf("netmask = `%s'\n", netmask);
#endif

	err = oskit_freebsd_net_ifconfig(eif, IFNAME, ipaddr, netmask);
	assert(!err);

	if ((err = oskit_freebsd_net_add_default_route(gateway))) {
		printf("couldn't add default route to %s\n(error=%d; `%s')\n", 
			gateway, err, strerror(err));
	}
}
