/*
 * Copyright (c) 1997 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
#include <stdlib.h>

#include "support.h"

void *memalign(size_t alignment, size_t size);

/*
 * Alternate version of the standard malloc functions that expect the
 * caller to keep track of the size of allocated chunks.  These
 * versions are _much_ more memory-efficient when allocating many
 * chunks naturally aligned to their (natural) size (e.g. allocating
 * naturally-aligned pages or superpages), because normal memalign
 * requires a prefix between each chunk which will create horrendous
 * fragmentation and memory loss.  Chunks allocated with these
 * functions must be freed with sfree() rather than the ordinary
 * free().
 */
void *smalloc(size_t size)
{
	return malloc(size);
}

void *smemalign(size_t alignment, size_t size)
{
	return memalign(alignment, size);
}

void *scalloc(size_t size)
{
	return calloc(size, 1);
}

void *srealloc(void *buf, size_t old_size, size_t new_size)
{
	return realloc(buf, new_size);
}

void sfree(void *buf, size_t size)
{
	return free(buf);
}


/*
 * These functions are specific to the LMM implementation of malloc;
 * they additionally take an LMM flags word
 * so you can explicitly ask for specific types of memory.
 */

void *mallocf(size_t size, unsigned flags)
{
	void *p; 
	p = malloc(size);
	return p; 
}

void *memalignf(size_t alignment, size_t size, unsigned flags)
{
	void *p; 
	p = memalign(alignment, size);
	return p; 
}

void *smallocf(size_t size, unsigned flags)
{
	void *p; 
	p = malloc(size);
	return p; 
}

void *smemalignf(size_t alignment, size_t size, unsigned flags)
{
	void *p; 
	p = memalign(alignment, size);
	return p; 
}

/*
 * These are the lowest level functions, and require the memory lock
 * to be taken.
 */
void *calloc(size_t nelt, size_t eltsize)
{
	void *p;
	void *dl_calloc(size_t nelt, size_t eltsize);
	
	mem_lock();
	p = dl_calloc(nelt, eltsize);
	mem_unlock();
	return p;
}

void free(void *buf)
{
	void dl_free(void *buf);

	mem_lock(buf);
	dl_free(buf);
	mem_unlock();
}

void *malloc(size_t size)
{
	void *p;
	void *dl_malloc(size_t size);
	
	mem_lock();
	p = dl_malloc(size);
	mem_unlock();
	return p;
}

void *memalign(size_t alignment, size_t size)
{
	void *p;
	void *dl_memalign(size_t alignment, size_t size);
	
	mem_lock();
	p = dl_memalign(alignment, size);
	mem_unlock();
	return p;
}

void *realloc(void *buf, size_t new_size)
{
	void *p;
	void *dl_realloc(void *buf, size_t new_size);
	
	mem_lock();
	p = dl_realloc(buf, new_size);
	mem_unlock();
	return p;
}

void *valloc(size_t size)
{
	void *p;
	void *dl_valloc(size_t size);
	
	mem_lock();
	p = dl_valloc(size);
	mem_unlock();
	return p;
}

void *pvalloc(size_t size)
{
	void *p;
	void *dl_pvalloc(size_t size);
		
	mem_lock();
	p = dl_pvalloc(size);
	mem_unlock();
	return p;
}

