/*
 * populate the bmod file system with a sub tree of the real file system
 * Usage: 
 *
 * 	oskit_dir_t *
 * 	oskit_unix_bmod_init();
 *
 * will create a bmod filesystem and return a pointer to the root. This
 * should be passed to fs_init.
 *
 *	int
 *	oskit_unix_bmod_copydir(oskit_dir_t *rootdir, const char *path)
 *
 * will create all the directories in the path, and then copy the directory
 * tree at the last component into the bmod fs at the equiv location.
 *
 * Of course, writes will end up in the bmodfs and hence be transient.
 */
#include <oskit/fs/file.h>
#include <oskit/fs/openfile.h>
#include <oskit/fs/dir.h>
#include <oskit/fs/bmodfs.h>
#include <oskit/fs/soa.h>
#include <oskit/c/sys/types.h>
#include <oskit/c/assert.h>

#define NEEDBSDTYPES
#include "native.h"

/*
 * There is no way we get the BSD includes right; so just do things
 * by hand.  Pray for dev_t, ino_t, etc.  
 * The size of native_stat should be 96.
 */

/*
 * Structure defined by POSIX.4 to be like a timeval.
 */
struct timespec {
        time_t  tv_sec;         /* seconds */
        long    tv_nsec;        /* and nanoseconds */
};

extern int errno;

#define S_ISDIR(m)      (((m) & 0170000) == 0040000)    /* directory */

struct native_dirent {
        u_int32_t d_fileno;             /* file number of entry */
        u_int16_t d_reclen;             /* length of this record */
        u_int8_t  d_type;               /* file type, see below */
        u_int8_t  d_namlen;             /* length of string in d_name */
#ifdef _POSIX_SOURCE
        char    d_name[255 + 1];        /* name must be no longer than this */
#else
#define MAXNAMLEN       255
        char    d_name[MAXNAMLEN + 1];  /* name must be no longer than this */
#endif
};

struct native_stat {
        dev_t     st_dev;               /* inode's device */ 
        ino_t     st_ino;               /* inode's number */
        mode_t    st_mode;              /* inode protection mode */
        nlink_t   st_nlink;             /* number of hard links */
        uid_t     st_uid;               /* user ID of the file's owner */
        gid_t     st_gid;               /* group ID of the file's group */
        dev_t     st_rdev;              /* device type */
#ifndef _POSIX_SOURCE 
        struct  timespec st_atimespec;  /* time of last access */  
        struct  timespec st_mtimespec;  /* time of last data modification */
        struct  timespec st_ctimespec;  /* time of last file status change */
#else 
        time_t    st_atime;             /* time of last access */
        long      st_atimensec;         /* nsec of last access */
        time_t    st_mtime;             /* time of last data modification */
        long      st_mtimensec;         /* nsec of last data modification */
        time_t    st_ctime;             /* time of last file status change */
        long      st_ctimensec;         /* nsec of last file status change */
#endif
        off_t     st_size;              /* file size, in bytes */
        int64_t   st_blocks;            /* blocks allocated for file */
        u_int32_t st_blksize;           /* optimal blocksize for I/O */ 
        u_int32_t st_flags;             /* user defined flags for file */
        u_int32_t st_gen;               /* file generation number */ 
        int32_t   st_lspare; 
        int64_t   st_qspare[2];
}; 

static int		verbose;
static void		copy_dir(int, oskit_dir_t *);
static oskit_dir_t     *make_dir(oskit_dir_t *dir, const char *path);

oskit_dir_t *
oskit_unix_bmod_init()
{
	return oskit_bmod_init();
}

int
oskit_unix_bmod_copydir(oskit_dir_t *rootdir, const char *path)
{
	oskit_dir_t	*dir;
	int		fd;

	printf("Creating %s\n", path);

	if (path[0] == '.' && path[1] == 0)
		dir = rootdir;
	else 
		dir = make_dir(rootdir, path);

        fd = NATIVEOS(open)(path, 0 /* don't use O_RDONLY! */);
	
	if (fd == -1) {
		printf("Couldn't open %s, errno = %d\n", path, errno);
	} else {
		if (NATIVEOS(fchdir)(fd) == 0) {
			copy_dir(fd, dir);
			NATIVEOS(close)(fd);
			/* XXX fchdir back to where we were */
		} else {
			printf("couldn't fchdir to %s/errno=%d\n", path,errno);
		}
	}
	return 0;
}

static oskit_dir_t *
make_dir(oskit_dir_t *dir, const char *path)
{
	char		buf[128], *dp = buf, *bp = path;
	oskit_dir_t	*ddir, *ndir;
	oskit_file_t	*dfile;

	while (*bp && *bp != '/')
		*dp++ = *bp++;

	*dp = '\0';

	oskit_dir_mkdir(dir, buf, 666);
	oskit_dir_lookup(dir, buf, &dfile);
	oskit_file_query(dfile, &oskit_dir_iid, (void**)&ddir);
	oskit_file_release(dfile);

	if (*bp == '/') {
		bp++;
		ndir = make_dir(ddir, bp);
		oskit_dir_release(ddir);
		return ndir;
	}
	return ddir;
}
           
static void       
copy_dir(int dirfd, oskit_dir_t *dst)
{          
	static char absname[4096] = "/";
	char buf[1024];
	long base = 0;
	int r;

	while ((r = NATIVEOS(getdirentries)(dirfd, buf, sizeof buf, &base)) > 0)
	{
		struct native_dirent *d = (struct native_dirent *)buf;
		while ((void*)d < (void*)(buf + r)) {
			char *name = (char *)malloc(d->d_namlen + 1);
			struct native_stat stb;
			int rs;
			
			memcpy(name, d->d_name, d->d_namlen);
			name[d->d_namlen] = 0;
			
			if (!strcmp(name, ".") || !strcmp(name, ".."))
				goto next;

			rs = NATIVEOS(stat)(name, &stb);
			if (rs == -1) {
				printf("can't stat %s/errno = %d, skipping\n", 
					name, errno);
				goto next;
			}
				
			if (S_ISDIR(stb.st_mode)) 
			{
				oskit_dir_t *ddir;
				oskit_file_t *dfile;
				int oldlen = strlen(absname);
				int fd = NATIVEOS(open)(name, 
					0 /* don't use O_RDONLY! */);
				strcat(absname, name);
				strcat(absname, "/");
				NATIVEOS(fchdir)(fd);
				oskit_dir_mkdir(dst, name, 666);
				oskit_dir_lookup(dst, name, &dfile);
				oskit_file_query(dfile, &oskit_dir_iid, 
					(void**)&ddir);
				oskit_file_release(dfile);
				copy_dir(fd, ddir); 
				oskit_dir_release(ddir);
				NATIVEOS(close)(fd);
				NATIVEOS(fchdir)(dirfd);
				absname[oldlen] = 0;
			} else {
				static char fb[65536];
				oskit_error_t rc;
				oskit_file_t *file;
				oskit_openfile_t *ofile;
				oskit_u32_t actual;
				int b, fd = NATIVEOS(open)(name, 
					0 /* don't use O_RDONLY!*/);
				if (fd == -1) {
					printf("couldn't open %s/errno=%d\n",
						name, errno);
					goto next;
				}
				if (verbose)
					printf("copying %s%s ...\n", 
						absname, name);

				rc = oskit_dir_create(dst, name, 0, 666, &file);
				assert(rc == 0);
				/* it's the bmod fs, oskit_file_open would
				 * return rc == 0 & ofile == 0.  Must use 
				 * adaptor as in liboskit_c::open().
				 */
				rc = oskit_soa_openfile_from_file(file,
					OSKIT_O_WRONLY, &ofile);
				assert(rc == 0 && ofile);

				while ((b = NATIVEOS(read)
						(fd, fb, sizeof fb)) > 0) 
				{
					rc = oskit_openfile_write(ofile, 
						fb, b, &actual);
					assert(rc == 0 && b == actual);
				}
				NATIVEOS(close)(fd);
				oskit_openfile_release(ofile);
				oskit_file_release(file);
			}
		next:
			d = ((void *)d) + d->d_reclen;
			free(name);
		}
	} 
}
