/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Handle the details of stack guards. For the x86 we must use a task
 * gate to handle stack overflow since faults dump state onto the current
 * stack when there is no priv level change (as is the case in the base
 * oskit). That of course will lead to a double fault, and then a machine
 * reset without any idea of what happened. By installing a task gate for
 * for T_DOUBLE, the double fault will cause a switch into a different task
 * with a new stack. At that point we can figure out what thread caused the
 * stack overflow, and then shutdown gracefully. It would be nice if we
 * could recover and just kill that thread, but a double fault is deemed
 * non-recoverable, and appears to leave the machine in an odd state.
 */

#include <threads/pthread_internal.h>

#ifdef	STACKGUARD
#include <oskit/c/sys/mman.h>
#include <oskit/io/absio.h>
#include <oskit/svm.h>

/*
 * Init the VM code.
 */
void
pthread_init_guard(void)
{
	svm_init((oskit_absio_t *) 0);
}

/*
 * Set the memory protection for a region. Used for stack guards.
 */
int
pthread_mprotect(oskit_addr_t addr, size_t length, int prot)
{
	int		mapping_bits, rcode;

	if (addr & (PAGE_SIZE - 1)) {
		printf("pthread_mprotect: addr 0x%x not aligned\n", addr);
		return EINVAL;
	}

	if (length & (PAGE_SIZE - 1)) {
		printf("pthread_mprotect: length %d not aligned\n", length);
		return EINVAL;
	}

	switch (prot) {
	case PROT_READ|PROT_WRITE:
		mapping_bits = SVM_PROT_READ|SVM_PROT_WRITE;
		break;
	case PROT_READ:
		mapping_bits = SVM_PROT_READ;
		break;
	default:
		panic("pthread_mprotect: Bad prot: 0x%x", prot);
		break;
	}

	rcode = svm_protect(addr, length, mapping_bits);

	return rcode;
}
#endif
