/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <threads/pthread_internal.h>

/*
 * Timed sleep.
 * 
 * The calling thread goes to sleep for the given amount of time, in
 * milliseconds. This is a primitive sleep, not implemented in terms
 * of pthread_cond_timedwait, since that takes an absolute time.
 */
int
pthread_sleep(oskit_s64_t timeout)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			rval, s;
	oskit_itimerspec_t	timespec = {{ 0, 0 }, { 0, 0 }};

	if (pthread == IDLETHREAD)
		panic("pthread_sleep: Idle thread tried to sleep");

	/*
	 * Must take the thread lock at splhigh() since the timeout
	 * is delivered at interrupt level.
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);

	/*
	 * Zero timeout means sleep until woken.
	 */
	if (timeout) {
		timespec.it_value.tv_sec  += timeout / 1000;
		timespec.it_value.tv_nsec += (timeout % 1000) * 1000000;
		oskit_timer_settime(pthread->sleeptimer, 0, &timespec);
	}

	pthread->flags &= ~THREAD_TIMEDOUT;
	pthread->flags |=  THREAD_SLEEPING;
	
#ifdef  THREADS_DEBUG
	pthread_lock(&threads_sleepers_lock);
	threads_sleepers++;
	pthread_unlock(&threads_sleepers_lock);
#endif
	pthread_sched_reschedule(RESCHED_BLOCK, &pthread->lock);

#ifdef  THREADS_DEBUG
	pthread_lock(&threads_sleepers_lock);
	threads_sleepers--;
	pthread_unlock(&threads_sleepers_lock);
#endif
	/*
	 * Check for timeout.
	 */
	pthread_lock(&pthread->lock);

	if (pthread->flags & THREAD_TIMEDOUT) {
		pthread->flags &= ~THREAD_TIMEDOUT;
		rval = ETIMEDOUT;
	}
	else
		rval = 0;

	pthread_unlock(&pthread->lock);
	splx(s);

	/* Look for a cancelation point, but only if not in a driver sleep */
	if (!pthread->sleeprec)
		pthread_testcancel();

	return rval;
}

/*
 * Wake up a thread that put itself to sleep.
 */
int
pthread_wakeup(pthread_t tid)
{
	pthread_thread_t	*pthread;
	int			s;

	if (tid < 0 || tid >= THREADS_MAX_THREAD ||
	    ((pthread = tidtothread(tid)) == NULL_THREADPTR))
		return EINVAL;

	if (CURPTHREAD()->tid == tid)
		return EINVAL;

	/*
	 * Must take the thread lock at splhigh() since the timeout
	 * is delivered at interrupt level.
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);

	if (pthread->flags & THREAD_SLEEPING)
		pthread_wakeup_locked(pthread);
	else
		pthread_unlock(&pthread->lock);
	
	splx(s);

	check_yield();

	return 0;
}

/*
 * Internal version. Thread lock is held. Interupts are disabled.
 */
int
pthread_wakeup_locked(pthread_thread_t *pthread)
{
	/* disarm the timer. */
	if (pthread->flags & THREAD_TIMERSET) {
		oskit_itimerspec_t	timespec = {{ 0, 0 }, { 0, 0 }};
		
		timespec.it_value.tv_sec  = 0;
		timespec.it_value.tv_nsec = 0;
		oskit_timer_settime(pthread->sleeptimer, 0, &timespec);
		pthread->flags &= ~THREAD_TIMERSET;		
	}
		
	pthread->flags &= ~THREAD_SLEEPING;
	pthread_unlock(&pthread->lock);
	
	return pthread_sched_setrunnable(pthread);
}

/*
 * Internal version. Must take the thread lock.
 */
int
pthread_wakeup_unlocked(pthread_thread_t *pthread)
{
	int			resched;
	int			s;

	/*
	 * Must take the thread lock at splhigh() since the timeout
	 * is delivered at interrupt level.
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);
	
	/* disarm the timer. */
	if (pthread->flags & THREAD_TIMERSET) {
		oskit_itimerspec_t timespec = {{ 0, 0 }, { 0, 0 }};
		
		timespec.it_value.tv_sec  = 0;
		timespec.it_value.tv_nsec = 0;
		oskit_timer_settime(pthread->sleeptimer, 0, &timespec);
		pthread->flags &= ~THREAD_TIMERSET;		
	}
		
	pthread->flags &= ~THREAD_SLEEPING;
	pthread_unlock(&pthread->lock);

	resched = pthread_sched_setrunnable(pthread);

	splx(s);
	return resched;
}

/*      
 * Timer expiration.
 */
oskit_error_t
pthread_sleep_timeout(struct oskit_iunknown *listener, void *arg)
{
	pthread_thread_t	*pthread = arg;
	int			s;

	/*
	 * Note that pthread lock is taken in an interrupt handler!
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);

	if (threads_debug)
		printf("pthread_sleep_timeout: 0x%x 0x%x 0x%x 0x%x\n",
		       (int) CURPTHREAD(), s,
		       (int) pthread, (int) pthread->flags);

	/*
	 * Already woken up?
	 */
	if (! (pthread->flags & THREAD_SLEEPING)) {
		pthread_unlock(&pthread->lock);
		splx(s);
		return 0;
	}

	/*
	 * Nope, place it back on the runq. Request an AST if the current
	 * thread is no longer the thread that should be running.
	 */
	pthread->flags |=  THREAD_TIMEDOUT;
	pthread->flags &= ~(THREAD_SLEEPING|THREAD_TIMERSET);
	pthread_unlock(&pthread->lock);

	if (pthread_sched_setrunnable(pthread))
		softint_request(SOFTINT_ASYNCREQ);

	if (threads_debug)
		printf("pthread_sleep_timeout ends\n");

	splx(s);
	return 0;
}
