/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#ifdef	DEFAULT_SCHEDULER
/*
 * Simple scheduler. This scheduler implements everything needed for
 * the SCHED_FIFO and SCHED_RR policies.
 */
#include <threads/pthread_internal.h>

/*
 * The RunQ is a multilevel queue of doubly linked lists. Use a bitmask
 * to indicate whichrunq is non-empty, with the least significant bit
 * being the highest priority (cause off ffs).
 */
#define MAXPRI		(PRIORITY_MAX + 1)

/*
 * These are internal to this file.
 */
static queue_head_t	threads_runq[MAXPRI] = { {0} };
static int		threads_runq_count   = 0;
static oskit_u32_t	threads_whichrunq;

/*
 * The scheduler lock is used elsewhere.
 */
pthread_lock_t		pthread_sched_lock    = PTHREAD_LOCK_INITIALIZER;

extern int		ffs();

#ifdef	MEASURE
#include <oskit/machine/proc_reg.h>

struct pthread_stats {
	int			wakeups;
	unsigned long		wakeup_cycles;
	int			switches;
	unsigned long		switch_cycles;
};
static struct pthread_stats stats;
void	dump_scheduler_stats();
#endif

/*
 * Initialize the runqs to empty.
 */
void
pthread_init_scheduler(void)
{
	int	i;
		
	for (i = 0; i < MAXPRI; i++)
		queue_init(&threads_runq[i]);
#ifdef MEASURE
	atexit(dump_scheduler_stats);
#endif
}

/*
 * Are there any threads on the runq?
 */
inline int
pthread_runq_empty(void)
{
	return (threads_whichrunq == 0);
}

/*
 * Get the highest priority scheduled thread.
 */
inline int
pthread_runq_maxprio(void)
{
	int	prio;
	
	if (pthread_runq_empty())
		return -1;
	else {
		prio = ffs(threads_whichrunq);
		
		return PRIORITY_MAX - (prio - 1);
	}
}

/*
 * Determine if a pthread is on the runq. Use a separate field 
 * since using the flags would require locking the thread. Use the
 * queue chain pointer instead, setting it to zero when a thread is
 * removed from the queue.
 */
inline int
pthread_runq_onrunq(pthread_thread_t *pthread)
{
	return (int) pthread->runq.next;
}

/*
 * Add and remove threads from the runq. The runq lock should be locked,
 * and interrupts disabled.
 */

/*
 * Insert at the tail of the runq.
 */
inline void
pthread_runq_insert_tail(pthread_thread_t *pthread)
{
	int		prio  = PRIORITY_MAX - pthread->priority;
	queue_head_t	*phdr = &threads_runq[prio];

	queue_enter(phdr, pthread, pthread_thread_t *, runq);

	threads_whichrunq |= (1 << prio);
	threads_runq_count++;
}

/*
 * Insert at the head of the runq.
 */
inline void
pthread_runq_insert_head(pthread_thread_t *pthread)
{
	int		prio  = PRIORITY_MAX - pthread->priority;
	queue_head_t	*phdr = &threads_runq[prio];

	queue_enter_first(phdr, pthread, pthread_thread_t *, runq);

	threads_whichrunq |= (1 << prio);
	threads_runq_count++;
}

/*
 * Dequeue highest priority pthread.
 */
OSKIT_INLINE pthread_thread_t *
pthread_runq_dequeue(void)
{
	int			prio  = ffs(threads_whichrunq) - 1;
	queue_head_t		*phdr = &threads_runq[prio];
	pthread_thread_t	*pnext;

	queue_remove_first(phdr, pnext, pthread_thread_t *, runq);
	pnext->runq.next = (queue_entry_t) 0;	

	threads_runq_count--;
	if (queue_empty(phdr))
		threads_whichrunq &= ~(1 << prio);

	return pnext;
}

/*
 * Remove an arbitrary thread from the runq.
 */
inline void
pthread_runq_remove(pthread_thread_t *pthread)
{
	int		prio  = PRIORITY_MAX - pthread->priority;
	queue_head_t	*phdr = &threads_runq[prio];

	queue_remove(phdr, pthread, pthread_thread_t *, runq);
	pthread->runq.next = (queue_entry_t) 0;	

	threads_runq_count--;
	if (queue_empty(phdr))
		threads_whichrunq &= ~(1 << prio);
}

/*
 * This handles the details of finding another thread to switch to.
 * Return value to indicate whether a switch actually happened.
 */
int
pthread_sched_dispatch(resched_flags_t reason, pthread_thread_t *pnext)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s;
#ifdef  MEASURE
	unsigned long		before, after;
#endif	
	s = splhigh();

#ifdef  MEASURE
	before = get_tsc();
#endif
	/*
	 * Going to muck with the scheduler queues, so take that lock.
	 */
	pthread_lock(&pthread_sched_lock);

	/*
	 * Decide what to do with the current thread.
	 */
	switch (reason) {
	case RESCHED_USERYIELD:
		if (pthread == IDLETHREAD)
			panic("pthread_sched_reschedule: Idlethread!\n");

		/*
		 * A user directed yield forces the thread to the back
		 * of the queue.
		 */
		pthread_runq_insert_tail(pthread);
		break;
		
	case RESCHED_YIELD:
		if (pthread == IDLETHREAD)
			panic("pthread_sched_reschedule: Idlethread!\n");

		/*
		 * A involuntary yield forces the thread to the front 
		 * of the queue. It will probably be rerun right away!
		 */
		pthread_runq_insert_head(pthread);
		break;
		
	case RESCHED_PREEMPT:
		if (pthread == IDLETHREAD)
			panic("pthread_sched_reschedule: Idlethread!\n");

		/*
		 * Time based preemption. If the policy is RR, then the
		 * thread goes to the back of the queue if it has used
		 * all of its ticks. Otherwise it stays at the head, of
		 * course.
		 *
		 * If the policy is FIFO, this has no real effect, other
		 * than to possibly allow a higher priority thread to get
		 * the CPU.
		 */
		if (pthread->policy == SCHED_RR) {
			if (--pthread->ticks == 0) {
				pthread_runq_insert_tail(pthread);
				pthread->ticks = SCHED_RR_INTERVAL;
			}
			else
				pthread_runq_insert_head(pthread);
		}
		else if (pthread->policy == SCHED_FIFO)
			pthread_runq_insert_head(pthread);
		break;

	case RESCHED_INTERNAL:
		/*
		 * This will be the idle thread.
		 */
		break;
		
	default:
		/*
		 * All other rescheduling modes are blocks, and thus ignored.
		 */
		if (pthread == IDLETHREAD)
			panic("pthread_sched_reschedule: Idlethread!\n");
		break;
	}

	/*
	 * Clear preemption flag
	 */
	pthread->preempt = 0;

	/*
	 * Now find a thread to run.
	 */
	if (pnext) {
		/*
		 * Locked thread was provided, so done with the scheduler.
		 */
		pthread_unlock(&pthread_sched_lock);
	}
	else {
		if (pthread_runq_empty()) {
#ifdef  THREADS_DEBUG0
			pthread_lock(&threads_sleepers_lock);
			if (! threads_sleepers)
				panic("RESCHED: "
				      "Nothing to run. Might be deadlock!");
			pthread_unlock(&threads_sleepers_lock);
#endif
			pnext = IDLETHREAD;
		}
		else
			pnext = pthread_runq_dequeue();
		
		pthread_unlock(&pthread_sched_lock);

		/*
		 * Avoid switch into same thread. 
		 */
		if (pnext == pthread) {
			pthread_unlock(&pthread->lock);
			splx(s);
			return 0;
		}
		
		pthread_lock(&pnext->lock);
	}

#ifdef  MEASURE
	after = get_tsc();
	if (after > before) {
		stats.switches++;
		stats.switch_cycles += (after - before);
	}
#endif
	/*
	 * Switch to next thread. The thread lock for the current thread
	 * is released in the context switch code, while the lock for
	 * next thread is carried across the context switch and is released
	 * once the switch is committed.
	 */
	thread_switch(pnext, &pthread->lock, &CURPTHREAD());

	/*
	 * Thread has switched back in. Thread lock is locked.
	 *
	 * Look for exceptions before letting it return to whatever it
	 * was doing before it yielded.
	 */
	if (pthread->sleeprec) {
		/*
		 * Thread is returning to an osenv_sleep. The thread must
		 * be allowed to return, even if it was killed, so the driver
		 * state can be cleaned up. The death will be noticed later.
		 */
		;
	}
	else if ((pthread->flags &
		  (THREAD_KILLED|THREAD_EXITING)) == THREAD_KILLED) {
		/*
		 * Thread was canceled. Time to actually reap it, but only
		 * if its not already in the process of exiting.
		 *
		 * XXX: The problem is if the process lock is still
		 * held.  Since the rest of the oskit is not set up to
		 * handle cancelation of I/O operations, let threads
		 * continue to run until the process lock is released.
		 * At that point the death will be noticed.
		 */
		if (! osenv_process_locked()) {
			pthread_exit_locked((void *) PTHREAD_CANCELED);

			/*
			 * Never returns
			 */
		}
	}

	pthread_unlock(&pthread->lock);
	splx(s);
	return 1;
}

/*
 * Here starts the external interface functions.
 */

/*
 * Generic switch code. Find a new thread to run and switch to it.
 */
int
pthread_sched_reschedule(resched_flags_t reason, pthread_lock_t *plock)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s, rc;

	s = splhigh();

	/*
	 * Need to lock the thread lock and then release the provided
	 * spinlock. This provides atomicity with regards to the thread
	 * that is switching out since in all cases the lock is either
	 * the thread lock, or a queue that it just put itself on.
	 */
	if (plock != &(pthread->lock)) {
		pthread_lock(&(pthread->lock));
		
		if (plock)
			pthread_unlock(plock);
	}

	rc = pthread_sched_dispatch(reason, 0);

	splx(s);
	return rc;
}

/*
 * Directed switch. The current thread is blocked with the waitstate.
 */
void
pthread_sched_handoff(int waitstate, pthread_thread_t *pnext)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s;

	s = splhigh();
	pthread_lock(&pnext->lock);
	pthread_lock(&pthread->lock);
	
	if (waitstate) {
		pthread->flags |= waitstate;
		pthread_sched_dispatch(RESCHED_BLOCK, pnext);
	}
	else
		pthread_sched_dispatch(RESCHED_YIELD, pnext);
	
	splx(s);
}

/*
 * Wakeup a thread. This does not schedule the thread, but only terminates
 * the wait state. If the thread was canceled, tell the caller.
 */
int
pthread_sched_thread_wakeup(int waitstate, pthread_thread_t *pthread)
{
	int		s, okay = 1;

	s = splhigh();
	pthread_lock(&pthread->lock);
	pthread->flags &= ~waitstate;

	if (pthread->flags & THREAD_CANCELED) {
		pthread_unlock(&pthread->lock);
		pthread_sched_setrunnable(pthread);
		okay = 0;
	}
	else
		pthread_unlock(&pthread->lock);
	
	splx(s);
	return okay;
}

/*
 * Place a thread that was blocked for some reason, back on the runq.
 * Both the thread lock and the scheduler lock is taken. That is not
 * strictly necessary since there is only a single scheduler, even in
 * SMP mode, but it will indicate where deadlock might happen.
 *
 * Return a boolean value indicating whether the current thread is still
 * the thread that should be running. In this case, it has to still be
 * the highest priority thread. The caller will need to take appropriate
 * action, if any.
 */
int 
pthread_sched_setrunnable(pthread_thread_t *pthread)
{
	int		s;
#ifdef  MEASURE
	unsigned long	before, after;
#endif	
	s = splhigh();
#ifdef  MEASURE
	before = get_tsc();
#endif
	pthread_lock(&pthread->lock);
	pthread_lock(&pthread_sched_lock);
	
	if (pthread_runq_onrunq(pthread))
		panic("pthread_sched_setrunnable: Already on runQ: 0x%x(%d)",
		      (int) pthread, pthread->tid);

	pthread_runq_insert_tail(pthread);

	if (CURPTHREAD()->priority < pthread_runq_maxprio())
		CURPTHREAD()->preempt = 1;

	pthread_unlock(&pthread->lock);
	pthread_unlock(&pthread_sched_lock);
#ifdef  MEASURE
	after = get_tsc();
	if (after > before) {
		stats.wakeups++;
		stats.wakeup_cycles += (after - before);
	}
#endif
	splx(s);

	return CURPTHREAD()->preempt;
}

/*
 * Change the state of a thread. In this case, its the priority and
 * policy that are possibly being changed. Return an indicator to the
 * caller, if the change requires a reschedule.
 *
 * Currently, the priority inheritance code is not SMP safe. Also, it is
 * specific to this scheduler module.  */
int
pthread_sched_change_state(pthread_thread_t *pthread, int newprio, int policy)
{
	int	s;

	s = splhigh();
	pthread_lock(&pthread->lock);

	switch (policy) {
	case -1:
		/* No change specified */
		break;
		
	case SCHED_FIFO:
	case SCHED_RR:
		pthread->policy = policy;
		break;

	case SCHED_DECAY:
		panic("pthread_setschedparam: SCHED_DECAY not supported yet");
		break;

	default:
		panic("pthread_sched_change_state: Bad policy specified");
	}

	if (pthread->base_priority == newprio)
	    goto done;

	pthread->base_priority = newprio;

#ifdef  PRI_INHERIT
	if (pthread->base_priority < pthread->priority) {
		if (! pthread->inherits_from)
			pthread_priority_decreasing_recompute(pthread);
	}
	else {
		pthread->priority = newprio;
		if (pthread->waiting_for)
			pthread_priority_increasing_recompute(pthread);

		if (pthread->inherits_from &&
		    newprio > pthread->inherits_from->priority) {
			pthread->inherits_from = NULL_THREADPTR;
		}
	}
	pthread_lock(&pthread_sched_lock);
#else
	pthread_lock(&pthread_sched_lock);
	if (pthread_runq_onrunq(pthread)) {
		pthread_runq_remove(pthread);
		pthread->priority = newprio;
		pthread_runq_insert_tail(pthread);
	}
	else {
		/*
		 * If its not on the runq, the thread priority can
		 * just be changed since there are no current
		 * dependencies on it. If however, the current thread
		 * has its priority changed, a reschedule might be
		 * necessary.
		 */
		pthread->priority = newprio;
	}
#endif	
	if (CURPTHREAD()->priority < pthread_runq_maxprio())
		CURPTHREAD()->preempt = 1;
	
	pthread_unlock(&pthread_sched_lock);
   done:
	pthread_unlock(&pthread->lock);
	splx(s);

	return CURPTHREAD()->preempt;
}

#ifdef	MEASURE
void
dump_scheduler_stats()
{
	printf("Wakeups:	%d\n", stats.wakeups);
	printf("Wakeup Cycles   %u\n", stats.wakeup_cycles);
	printf("Switches:	%d\n", stats.switches);
	printf("Switch Cycles   %u\n", stats.switch_cycles);
}
#endif
#endif /* DEFAULT_SCHEDULER */
