/*
 * Copyright (c) 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <threads/pthread_internal.h>

/*
 * Unlock a mutex.
 */
int
pthread_mutex_unlock(pthread_mutex_t *m)
{
	pthread_thread_t	*pnext = NULL_THREADPTR;
	int			p;
	
	p = splpreempt();
	pthread_lock(&m->mlock);

	if (m->holder != CURPTHREAD()) {
		/*
		 * This panic should be under mutex type control ...
		 */
		pthread_mutex_panic(m, "unlock", "wrong owner");
	}

	/*
	 * Decrement the count and look for a possible recursive lock.
	 */
	if (--m->count != 0) {
		if (m->count < 0)
			pthread_mutex_panic(m, "unlock", "negative count");
		
		if (m->type == PTHREAD_MUTEX_RECURSIVE) {
			pthread_unlock(&m->mlock);
			splx(p);
			return 0;
		}

		pthread_mutex_panic(m, "unlock", "non-zero count");
	}

	/*
	 * If the queue is empty, the mutex is free, and the lock can be
	 * released. If the queue is non-empty, remove one and transfer
	 * ownership.
	 */
	if (queue_empty(&(m->waiters))) {
		m->holder = NULL_THREADPTR;
		m->count  = 0;
		pthread_unlock(&m->mlock);
		splx(p);
		return 0;
	}

	/*
	 * Grant to next thread directly.
	 */
	m->holder = pnext = pthread_dequeue_fromQ(&(m->waiters));
	m->count  = 1;
	pthread_sched_setrunnable(pnext);
#ifdef  PRI_INHERIT
	if (m->inherit)
		pthread_priority_uninherit(pnext);
#endif
#ifdef  CPU_INHERIT
	/*
	 * Wake up all of the other threads to undo donation.
	 */
	if (m->inherit)
		pthread_resume_allQ(&(m->waiters));
#endif
	pthread_unlock(&(m->mlock));
	splx(p);
	
	/*
	 * Check for reschedule since the runq might have been changed.
	 */
	check_yield();

        return 0;
}
