/*
 * Copyright (c) 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <threads/pthread_internal.h>

/*
 * Try to lock a mutex. Non-blocking variant.
 */
int
pthread_mutex_trylock(pthread_mutex_t *m)
{
	int	p;
	
	p = splpreempt();
	pthread_lock(&m->mlock);

	/*
	 * Try to get the mutex. If successful, set owner and count.
	 */
	if (m->holder == 0) {
		m->holder = (void *) CURPTHREAD();		
		m->count  = 1;
		pthread_unlock(&m->mlock);
		splx(p);
		return 0;
	}
	pthread_unlock(&m->mlock);
	splx(p);
	return EBUSY;
}

/*
 * Try to lock a mutex. Blocking variant.
 */
int
pthread_mutex_lock(pthread_mutex_t *m)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			p;
	
	p = splpreempt();
	pthread_lock(&m->mlock);

	/*
	 * Try to get the lock. If successful, set owner and count.
	 */
	if (m->holder == 0) {
		m->holder = (void *) CURPTHREAD();		
		m->count  = 1;
		pthread_unlock(&m->mlock);
		splx(p);
		return 0;
	}

	/*
	 * Can't get it. Look for a recursive lock. 
	 */
	if (m->holder == CURPTHREAD()) {
		if (m->type == PTHREAD_MUTEX_RECURSIVE) {
			m->count++;
			pthread_unlock(&m->mlock);
			splx(p);
			return 0;
		}
		else
			pthread_mutex_panic(m, "lock", "recursive attempt");
	}

#ifdef  CPU_INHERIT
  again:
#endif
	/*
	 * Okay, time to block ...
	 */
	queue_check(&(m->waiters), pthread);
	
#ifdef	PRI_INHERIT
	if (m->inherit)
		pthread_priority_inherit(m->holder);
#endif
#ifdef  CPU_INHERIT
	if (m->inherit) {
		pthread_sched_thread_wait(m->holder,
					  &(m->waiters), &(m->mlock));

		/*
		 * All threads are woken up to undo the donation.
		 */
		if (m->holder != CURPTHREAD()) {
			/*printf("pthread_mutex_lock: "
			       "0x%x 0x%x(%d) 0x%x(%d)\n",
			       (int) m, (int) m->holder,
			       (int) ((pthread_thread_t *)m->holder)->tid,
			       (int) CURPTHREAD(), CURPTHREAD()->tid); */
			goto again;
		}
	}
	else
#endif
	{
	queue_enter(&(m->waiters), pthread, pthread_thread_t *, chain);
	pthread_sched_reschedule(RESCHED_BLOCK, &(m->mlock));
	}

	/*
	 * Unlock assures that this thread wakes up owning the mutex.
	 */
	if (m->holder != CURPTHREAD())
		pthread_mutex_panic(m, "lock", "Not the owner");
	
	splx(p);
        return 0;
}

void
pthread_mutex_panic(pthread_mutex_t *m, char *type, char *msg)
{
	pthread_thread_t	*pholder = m->holder;
	
	printf("pthread_mutex_%s: %s - 0x%x\n", type, msg, (int) m);

	if (pholder && pholder != CURPTHREAD()) {
		printf("Backtrace for owner 0x%x(%d)\n",
		       (int) pholder, pholder->tid);
		threads_stack_back_trace(pholder->tid, 16);
	}
	panic("pthread_mutex_panic");
}

