/*
 * Copyright (c) 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Define macros for low-level locking (spin locks, test and set, etc.)
 *
 * Eventually we will have three four execution models:
 *
 * 1. Nonpreemptive uniprocessor (PTHREAD_MODEL_NPUNI - default)
 * 2. Nonpreemptive multiprocessor (PTHREAD_MODEL_NPMULTI)
 * 3. Preemptive uniprocessor (PTHREAD_MODEL_PUNI)
 * 4. Preemptive multiprocessor (PTHREAD_MODEL_PMULTI)
 *
 * Note: only 1 and 2 are supported right now.
 *
 * Locking requirements:
 *
 * 1. Nonpreemptive uniprocessor:  No locking is required.
 *
 * 2. Nonpreemptive multiprocessor: Spin locks or equivalent are
 * required to provide mutual exclusion to internal data structures
 * among multiple processors.
 *
 * 3. Preemptive uniprocessor:  Spin locks or interrupt disabling.
 *
 * 4. Preemptive multiprocessor:  Spin locks or equivalent (see 2).
 * */

/* right now, simply use spin locks for all models */

#ifndef _OSKIT_PTHREADS_LOCKING_H_
#define _OSKIT_PTHREADS_LOCKING_H_

#if defined(SMP) || defined(THREADS_DEBUG)
#define PTHREAD_LOCK_INITIALIZER SPIN_LOCK_INITIALIZER
#define pthread_lock_t spin_lock_t
#define pthread_lock_init(lock) spin_lock_init((spin_lock_t *) lock)
#define pthread_try_lock(lock) spin_try_lock((spin_lock_t *) lock)
#define pthread_lock_locked(lock) spin_lock_locked((spin_lock_t *) lock)
#define pthread_unlock(lock) spin_unlock((spin_lock_t *) lock)

/*
 * When DEBUG is on, try a bit of deadlock detection on locks.
 */
#ifdef	THREADS_DEBUG
OSKIT_INLINE void
pthread_lock(pthread_lock_t *lock)
{
	extern		void pthread_lock_panic(pthread_lock_t *lock);
#ifdef  SMP
	int		i;
	
	for (i = 0; i < 2000000; i++)
#endif
		if (spin_try_lock(lock))
			return;

	pthread_lock_panic(lock);
}
#else
#define pthread_lock(lock) spin_lock((spin_lock_t *) lock)
#endif

#else
#define pthread_lock_t spin_lock_t
#define PTHREAD_LOCK_INITIALIZER	SPIN_LOCK_INITIALIZER
#define pthread_lock_init(lock)		((void)0)
#define pthread_try_lock(lock)		(1)
#define pthread_lock_locked(lock)	(0)
#define pthread_unlock(lock)		((void)0)
#define pthread_lock(lock)		((void)0)
#endif

#endif /* _OSKIT_PTHREADS_LOCKING_H_ */
