/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Internal routines for threads package.
 */
#include <threads/pthread_internal.h>

/*
 * This is first routine called for every thread.
 */
void
pthread_start_thread(void (*function)(void *), void *argument)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	
	/*
	 * All threads start locked.
	 */
	pthread_unlock(&pthread->lock);

	/*
	 * All threads start out with ints blocked, which must be reset.
	 * The idle thread is run preemption disabled. All other threads
	 * run with both enabled.
	 */
	if (pthread == IDLETHREAD)
		splx(SPLPREEMPT);
	else
		splx(0);

	if (threads_debug)
		printf("pthread_start(%d): 0x%x(%d) f:0x%x a:0x%x\n",
		       THISCPU, (int) pthread, pthread->tid,
		       (int) function, (int) argument);
	
	function(argument);

        /* exit with no error */
        pthread_exit(0);

        /* NOT REACHED */
}

/*
 * The idle thread exists to spin looking for threads to run, and provide
 * a place for pthread_delay() to get called so that in usermode CPU time
 * is not wildly consumed.
 */
void
pthread_idle_function(void *arg)
{
	if (threads_debug)
		printf("pthread_idle(%d): starting\n", THISCPU);

	while (1) {
		if (! machine_intr_enabled())
			panic("pthread idleloop: Interrupts disabled!");
		
		pthread_reap_threads();

		/*
		 * If nothing to schedule call the delay function
		 */
		if (!pthread_sched_reschedule(RESCHED_INTERNAL, 0))
			pthread_delay();
	}
}

/*
 * Unblock all the threads in the given queue.
 *
 * The caller should have locked the Q.
 */
void
pthread_resume_allQ(queue_head_t *queue)
{
	pthread_thread_t	*pnext;

	queue_iterate(queue, pnext, pthread_thread_t *, chain) {
		if (pnext == CURPTHREAD())
			panic("pthread_resume_allQ");
		
		pthread_sched_setrunnable(pnext);
	}

	queue_init(queue);
}

/*
 * Remove the highest priority item from the Q and return it.
 *
 * The caller should have locked the Q.
 */
pthread_thread_t *
pthread_dequeue_fromQ(queue_head_t *queue)
{
	pthread_thread_t	*pnext = 0;

#ifdef  DEFAULT_SCHEDULER
	pthread_thread_t	*ptmp;
	int			maxpri = -1;

	queue_iterate(queue, ptmp, pthread_thread_t *, chain) {
		if (ptmp == CURPTHREAD())
			panic("pthread_resume_fromQ");
		
		if (ptmp->priority > maxpri) {
			maxpri = ptmp->priority;
			pnext  = ptmp;
		}
	}
	queue_remove(queue, pnext, pthread_thread_t *, chain);
#else
	queue_remove_first(queue, pnext, pthread_thread_t *, chain);
#endif
	return pnext;
}

/*
 * Remove a specific thread from a Q, but only if that thread is really
 * on the queue.
 *
 * The caller should have locked the Q.
 */
pthread_thread_t *
pthread_remove_fromQ(queue_head_t *queue, pthread_thread_t *pthread)
{
	pthread_thread_t	*ptmp = NULL_THREADPTR;

	queue_iterate(queue, ptmp, pthread_thread_t *, chain) {
		if (ptmp == CURPTHREAD())
			panic("pthread_remove_fromQ");
		
		if (ptmp == pthread) {
			queue_remove(queue, ptmp, pthread_thread_t *, chain);
			return ptmp;
		}
	}

	return NULL_THREADPTR;
}

void
pthread_lock_panic(pthread_lock_t *lock)
{
	panic("spin_lock: "
	      "Lock 0x%x locked for a long time. Maybe you deadlocked!",
	      (int) lock);
}
