/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
#ifdef  PRI_INHERIT
#ifndef DEFAULT_SCHEDULER
This priority inheritance algorithm is only good for the simple scheduler.
#endif

/*
 * This priority inheritance algorithm is based on the paper "Removing
 * Priority Inversion from an Operating System," by Steven Sommer.
 */
#include <threads/pthread_internal.h>

int	threads_priority_debug = 0;

void	pthread_priority_decreasing_recompute(pthread_thread_t *pthread);
void	pthread_priority_increasing_recompute(pthread_thread_t *pthread);

int	pthread_runq_onrunq(pthread_thread_t *pthread);
void	pthread_runq_insert_tail(pthread_thread_t *pthread);
void	pthread_runq_remove(pthread_thread_t *pthread);

/*
 * This is brought in from the scheduler.
 */
extern pthread_lock_t	pthread_sched_lock;


OSKIT_INLINE void
threads_change_priority(pthread_thread_t *pthread, int newprio)
{
	if (pthread_runq_onrunq(pthread)) {
		pthread_runq_remove(pthread);
		pthread->priority = newprio;
		pthread_runq_insert_tail(pthread);
	}
	else
		pthread->priority = newprio;
}

void
pthread_priority_inherit(pthread_thread_t *pwaiting_for)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s;

	/*
	 * Do priority inheritance. I'm not really sure how this will
	 * work on an SMP, so grab the scheduler lock which will prevent
	 * anything from happening elsewhere. Sledgehammer!
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);
	pthread_lock(&pthread_sched_lock);

	/*
	 * Add blocked thread to targets' list of waiters.
	 */
	pthread_lock(&pwaiting_for->lock);
	queue_enter(&pwaiting_for->waiters, pthread,
		    pthread_thread_t *, waiters_chain);
	pthread_unlock(&pwaiting_for->lock);

	/*
	 * Set the waiting link so that the chain of waiters
	 * can be followed when doing the priority transfer.
	 */
	pthread->waiting_for = pwaiting_for;

	pthread_priority_increasing_recompute(pthread);

	pthread_unlock(&pthread->lock);
	pthread_unlock(&pthread_sched_lock);
	splx(s);
}

/*
 * Do a transitive priority transfer. Following the waiting_for links,
 * transfering higher priority to lower priority threads.
 *
 * pthread is locked.
 */
void
pthread_priority_increasing_recompute(pthread_thread_t *pthread)
{
	pthread_thread_t	*pwaiting_for = pthread->waiting_for;

	pthread_lock(&pwaiting_for->lock);
	do {
		if (pthread->priority <= pwaiting_for->priority) {
			pthread_unlock(&pwaiting_for->lock);
			break;
		}

		if (threads_priority_debug)
			printf("Inherit: Transferring priority: "
			       "From 0x%x(%d/%d) to 0x%x(%d/%d)\n",
			       (int) pthread, pthread->tid, pthread->priority,
			       (int) pwaiting_for, pwaiting_for->tid,
			       pwaiting_for->priority);

		threads_change_priority(pwaiting_for, pthread->priority);
		pwaiting_for->inherits_from = pthread;
		
		pthread = pwaiting_for;
		pwaiting_for = pwaiting_for->waiting_for;
		pthread_unlock(&pthread->lock);
		if (pwaiting_for)
			pthread_lock(&pwaiting_for->lock);
	} while (pwaiting_for);
}	

/*
 * Undo the effects of a priority inheritance after a thread unblocks
 * another thread.
 */
void
pthread_priority_uninherit(pthread_thread_t *punblocked)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	pthread_thread_t	*ptmp1;
	int			s;

	/*
	 * I'm not really sure how this will work on an SMP, so grab
	 * the scheduler lock which will prevent anything from happening
	 * elsewhere. Sledgehammer!
	 */
	s = splhigh();
	pthread_lock(&pthread->lock);
	pthread_lock(&punblocked->lock);
	pthread_lock(&pthread_sched_lock);

	/*
	 * Remove the unblocked thread from the set of threads blocked
	 * by the current thread.
	 */
	queue_remove(&pthread->waiters,
		     punblocked, pthread_thread_t *, waiters_chain);

	/*
	 * The rest of the waiters are now waiting on the unblocked thread
	 * since it won the prize. Indicate that, and add the entire set
	 * of threads to the list of threads that punblocked is blocking.
	 */
	while (! queue_empty(&pthread->waiters)) {
		queue_remove_first(&pthread->waiters,
				   ptmp1, pthread_thread_t *, waiters_chain);

		ptmp1->waiting_for = punblocked;
		
		queue_enter(&punblocked->waiters,
			    ptmp1, pthread_thread_t *, waiters_chain);
	}
	punblocked->waiting_for = NULL_THREADPTR;

	/*
	 * Now recompute the priorities.
	 */
	if (pthread->inherits_from)
		pthread_priority_decreasing_recompute(pthread);
		      
	pthread_unlock(&pthread_sched_lock);
	pthread_unlock(&punblocked->lock);
	pthread_unlock(&pthread->lock);
	splx(s);
}

void
pthread_priority_decreasing_recompute(pthread_thread_t *pthread)
{
	int			priority = pthread->priority;
	pthread_thread_t	*pnext = 0, *ptmp;
	int			maxpri = -1;

	/*
	 * Find the highest priority thread from the set of threads
	 * waiting on pthread.
	 */
	queue_iterate(&pthread->waiters, ptmp, pthread_thread_t *, chain) {
		if (ptmp->priority > maxpri) {
			maxpri = ptmp->priority;
			pnext  = ptmp;
		}
	}

	/*
	 * If there is a waiting thread, and its priority is greater
	 * then this threads' priority, then inherit priority
	 * from that thread. Otherwise, reset this threads' priority
	 * back to its base priority since either there is nothing to
	 * inherit from (empty waiters), or its base priority is better
	 * than any of the waiting threads.
	 */
	if (pnext) {
		pthread_lock(&pnext->lock);

		if (threads_priority_debug)
			printf("Uninherit: Transferring priority: "
			       "From 0x%x(%d/%d) to 0x%x(%d/%d)\n",
			       (int) pnext, pnext->tid, pnext->priority,
			       (int) pthread, pthread->tid,
			       pthread->priority);

		if (pnext->priority > pthread->base_priority) {
			threads_change_priority(pthread, pnext->priority);
			pthread->inherits_from = pnext;
		}
		pthread_unlock(&pnext->lock);
	}
	else {
		if (threads_priority_debug)
			printf("Resetting priority back to base: "
			       "0x%x(%d) - %d to %d\n",
			       (int) pthread, pthread->tid,
			       pthread->priority, pthread->base_priority);

		threads_change_priority(pthread, pthread->base_priority);
		pthread->inherits_from = 0;
	}

	/*
	 * If this threads' priority was lowered, and another thread is
	 * inheriting from it, must propogate the lowered priority down
	 * the chain. This would not happen when a thread unblocks another
	 * thread. It could happen if an external event caused a blocked
	 * thread to change state, and that thread was donating its priority.
	 */
	if (pthread->priority < priority) {
		if (pthread->waiting_for &&  
		    pthread->waiting_for->inherits_from == pthread) {
			pthread_lock(&pthread->waiting_for->lock);
			pthread_priority_decreasing_recompute(pthread->
							      waiting_for);
			pthread_unlock(&pthread->waiting_for->lock);
		}
	}
}

/*
 * A thread is killed, but waiting on a thread. Must undo the inheritance.
 *
 * Interrupts should be disabled, and the thread locked.
 */
void
pthread_priority_kill(pthread_thread_t *pthread)
{
	pthread_thread_t	*pwaiting_for = pthread->waiting_for;
	
	pthread_lock(&pwaiting_for->lock);
	
	queue_remove(&pwaiting_for->waiters, pthread,
		     pthread_thread_t *, waiters_chain);

	pthread->waiting_for = NULL_THREADPTR;
	
	if (pwaiting_for->inherits_from == pthread)
		pthread_priority_decreasing_recompute(pwaiting_for);
	
	pthread_unlock(&pthread->waiting_for->lock);
}
#endif
