/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Exit code.
 */
#include <threads/pthread_internal.h>

static pthread_lock_t	reaper_lock;
static queue_head_t	reaper_queue;

/*
 * Just a dummy lock to satisfy the switch code.
 */
pthread_lock_t		threads_exit_lock    = PTHREAD_LOCK_INITIALIZER;

extern int		threads_alive;

void
pthread_exit(void *status)
{
	pthread_thread_t  *pthread = CURPTHREAD();
	int		  p;

	p = splhigh();
	pthread_lock(&pthread->lock);

	pthread_exit_locked(status);

	/*
	 * Never returns.
	 */
}

/*
 * Internal version. Thread lock is locked.
 */
void
pthread_exit_locked(void *status)
{
	pthread_thread_t  *pthread = CURPTHREAD();
		
	if (threads_debug)
		printf("pthread_exit(%d): 0x%x(%d) 0x%x\n",
		       THISCPU, (int) pthread, pthread->tid, (int) status);

	if (pthread->flags & THREAD_EXITING) {
		panic("pthread_exit: Recursive call: 0x%x(%d)\n",
		      (int) pthread, pthread->tid);
	}

	pthread->flags  |= THREAD_EXITING;
	pthread->exitval = (oskit_u32_t) status;
	pthread_unlock(&pthread->lock);

	/*
	 * A little iffy. Drop the spl and do whatever callouts are
	 * required. No locks are held!
	 */
	spllow();

	/*
	 * Call the cancelation handlers.
	 */
	pthread_call_cleanup_handlers();
		
	/*
	 * Call the key destructors.
	 */
	pthread_call_key_destructors();

#ifdef  PRI_INHERIT
	/*
	 * If this thread is inheriting from a thread (which means a thread
	 * or threads is waiting on it), its probably a bad thing cause
	 * some resource is locked up.
	 */
	if (! queue_empty(&pthread->waiters))
		printf("PTHREAD_EXIT: TID(%d) is exiting, but other "
		       "threads are waiting for it. DEADLOCK?\n",
		       pthread->tid);
#endif
	if (--threads_alive == 0)
		exit(0);
		
	/*
	 * Let the idle thread clean up the thread.
	 */
	splpreempt();
	pthread_lock(&reaper_lock);
	queue_enter(&reaper_queue, pthread, pthread_thread_t *, chain);
	pthread_sched_reschedule(RESCHED_BLOCK, &reaper_lock);
	
	/*
	 * Never returns.
	 */
}

/*
 * Reaper support. Let the idle thread clean up dead threads by calling
 * this each time through the idle loop.
 */
void
pthread_reap_thread(pthread_thread_t *pthread)
{
	int		s;
	
	s = splhigh();
	pthread_lock(&pthread->lock);
		
	if (pthread->flags & THREAD_DETACHED) {
		splx(s);
		pthread_destroy_internal(pthread);
	}
	else {
		if (pthread->joining_me) {
			pthread_thread_t	*joiner = pthread->joining_me;
			
			/*
			 * Make sure that thread is still in the
			 * joinwait, and restart it if so.
			 */
			pthread_lock(&joiner->lock);
			if (joiner->flags & THREAD_JOINWAIT) {
				joiner->flags &= ~THREAD_JOINWAIT;
				pthread_unlock(&joiner->lock);
				pthread_sched_setrunnable(joiner);
			}
			else
				pthread_unlock(&joiner->lock);
		}

		pthread->flags |= THREAD_EXITED;
		pthread_unlock(&pthread->lock);
		splx(s);
	}
}

void
pthread_reap_threads(void)
{
	pthread_thread_t	*pthread;
	
	pthread_lock(&reaper_lock);
	while (! queue_empty(&reaper_queue)) {
		queue_remove_first(&reaper_queue,
				   pthread, pthread_thread_t *, chain);
		pthread_reap_thread(pthread);
	}
	pthread_unlock(&reaper_lock);
}

void
pthread_init_exit()
{
	pthread_lock_init(&reaper_lock);
	queue_init(&reaper_queue);
}
