/*
 * Copyright (c) 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <threads/pthread_internal.h>
#include <oskit/time.h>

/*
 * Wait on a condition. The timedwait version is below. Because of the
 * timer, conditions have to be manipulated at splhigh to avoid deadlock
 * with the timer interrupt, which needs to lock both the thread and the
 * condition so it can remove the thread from the waitlist and restart
 * it. The length of time at splhigh is unfortunate.
 */
int
pthread_cond_wait(pthread_cond_t *c, pthread_mutex_t *m)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s;

	pthread_testcancel();

	s = splhigh();
	pthread_lock(&(c->lock));

        /* place ourself on the queue */
	queue_check(&(c->waiters), pthread);
	queue_enter(&(c->waiters), pthread, pthread_thread_t *, chain);
	
        /* unlock mutex */
        pthread_mutex_unlock(m);

	pthread_lock(&pthread->lock);
	pthread_unlock(&(c->lock));

	pthread->flags    |= THREAD_CONDWAIT;
	pthread->waitcond  = c;

	/* and block */
	pthread_sched_reschedule(RESCHED_BLOCK, &pthread->lock);

	splx(s);

        /* grab mutex before returning */
        pthread_mutex_lock(m);

	/* Look for a cancelation point */
	pthread_testcancel();

        return 0;
}

/*
 * Internal (safe) version that does not call pthread_testcancel.
 */
int
pthread_cond_wait_safe(pthread_cond_t *c, pthread_mutex_t *m)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s;

	s = splhigh();
	pthread_lock(&(c->lock));

        /* place ourself on the queue */
	queue_check(&(c->waiters), pthread);
	queue_enter(&(c->waiters), pthread, pthread_thread_t *, chain);
	
        /* unlock mutex */
        pthread_mutex_unlock(m);

	pthread_lock(&pthread->lock);
	pthread_unlock(&(c->lock));

	pthread->flags    |= THREAD_CONDWAIT;
	pthread->waitcond  = c;

	/* and block */
	pthread_sched_reschedule(RESCHED_BLOCK, &pthread->lock);

	splx(s);

        /* grab mutex before returning */
        pthread_mutex_lock(m);

        return 0;
}

/*
 * The timed version. Time is given as an absolute time in the future.
 */
int
pthread_cond_timedwait(pthread_cond_t *c, pthread_mutex_t *m,
		       oskit_timespec_t *abstime)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			s, error = 0;
	oskit_timespec_t	now;
        struct oskit_itimerspec ts = {{ 0, 0 }, { 0, 0 }};
	extern oskit_clock_t    *oskit_system_clock;    /* XXX */
	
	pthread_testcancel();

	s = splhigh();
	pthread_lock(&(c->lock));
	
	/*
	 * Oh, this is such a dumb interface! Now must check that the abstime
	 * has not passed, which means another check of the system clock,
	 * which is what the caller just did. But since the caller might not
	 * have disabled interrupts, the time could have passed by this point.
	 */
	oskit_clock_gettime(oskit_system_clock, &now);
	if (now.tv_sec > abstime->tv_sec ||
	    (now.tv_sec == abstime->tv_sec &&
	     now.tv_nsec >= abstime->tv_nsec)) {
		error = ETIMEDOUT;
		pthread_unlock(&(c->lock));
		/* unlock mutex */
		pthread_mutex_unlock(m);
		goto skip;
	}

        /* place ourself on the queue */
	queue_check(&(c->waiters), pthread);
	queue_enter(&(c->waiters), pthread, pthread_thread_t *, chain);

        /* unlock mutex */
        pthread_mutex_unlock(m);

	pthread_lock(&pthread->lock);
	pthread_unlock(&(c->lock));

	/* Start the timer */
	ts.it_value.tv_sec  = abstime->tv_sec;
	ts.it_value.tv_nsec = abstime->tv_nsec;
        oskit_timer_settime(pthread->condtimer, OSKIT_TIMER_ABSTIME, &ts);

	pthread->flags    |= THREAD_CONDWAIT;
	pthread->waitcond  = c;

#ifdef  THREADS_DEBUG
	pthread_lock(&threads_sleepers_lock);
	threads_sleepers++;
	pthread_unlock(&threads_sleepers_lock);
#endif
	/* and block */
	pthread_sched_reschedule(RESCHED_BLOCK, &pthread->lock);

#ifdef  THREADS_DEBUG
	pthread_lock(&threads_sleepers_lock);
	threads_sleepers--;
	pthread_unlock(&threads_sleepers_lock);
#endif
	/*
	 * Look for timeout.
	 */
	pthread_lock(&pthread->lock);

	if (pthread->flags & THREAD_TIMEDOUT) {
		pthread->flags &= ~THREAD_TIMEDOUT;
		error = ETIMEDOUT;
	}
	else {
		error = 0;
		
		/* disarm the timer. */
		ts.it_value.tv_sec  = 0;
		ts.it_value.tv_nsec = 0;
		oskit_timer_settime(pthread->condtimer, 0, &ts);
	}

	pthread_unlock(&pthread->lock);
	
   skip:
	splx(s);
	
        /* grab mutex before returning */
        pthread_mutex_lock(m);

	/* Look for a cancelation point */
	pthread_testcancel();

        return error;
}

/*      
 * Timer expiration. Note that two locks are taken at interrupt level;
 * the condition lock and the pthread lock.
 */
oskit_error_t
pthread_condwait_timeout(struct oskit_iunknown *listener, void *arg)
{
	pthread_thread_t	*pthread = arg;
	pthread_cond_t		*c = pthread->waitcond;
	int			s;

	s = splhigh();
	pthread_lock(&pthread->lock);

	if (threads_debug)
		printf("pthread_condwait_timeout: 0x%x 0x%x 0x%x 0x%x\n",
		       (int) CURPTHREAD(), s,
		       (int) pthread, (int) pthread->flags);

	/*
	 * Already woken up?
	 */
	if (! (pthread->flags & THREAD_CONDWAIT)) {
		pthread_unlock(&pthread->lock);
		splx(s);
		return 0;
	}

	/*
	 * Otherwise, lock the condition and remove the thread from
	 * the condition queue.
	 */
	pthread_lock(&(c->lock));
	queue_remove(&(c->waiters), pthread, pthread_thread_t *, chain);
	pthread_unlock(&(c->lock));

	pthread->flags |=  THREAD_TIMEDOUT;
	pthread->flags &= ~THREAD_CONDWAIT;
	pthread_unlock(&pthread->lock);

	/*
	 * And place it back on the runq. Request an AST if the current
	 * thread is no longer the thread that should be running.
	 */
	if (pthread_sched_setrunnable(pthread))
		softint_request(SOFTINT_ASYNCREQ);

	if (threads_debug)
		printf("pthread_condwait_timeout ends\n");

	splx(s);
	return 0;
}

/*
 * Internal function. Make the given thread (already locked) appear to
 * be waiting on the given condition variable.
 *
 * The thread is locked.
 */
void
pthread_cond_wait_other(pthread_cond_t *c, pthread_thread_t *pthread)
{
	int		s;

	s = splhigh();
	pthread_lock(&(c->lock));

        /* place ourself on the queue */
	queue_check(&(c->waiters), pthread);
	queue_enter(&(c->waiters), pthread, pthread_thread_t *, chain);
	
	pthread->flags    |= THREAD_CONDWAIT;
	pthread->waitcond  = c;
	
	pthread_unlock(&(c->lock));
	splx(s);
}
