/*
 * Copyright (c) 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <threads/pthread_internal.h>

/*
 * Signal a condition, waking up the highest priority thread waiting.
 */
int
pthread_cond_signal(pthread_cond_t *c)
{
	int			s;
	pthread_thread_t	*pnext;
	
	s = splhigh();
	pthread_lock(&(c->lock));
	
	if (queue_empty(&(c->waiters))) {
		pthread_unlock(&(c->lock));
		splx(s);
		return 0;
	}
	
	pnext = pthread_dequeue_fromQ(&(c->waiters));
	pthread_unlock(&(c->lock));

	/*
	 * Note race condition. Someone can try to cancel this thread
	 * between the time it comes off the queue, but before the state
	 * gets changed. pthread_cancel will have to deal with this.
	 */
	pthread_lock(&pnext->lock);
	pnext->flags &= ~THREAD_CONDWAIT;
	pnext->waitcond = 0;
	pthread_unlock(&pnext->lock);
	pthread_sched_setrunnable(pnext);
	splx(s);

	/*
	 * Yield if no longer highest priority.
	 */
	check_yield();

        return 0;
}

/*
 * Internal function. Terminate a condition wait for the first thread on
 * the queue, but do not reschedule the thread; just return it.
 */
pthread_thread_t *
pthread_cond_wakeup(pthread_cond_t *c)
{
	pthread_thread_t	*pnext;
	int			s;
	
	s = splhigh();
  again:
	pthread_lock(&(c->lock));
	
	if (queue_empty(&(c->waiters))) {
		pthread_unlock(&(c->lock));
		splx(s);
		return NULL_THREADPTR;
	}
	
	pnext = pthread_dequeue_fromQ(&(c->waiters));
	pthread_unlock(&(c->lock));

	/*
	 * Note race condition. Someone can try to cancel this thread
	 * between the time it comes off the queue, but before the state
	 * gets changed. See pthread_cancel().
	 */
	pthread_lock(&pnext->lock);
	pnext->flags &= ~THREAD_CONDWAIT;
	pnext->waitcond = 0;

	/*
	 * Check for cancelation. If canceled, reschedule this thread so
	 * the cancel is noticed. Then try for a new one.
	 */
	if (pnext->flags & THREAD_CANCELED) {
		pthread_unlock(&pnext->lock);
		pthread_sched_setrunnable(pnext);
		goto again;
	}	
	
	pthread_unlock(&pnext->lock);
	splx(s);

        return pnext;
}

/*
 * Internal function. Terminate a condition wait for a particular thread.
 * This does not ready the thread. The thread might not be on the condition
 * anymore.
 *
 * The thread is not locked!
 */
void
pthread_cond_wakeup_other(pthread_cond_t *c, pthread_thread_t *pthread)
{
	int			s;
	pthread_thread_t	*ptmp;
	queue_head_t		*q = &(c->waiters);
	
	s = splhigh();
	pthread_lock(&(c->lock));
	
	queue_iterate(q, ptmp, pthread_thread_t *, chain) {
		if (ptmp == pthread) {
			queue_remove(q, ptmp, pthread_thread_t *, chain);
			pthread_unlock(&c->lock);

			/*
			 * Note race condition (described above).
			 */
			pthread_lock(&pthread->lock);
			pthread->flags &= ~THREAD_CONDWAIT;
			pthread->waitcond = 0;
			pthread_unlock(&pthread->lock);
			splx(s);
			return;
		}
	}
	pthread_unlock(&c->lock);
	splx(s);
}
