/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Cancel Code.
 */
#include <threads/pthread_internal.h>
#include <threads/pthread_ipc.h>
#include <oskit/c/malloc.h>

int
pthread_cancel(pthread_t tid)
{
	pthread_thread_t  *pthread;
	int		  p;

	if (tid < 0 || tid >= THREADS_MAX_THREAD ||
	    ((pthread = tidtothread(tid)) == NULL_THREADPTR))
		return EINVAL;

	if (CURPTHREAD()->tid == tid) {
		pthread_exit((void *) -1);
		return 0;
	}

	p = splhigh();
	pthread_lock(&pthread->lock);

	if (pthread->flags & (THREAD_EXITED|THREAD_EXITING|THREAD_CANCELED)) {
		pthread_unlock(&pthread->lock);
		splx(p);
		return 0;
	}
	pthread->flags |= THREAD_CANCELED;

	/*
	 * Test cancel state. If disabled it must be deferred until
	 * cancelation is enabled.
	 */
	if (pthread->cancelstate == PTHREAD_CANCEL_DISABLE) {
		pthread_unlock(&pthread->lock);
		splx(p);
		return 0;
	}

	/*
	 * Test cancel type. In either case (ASYNC or DEFERRED), the thread
	 * is woken is up. The difference is that ASYNC forces the thread
	 * into pthread_exit immediately upon being switched back in, while
	 * DEFERRED allows the thread to continue executing until it
	 * reaches a cancelation point.  So, pthread_testcancel just calls
	 * pthread_exit if the cancelpending is set.
	 */
	if (pthread->canceltype != PTHREAD_CANCEL_DEFERRED)
		pthread->flags |= THREAD_KILLED;

#ifdef  PRI_INHERIT
	/*
	 * If this thread is inheriting from a thread (which means a thread
	 * or threads is waiting on it), its probably a bad thing cause
	 * some resource is locked up.
	 */
	if (! queue_empty(&pthread->waiters))
		printf("PTHREAD_CANCEL: TID(%d) was canceled, but other "
		       "threads are waiting for it. DEADLOCK?\n",
		       pthread->tid);

	/*
	 * If instead this thread is donating its priority to another
	 * thread that holds a resource, just undo the inheritance.
	 */
	if (pthread->waiting_for)
		pthread_priority_kill(pthread);
#endif
	
	/*
	 * All kinds of conditions are possible at this point.
	 */
#ifdef  CPU_INHERIT
	/*
	 * CPUI recv wait.
	 */
	if (pthread->flags & THREAD_CPUIRECV_WAIT) {
		pthread_sched_recv_cancel(pthread);
		goto done;
	}
#endif	
	/*
	 * IPC Wait.
	 */
	if (pthread->flags & THREAD_IPCWAIT_FLAG) {
		pthread_ipc_cancel(pthread);
		goto done;
	}
	
	/*
	 * If the thread is in an osenv_sleep(), issue a wakeup.
	 * The thread will be allowed to return through the sleep, to
	 * be caught sometime later. This allows driver state to be
	 * cleaned up before the thread is actually killed off.
	 */
	if (pthread->sleeprec) {
		pthread_unlock(&pthread->lock);
		osenv_wakeup(pthread->sleeprec, OSENV_SLEEP_CANCELED);
		goto done;
	}

	/*
	 * If the thread is THREAD_SLEEPING, then restart it. The death 
	 * will be noticed before the thread is allowed to return from 
	 * the call.
	 */
	if (pthread->flags & THREAD_SLEEPING) {
		pthread_unlock(&pthread->lock);
		pthread_wakeup_unlocked(pthread);
		goto done;
	}

	/*
	 * If the thread is THREAD_CONDWAIT, then restart it. The wrinkle
	 * is a race condition between the time the thread is taken off
	 * the condition queue and the time the thread state is changed.
	 */
	if (pthread->flags & THREAD_CONDWAIT) {
		pthread_cond_t	*c = pthread->waitcond;

		pthread_lock(&(c->lock));

		/*
		 * The thread was still on the Q, so its safe to change
		 * its state to reflect that it is not longer waiting
		 * on the condition. 
		 *
		 * If the thread was not on the Q, we caught the race,
		 * and do not have to do anything.
		 */
		if (pthread_remove_fromQ(&(c->waiters), pthread)) {
			pthread->flags   &= ~THREAD_CONDWAIT;
			pthread->waitcond = 0;
			pthread_unlock(&(c->lock));
			pthread_unlock(&pthread->lock);
			pthread_sched_setrunnable(pthread);
			goto done;
		}

		pthread_unlock(&pthread->lock);
		pthread_unlock(&(c->lock));
		goto done;
	}

	/*
	 * If the thread is THREAD_JOINWAIT, then restart it. 
	 */
	if (pthread->flags & THREAD_JOINWAIT) {
		pthread_thread_t	*joinee = pthread->joining_with;

		pthread_lock(&(joinee->lock));
		assert(joinee->joining_me);
		joinee->joining_me = NULL_THREADPTR;
		pthread_unlock(&(joinee->lock));

		pthread->flags &= ~THREAD_JOINWAIT;
		pthread_unlock(&pthread->lock);
		pthread_sched_setrunnable(pthread);

		goto done;
	}

	/*
	 * Must be running on another CPU. Must wait for it to be noticed.
	 */
	pthread_unlock(&pthread->lock);

   done:
	splx(p);
	check_yield();
	return 0;
}

int
pthread_setcancelstate(int state, int *oldstate)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			p;

	if (state != PTHREAD_CANCEL_ENABLE &&
	    state != PTHREAD_CANCEL_DISABLE)
		return EINVAL;

	p = splhigh();
	pthread_lock(&pthread->lock);
	*oldstate = pthread->cancelstate;
	pthread->cancelstate = state;
	pthread_unlock(&pthread->lock);
	splx(p);
	
	return 0;
}

int
pthread_setcanceltype(int type, int *oldtype)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int			p;

	if (type != PTHREAD_CANCEL_DEFERRED &&
	    type != PTHREAD_CANCEL_ASYNCHRONOUS)
		return EINVAL;

	p = splhigh();
	pthread_lock(&pthread->lock);
	*oldtype = pthread->canceltype;
	pthread->canceltype = type;
	pthread_unlock(&pthread->lock);
	splx(p);
	
	return 0;
}

void
pthread_testcancel(void)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	int                     s;

	s = splhigh();
	pthread_lock(&pthread->lock);

	if (pthread->cancelstate == PTHREAD_CANCEL_ENABLE &&
	    pthread->flags & THREAD_CANCELED) {
		pthread_exit_locked((void *) PTHREAD_CANCELED);

		/*
		 * Never returns.
		 */
	}

	pthread_unlock(&pthread->lock);
	splx(s);
}

/*
 * Push a cleanup handler.
 */
void
pthread_cleanup_push(void (*routine)(void *), void *arg)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	cleanup_t		*pcleanup;
	int			p;

	if ((pcleanup = (cleanup_t *) malloc(sizeof(cleanup_t))) == NULL)
		panic("pthread_cleanup_push: No more memory");

	pcleanup->func = routine;
	pcleanup->arg  = arg;

	/*
	 * Link it into the chain.
	 */
	p = splhigh();
	pthread_lock(&pthread->lock);
	pcleanup->next    = pthread->cleanups;
	pthread->cleanups = pcleanup;
	pthread_unlock(&pthread->lock);
	splx(p);
}

/*
 * Pop a cleanup handler, possibly executing it.
 */
void
pthread_cleanup_pop(int execute)
{
	pthread_thread_t	*pthread = CURPTHREAD();
	cleanup_t		*pcleanup;
	int			p;

	p = splhigh();
	pthread_lock(&pthread->lock);
	pcleanup          = pthread->cleanups;
	pthread->cleanups = pcleanup->next;
	pthread_unlock(&pthread->lock);
	splx(p);

	if (execute)
		pcleanup->func(pcleanup->arg);

	free(pcleanup);
}

/*
 * Call the cleanup handlers. Called from pthread_exit at termination.
 */
void
pthread_call_cleanup_handlers()
{
	pthread_thread_t	*pthread = CURPTHREAD();

	/*
	 * Don't have to lock the thread for accessing the cleanups, since
	 * only the current thread can muck with them.
	 */
	while (pthread->cleanups)
		pthread_cleanup_pop(1);
}
