/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Default implementation of osenv_sleep and friends.
 */ 

#include <threads/pthread_internal.h>
#include <oskit/debug.h>
#undef dump_stack_trace

void
pthread_init_osenv_sleep()
{
	/*
	 * A symbol to make sure this is picked up from the pthreads library.
	 */
}

void
osenv_sleep_init(osenv_sleeprec_t *sr) 
{
	pthread_thread_t	*pthread = CURPTHREAD();
	
	osenv_assert(sr);

	sr->data[0] = OSENV_SLEEP_WAKEUP;
	sr->data[1] = pthread;
}

int
osenv_sleep(osenv_sleeprec_t *sr) 
{
	volatile osenv_sleeprec_t *vsr      = sr;
	int			  s;
	pthread_thread_t	  *pthread;

	osenv_assert(sr);

	if (CURPTHREAD() == IDLETHREAD)
		panic("osenv_sleep: Idle thread tried to sleep");

	s = splhigh();
	if ((pthread = (pthread_thread_t *) sr->data[1]) == NULL_THREADPTR) {
		splx(s);
		return (int) vsr->data[0];
	}

	if (CURPTHREAD() != pthread)
		panic("osenv_sleep: Not the current thread!");

	if (pthread && threads_debug) {
		printf("osenv_sleep1: 0x%x(%d)\n", (int) pthread,pthread->tid);
	}

	/*
	 * Set the sleeprec in the thread for concelation handling.
	 * Then check for cancelation before going into the sleep.
	 */
	pthread_lock(&pthread->lock);
	if (pthread->flags & THREAD_CANCELED) {
		pthread_unlock(&pthread->lock);
		splx(s);
		return OSENV_SLEEP_CANCELED;
	}
	pthread->sleeprec = sr;
	pthread_unlock(&pthread->lock);
	
	/* Busy-wait until osenv_wakeup() clears the flag in the sleeprec */
	while ((pthread = (pthread_thread_t *) vsr->data[1])) {
		osenv_process_release();
		pthread_sleep(0);	/* No timeout, sleep forever */
		osenv_process_lock();
	}
	pthread = CURPTHREAD();
	pthread->sleeprec = 0;

	if (threads_debug)
		printf("osenv_sleep2: 0x%x(%d)\n",(int) pthread, pthread->tid);

	splx(s);
	return (int) vsr->data[0];
}

/*
 * This clears the condition that osenv_sleep is blocked on and
 * wakes up the thread blocked on it.
 */
void
osenv_wakeup(osenv_sleeprec_t *sr, int wakeup_status)
{
	pthread_thread_t	*pthread;
	int			resched, s;
	
	s = splhigh();

	/* Return immediately on spurious wakeup */
	if ((pthread = (pthread_thread_t *) sr->data[1]) == NULL_THREADPTR) {
		splx(s);
		return;
	}

	if (threads_debug)
		printf("osenv_wakeup: 0x%x(%d) status:%d\n",
		       (int) pthread, pthread->tid, wakeup_status);

	sr->data[0] = (void *)wakeup_status;
	sr->data[1] = (void *)0;
	resched = pthread_wakeup_unlocked(pthread);

	if (resched)
		softint_request(SOFTINT_ASYNCREQ);

	splx(s);
}
