/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include "svm_internal.h"

/*
 * Return information about each page in a region.
 * The region does not have to be part of a previously
 * For each page in the region, one byte in `pagemap' is
 * written indicating whether the page is in core and
 * if so, whether it has been referenced or modified
 * since the last svm_incore call or remapping.
 *
 * Address and len must be page aligned.
 */
int
svm_incore(oskit_addr_t addr, oskit_size_t len, char *pagemap)
{
	oskit_addr_t	base = addr;
	char		*p = pagemap;

	/*
	 * Check for page alignment of both address and length.
	 */
	if ((base && !page_aligned(base)) || !page_aligned(len))
		return OSKIT_E_INVALIDARG;

	/* XXX bad scene if we fault writing pagemap */
	SVM_LOCK();

	while (len) {
		pt_entry_t	*pte  = pdir_find_pte(base_pdir_pa, addr);

		if (!pte)
			*p = 0;
		else {
			oskit_addr_t pa = pte_to_pa(*pte);
			int mbits = unmap_modebits(pte_to_modebits(*pte));
#ifdef  DEBUG_SVM
			printf("svm_incore: "
			       "pte for 0x%x = 0x%x\n", (int) addr, (int) pte);
#endif
			if (mbits & PTE_MBITS_VALID) {
				*p = SVM_INCORE_INCORE;
				if (mbits & PTE_MBITS_REF)
					*p |= SVM_INCORE_REFERENCED;
				if (mbits & PTE_MBITS_MOD)
					*p |= SVM_INCORE_MODIFIED;
				if (mbits & (PTE_MBITS_REF|PTE_MBITS_MOD))
					*pte = create_pte(pa, map_modebits(mbits & ~(PTE_MBITS_REF|PTE_MBITS_MOD)));
			}
			else
				*p = 0;
		}

		addr += PAGE_SIZE;
		len  -= PAGE_SIZE;
		++p;
	}

	SVM_UNLOCK();
	return 0;
}
