/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * start_network.c
 *
 * start up a simple interface via bootp or keyboard input
 */
#include <oskit/startup.h>
#include <oskit/dev/error.h>
#include <oskit/dev/ethernet.h>
#include <oskit/dev/freebsd.h>
#include <oskit/dev/net.h>
#include <oskit/dev/linux.h>
#include <oskit/net/freebsd.h>
#include <oskit/net/socket.h>
#include <oskit/misc/sysconf.h>
#include <oskit/c/fd.h>
#include <oskit/startup.h>

#include <oskit/c/stdio.h>
#include <oskit/c/string.h>
#include <oskit/c/assert.h>

#include "bootp.h"

#ifdef	PTHREADS
#include <oskit/threads/pthread.h>

#define start_network	start_network_pthreads
#endif

#define VERBOSITY 0

/* this name shouldn't matter */
#define IFNAME1		"de0"
#define IFNAME2 	"hpplus0"


char *ifnames[] =
{	"de0",
	"hpplus0",
	0
};

static char ipaddr[20];
static char gateway[20];
static char netmask[20];
extern long secondswest;

/*
 * global variables related to networking
 */
extern char hostname[256];
static struct oskit_freebsd_net_ether_if *eif;

void
start_network()
{
	oskit_etherdev_t **etherdev;
	oskit_socket_factory_t *fsc;
	int   ndev, err;
	char* ifname;

	start_devices();

        /*
         * Find all the Ethernet device nodes.
         */
        ndev = osenv_device_lookup(&oskit_etherdev_iid, (void***)&etherdev);
        if (ndev <= 0)
                panic("no Ethernet adaptors found!");

        /* get ip addr via bootp or console */
        get_ipinfo(etherdev[0], ipaddr, gateway, netmask,
		hostname, sizeof hostname, &secondswest);

	err = oskit_freebsd_net_init(&fsc);
	assert(!err);
#ifdef  PTHREADS
	pthread_init_socketfactory(fsc);
#else
	oskit_register(&oskit_socket_factory_iid, (void *) fsc);
#endif

	err = oskit_freebsd_net_open_ether_if(etherdev[0], &eif);
	assert(!err);

#if VERBOSITY > 0
	printf("hostname = `%s'\n", hostname);
	printf("gateway = `%s'\n", gateway);
	printf("ipaddr = `%s'\n", ipaddr);
	printf("netmask = `%s'\n", netmask);
#endif

	ifname = ifnames[0];
	do {
		err = oskit_freebsd_net_ifconfig(eif, ifname, ipaddr, netmask);
	} while (err && ifname++);

	if (err)
		printf("%s: couldn't find an ethernet device!\n",
		       __FUNCTION__);

	assert(!err);

	if ((err = oskit_freebsd_net_add_default_route(gateway))) {
		printf("couldn't add default route to %s\n(error=%d; `%s')\n",
			gateway, err, strerror(err));
	}
}


/*
 * Having this initialization here makes `start_devices' (start_devices.c)
 * initialize and probe the net drivers, since linking in this file
 * means the program is using the network.
 */
void (*_init_devices_net) = oskit_linux_init_net;
