/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <oskit/startup.h>
#include <oskit/dev/dev.h>
#include <oskit/dev/linux.h>
#include <oskit/fs/netbsd.h>
#include <oskit/fs/filesystem.h>
#include <oskit/fs/file.h>
#include <oskit/fs/dir.h>
#include <oskit/io/blkio.h>
#include <oskit/diskpart/diskpart.h>
#include <oskit/principal.h>
#include <oskit/boolean.h>

#include <oskit/c/stdio.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/unistd.h>
#include <oskit/c/fs.h>
#include <oskit/c/fd.h>

#ifdef PTHREADS
#include <oskit/com/wrapper.h>
#include <oskit/threads/pthread.h>

void		fs_master_lock(void);
void		fs_master_unlock(void);
oskit_error_t	fs_netbsd_threaded_init(void);
#define fs_netbsd_init		fs_netbsd_threaded_init
#define	start_fs		start_fs_pthreads
#define	start_fs_on_blkio	start_fs_on_blkio_pthreads
#else
#define osenv_process_lock()
#define osenv_process_unlock()
#endif

extern int oskit_usermode_simulation;

static oskit_filesystem_t *fs;

oskit_principal_t *cur_principal; /* identity of current client process */

oskit_error_t oskit_get_call_context(const struct oskit_guid *iid, void **out_if)
{
    if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
        memcmp(iid, &oskit_principal_iid, sizeof(*iid)) == 0) {
        *out_if = cur_principal;
        oskit_principal_addref(cur_principal);
        return 0;
    }

    *out_if = 0;
    return OSKIT_E_NOINTERFACE;
}

static void
release_and_sync()
{
    fs_release();
    printf("Syncing disks ... ");
    oskit_filesystem_sync(fs, 1);
    printf("done.\n");
}


void
start_fs_on_blkio(oskit_blkio_t *part)
{
    oskit_identity_t id;
    oskit_dir_t *root;
    int rc;

    osenv_process_lock();

    /* initialize the file system code */
    rc = fs_netbsd_init();
    if (rc)
    {
        printf("fs_netbsd_init() failed:  errno 0x%x\n", rc);
        exit(rc);
    }

    /* establish client identity */
    id.uid = 0;
    id.gid = 0;
    id.ngroups = 0;
    id.groups = 0;
    rc = oskit_principal_create(&id, &cur_principal);
    if (rc)
    {
        printf("oskit_principal_create() failed:  errno 0x%x\n", rc);
        exit(rc);
    }

    /* mount a filesystem */
    rc = fs_netbsd_mount(part, 0, &fs);
    if (rc)
    {
	printf("fs_netbsd_mount() failed:  errno 0x%x\n", rc);
	exit(rc);
    }

    if (!oskit_usermode_simulation) {
	/* Don't need the part anymore, the filesystem has a ref. */
	oskit_blkio_release(part);
    }

#ifdef PTHREADS
    {
	oskit_filesystem_t	*wrappedfs;

	/* Wrap it up! */
	rc = oskit_wrap_filesystem(fs,
			(void (*)(void *))fs_master_lock, 
			(void (*)(void *))fs_master_unlock,
			0, &wrappedfs);

	if (rc)
	{
	    printf("oskit_wrap_filesystem() failed: errno 0x%x\n", rc);
	    exit(rc);
	}

	/* Don't need the fs anymore, the wrapper has a ref. */
	oskit_filesystem_release(fs);
	fs = wrappedfs;

	osenv_process_unlock();

	/*
	 * oskit_filesystem_getroot will return a wrapped root dir.
	 */
    }
#endif
    
    rc = oskit_filesystem_getroot(fs, &root);
    if (rc)
    {
	printf("oskit_filesystem_getroot() failed: errno 0x%x\n", rc);
	exit(rc);
    }

    rc = fs_init(root);
    if (rc)
    {
        printf("fs_fs_init() failed:  errno 0x%x\n", rc);
        exit(rc);
    }

    /* Don't need the root anymore, fs_init took a couple of refs. */
    oskit_dir_release(root);

    if (oskit_usermode_simulation) {
	atexit(fs_release);
    } else {
	atexit(release_and_sync);
    }

    /* DEBUG: watch for someone to corrupt us */
    /* setup_pointer_trap(&root->ops); */
}

void
start_fs(const char *diskname, const char *partname)
{
	int rc;
	oskit_blkio_t *part;

	osenv_process_lock();
	rc = start_disk(diskname, partname, 0, &part);
	osenv_process_unlock();
	if (rc) {
		printf("start_disk() failed:  ernno 0x%x\n", rc);
		exit(rc);
	}

	start_fs_on_blkio(part);
}
