/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * bootp.c 
 */
#include <oskit/net/bootp.h>

#include <oskit/c/sys/types.h>
#include <oskit/c/sys/stat.h>
#include <oskit/c/stdio.h>
#include <oskit/c/string.h>
#include <oskit/c/assert.h>
#include <oskit/c/unistd.h>
#include <oskit/c/fcntl.h>
#include <oskit/c/arpa/inet.h>
#include <oskit/c//errno.h>

#include "bootp.h"

#define WRITE(fd, m) 	write(fd, m, strlen(m))

/*
 * read an IP address from the console
 */
void
get_net_addr(char *buf)
{
	struct in_addr ip;

	do {
		fgets(buf, 16 /* sizeof "XXX.XXX.XXX.XXX" */, stdin);
		/* trailing newline shouldn't hurt... */
	} while (inet_aton(buf, &ip) == 0);
}

/*
 * first try to contact bootp server for ip addr, if 
 * that fails, ask the user
 */
void
get_ipinfo(oskit_etherdev_t *dev, char *ipaddr, char *gateway, char *netmask,
	char *hostname, int hostnamesize, long *howfarwest)
{
	struct bootp_net_info bootpinfo;
	int i, err;
	int have_dns, have_etc_hosts, have_domain, resolv_conf, host_conf;

	/* get the IP address & other info */
	err = bootp(dev, &bootpinfo);

	if (!err) {
		if (ipaddr && (bootpinfo.flags & BOOTP_NET_IP))
			strcpy(ipaddr, inet_ntoa(bootpinfo.ip));

		if (gateway && (bootpinfo.flags & BOOTP_NET_GATEWAY))
			strcpy(gateway, inet_ntoa(bootpinfo.gateway.addr[0]));

		if (netmask)
			if (bootpinfo.flags & BOOTP_NET_NETMASK)
				strcpy(netmask, inet_ntoa(bootpinfo.netmask));
			else
				strcpy(netmask, inet_ntoa(
					bootp_default_netmask(bootpinfo.ip)));

		if (howfarwest && (bootpinfo.flags & BOOTP_NET_TIME_OFFSET))
				*howfarwest = bootpinfo.time_offset;

		if ((bootpinfo.flags & BOOTP_NET_HOSTNAME) && hostname)
			strncpy(hostname, bootpinfo.hostname, hostnamesize);

		/*
		 * Now write DNS support files
		 */

		if (mkdir("/etc", 0777) < 0 && errno != EEXIST)
			perror("mkdir /etc");

		have_dns = bootpinfo.flags & BOOTP_NET_DNS_SERVER;
		have_domain = bootpinfo.flags & BOOTP_NET_DOMAINNAME;
		have_etc_hosts = access("/etc/hosts", R_OK) == 0;

		host_conf = open("/etc/host.conf",
				 O_RDWR|O_CREAT|O_TRUNC|O_EXCL, 
				 0644);
		if (host_conf >= 0) {
			if (have_dns)
				WRITE(host_conf, "bind\n");
			if (have_etc_hosts)
				WRITE(host_conf, "hosts\n");
			close(host_conf);
		} else
			perror("couldn't create /etc/host.conf");

		resolv_conf = open("/etc/resolv.conf",
				   O_RDWR|O_CREAT|O_TRUNC|O_EXCL,
				   0644);
		if (resolv_conf >= 0) {
			if (have_domain) {
				WRITE(resolv_conf, "search\t");
				WRITE(resolv_conf, bootpinfo.domainname);
				WRITE(resolv_conf, "\n");
			}
			if (have_dns) {
				struct bootp_addr_array ba = 
					bootpinfo.dns_server;
				for (i = 0; i < ba.len; i++) {
					char *s = inet_ntoa(ba.addr[i]);
					WRITE(resolv_conf, "nameserver\t");
					WRITE(resolv_conf, s);
					WRITE(resolv_conf, "\n");
				}
			}
			close(resolv_conf);
		} else
			perror("couldn't create /etc/resolv.conf");

		bootp_free(&bootpinfo);

	} else {
		if (ipaddr) {
			printf("What is my IP address: ");
			get_net_addr(ipaddr);
		}

		if (gateway) {
			printf("What is my default gateway: ");
			get_net_addr(gateway);
		}

		if (netmask) {
			printf("What is my netmask: ");
			get_net_addr(netmask);
		}

                if (hostname) {
                        printf("What is my hostname: ");
                        /* requires fd_set_console!!! */
                        fgets(hostname, hostnamesize, stdin);
                        hostname[strlen(hostname)-1] = '\n';
		}
	}
}
