/*
 * Copyright (c) 1997, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * this is a completely ad-hoc signal handling facility,
 *
 * compliant to pretty much nothing. Complicated by trying to support
 * both a single-threaded and multi-thraeded version of the library.
 */
#include <oskit/c/errno.h>
#include <sys/types.h>
#include <signal.h>
#include <strings.h>
#include <stdlib.h>

#ifdef THREAD_SAFE
#include <oskit/threads/pthread.h>

/* Need to lock the signal state. */
static pthread_mutex_t	sigmutex;
#define sig_lock()	pthread_mutex_lock(&sigmutex)
#define sig_unlock()	pthread_mutex_unlock(&sigmutex)

/*
 * sigmask is thread specific, and is accessed using the standard pthread
 * interface function pthread_sigmask(). This makes the implementation a
 * bit messier since sigprocmask duplicates some of pthread_sigmask.
 */
static inline void
getsigmask(sigset_t *oldmask)
{
	pthread_sigmask(0, 0, oldmask);
}

static inline void
setsigmask(sigset_t *newmask)
{
	pthread_sigmask(SIG_SETMASK, newmask, 0);
}
#else
#define sig_lock()
#define sig_unlock()

static sigset_t		__sigmask;
#define getsigmask(m)	(*(m) = __sigmask)
#define setsigmask(m)	(__sigmask = *(m))
#endif

static struct sigaction sigactions[NSIG];
static sigset_t		sigpend;

/* This handles the details of posting the signal to the application. */
void	really_deliver_signal(int sig, siginfo_t *code,struct sigcontext *scp);

/*
 * Init the signal code. Must be called from oskit_init_libc if the
 * application wants to properly handle signals.
 */
void
signals_init(void)
{
	int	i;

#ifdef  THREAD_SAFE
	pthread_mutex_init(&sigmutex, &pthread_mutexattr_default);
#endif
	/* Initialize the default signal actions. */
	for (i = 0; i < NSIG; i++)
		sigactions[i].sa_handler = SIG_DFL;

	libc_sendsig_init();
}

/*
 * sigaction
 */
int
sigaction(int sig, const struct sigaction *act, struct sigaction *oact)
{
	if (sig < 0 || sig >= NSIG)
		return errno = EINVAL, -1;

	sig_lock();

	if (oact)
		*oact = sigactions[sig];
	if (act)
		sigactions[sig] = *act;

	sig_unlock();
	return 0;
}

/*
 * sigprocmask
 */
int
sigprocmask(int how, const sigset_t *set, sigset_t *oset)
{
	sigset_t	sigmask;

	sig_lock();
	getsigmask(&sigmask);

	if (oset)
		*oset = sigmask;

	if (set) {
		switch (how) {
		case SIG_BLOCK:
			sigmask |= *set;
			break;
		case SIG_UNBLOCK:
			sigmask &= ~*set;
			break;
		case SIG_SETMASK:
			sigmask = *set;
			break;
		default:
			errno = EINVAL;
			return -1;
		}
	}
	setsigmask(&sigmask);
	sig_unlock();

	/*
	 * Look for pending signals that are now unblocked, and deliver.
	 */
	while (sigpend & ~sigmask) {
		int sig = ffs(sigpend & ~sigmask);

		raise(sig);
	}
	return 0;
}

/*
 * raise a signal. Note that taking the siglock here is actually bogus since
 * raise is called out of upcalls (setitimer) from the kernel library. If
 * the lock to held, all is lost.
 */
int
raise(int sig)
{
	struct sigcontext	sc;
	sigset_t		sigmask;
	siginfo_t		siginfo;

	if (sig < 0 || sig >= NSIG)
		return EINVAL;

	sig_lock();
	sigaddset(&sigpend, sig);

	getsigmask(&sigmask);
	if (sigismember(&sigmask, sig)) {
		sig_unlock();
		return 0;
	}

	/* create a stub sigcontext_t. */
	bzero(&sc, sizeof(sc));

	siginfo.si_signo           = sig;
	siginfo.si_code            = SI_USER;
	siginfo.si_value.sival_int = 0;

	really_deliver_signal(sig, &siginfo, &sc);
	/*
	 * signal lock is already released.
	 */

	return 0;
}

/*
 * Deliver a signal generated from a trap. This is the upcall from the
 * machine dependent code that created the sigcontext structure from
 * the trap state. Note that interrupt type signals are not generated this
 * way, and is really the reason this stuff is bogus.
 */
void
oskit_libc_sendsig(int sig, int code, struct sigcontext *scp)
{
	sigset_t		sigmask;
	siginfo_t		siginfo;

	sig_lock();

	/*
	 * Look at the sigmask. If the signal is blocked, it means
	 * deadlock since there will be no way for the signal to be
	 * unblocked. Thats a feature of this really dumb implementation
	 */
	getsigmask(&sigmask);
	if (sigismember(&sigmask, sig)) {
		panic("deliver_signal: "
		      "Signal %d (an exception) is blocked", sig);
	}

	siginfo.si_signo           = sig;
	siginfo.si_code            = SI_EXCEP;
	siginfo.si_value.sival_int = code;

	really_deliver_signal(sig, &siginfo, scp);

	/*
	 * signal lock is already released.
	 */
}

/*
 * This is called with the signal lock taken so that mask and the
 * sigacts structure is protected until the handler is actually
 * called. The lock is released before calling the handler, and control
 * returns to the caller with the lock unlocked. This is not the most
 * ideal way to do this, but it needs to be simple right now.
 */
void
really_deliver_signal(int sig, siginfo_t *info, struct sigcontext *scp)
{
	sigset_t		oldmask, sigmask;
	struct sigaction	*act;

	act = &sigactions[sig];

	if (act->sa_handler == SIG_IGN || act->sa_handler == SIG_ERR) {
		sig_unlock();
		return;
	}

	if (act->sa_handler == SIG_DFL) {
		/* Default action for all signals is termination */
		panic("libc sendsig: Signal %d caught but no handler", sig);
	}

	getsigmask(&oldmask);
	sigmask = oldmask;
	sigaddset(&sigmask, sig);
	sigmask |= sigactions[sig].sa_mask;
	sigdelset(&sigpend, sig);
	setsigmask(&sigmask);
	sig_unlock();

	/* call the handler */
	if (sigactions[sig].sa_flags & SA_SIGINFO)
		sigactions[sig].sa_sigaction(sig, info, (void *) scp);
	else
		((void (*)(int, int, struct sigcontext *))
		 sigactions[sig].sa_handler)(sig,
					     info->si_value.sival_int, scp);

	sig_lock();
	setsigmask(&oldmask);
	sig_unlock();
}

/*
 * Compatability.
 */
int
sigblock(int sigmask)
{
	sigset_t	oldmask;
	const sigset_t	newmask = sigmask;

	sigprocmask(SIG_BLOCK, &newmask, &oldmask);

	return (int) oldmask;
}

int
sigsetmask(int sigmask)
{
	sigset_t	oldmask;
	const sigset_t	newmask = sigmask;

	sigprocmask(SIG_SETMASK, &newmask, &oldmask);

	return (int) oldmask;
}
