/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <oskit/fs/file.h>
#include <oskit/fs/dir.h>
#include "fs.h"

/*
 * mount a directory denoted by `src' under destination `dst' 
 * in the current tree
 */
oskit_error_t
fs_mount(const char *path, oskit_file_t *file)
{
	oskit_dir_t *d;
	oskit_file_t *f;
	struct fs_mountpoint *mp;
	oskit_error_t rc;

	/* allow for fs_mount("/", file); */
	if (!fs_root) {
		if (!strcmp(path, "/")) {
			rc = oskit_file_query(file, &oskit_dir_iid, (void **)&d);
			oskit_file_release(f);
			if (rc)
				return rc == OSKIT_E_NOINTERFACE ? 
					OSKIT_ENOTDIR : rc;
			return fs_init(d);
		}
		return OSKIT_ENXIO;
	}
	/* Lookup the mount point */
	rc = fs_lookup(path, &f);
	if (rc)
		return rc;

	/* Make sure it's a directory */
	rc = oskit_file_query(f, &oskit_dir_iid, (void **)&d);
	oskit_file_release(f);
	if (rc)
		return rc == OSKIT_E_NOINTERFACE ? OSKIT_ENOTDIR : rc;

	/* Create a mountpoint node for it */
	mp = malloc(sizeof *mp);
	if (mp == NULL) {
		oskit_dir_release(d);
		return OSKIT_ENOMEM;
	}
	mp->subtree = file;
	oskit_file_addref(file);
	oskit_file_query(file, &oskit_iunknown_iid, (void**)&mp->subtree_iu);
	mp->mountover = d;
	oskit_dir_query(d, &oskit_iunknown_iid, (void**)&mp->mountover_iu);
	mp->next = fs_mountpoints;
	fs_mountpoints = mp;

	return 0;
}

