/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * lookup a file or directory
 *
 * Now depends on a global fs_cwd and fs_root holding current working
 * directory and root directory
 */

#include <oskit/fs/file.h>
#include <oskit/fs/openfile.h>
#include <oskit/fs/dir.h>

#if VERBOSITY > 2
#include <stdio.h>
#endif
#include <string.h>
#include <assert.h>
#include "fs.h"

#define MAXPATHLENGTH	1024
#define MAXSYMLINKDEPTH	20

#ifndef VERBOSITY
#define VERBOSITY 0
#endif

/*
 * look up file with name name relative to dir base
 *
 * symlinkdepth tells the number of followed symlinks
 */
oskit_error_t
fs_lookup(const char *name, oskit_file_t **fileret)
{
	oskit_error_t	rc;
	oskit_dir_t	*dir;
	oskit_file_t	*file;
	char		wname[MAXPATHLENGTH];
	char		*comp;
	char	 	*s;
	char		savechar;
	char		link[MAXPATHLENGTH];
	oskit_u32_t	linksize;
	int		symlinkdepth = 0;
	struct fs_mountpoint *mp;
	oskit_iunknown_t	*iu;

#if VERBOSITY > 2
	printf(__FUNCTION__": looking up `%s'\n", name);
#endif

	/* Copy the pathname so we can modify it */
	if (strlen(name) + 1 > sizeof(wname))
		return OSKIT_ENAMETOOLONG;
	strcpy(wname, name);
	s = wname;

	/* 
 	 * On loop entry, we hold an additional ref to the base directory.
	 * Fail if the root or cwd haven't been initialized.
	 */
	dir = (wname[0] == '/') ? fs_root : fs_cwd;
	if (dir == 0)
		return OSKIT_ENOENT;
	oskit_dir_addref(dir);

	/*
	 * Note that, to be consistent with standard practice,
	 * a lookup of "foo/" succeeds only if 'foo' is a directory.
	 */
	while (1) {
		/* skip slashes */
		while (*s == '/')
			s++;

		if (!*s) {	/* this was the last component */
			*fileret = (oskit_file_t*)dir;
			return 0;
		}

		/* Break out the next component name */
		comp = s;
		while (*s && *s != '/')
			s++;

		/* See if we're finding the parent of a mounted file system */
		if (comp[0] == '.' && comp[1] == '.' && (s == comp + 2)) {
			oskit_dir_query(dir, &oskit_iunknown_iid, (void**)&iu);
			mountparent:
			for (mp = fs_mountpoints; mp; mp = mp->next) {
				if (iu == mp->subtree_iu) {
					oskit_dir_release(dir);
					oskit_iunknown_release(iu);
					dir = mp->mountover;
					oskit_dir_addref(dir);
					iu = mp->mountover_iu;
					oskit_iunknown_addref(iu);
					goto mountparent;
				}
			}
			oskit_iunknown_release(iu);
		}

#if VERBOSITY > 2
		printf(__FUNCTION__": looking up comp=`%s'\n", comp);
#endif
		savechar = *s;
		*s = 0;
		rc = oskit_dir_lookup(dir, comp, &file);
		*s = savechar;
		if (rc) {
			oskit_dir_release(dir);
			return rc;
		}

		/* See if it's a mount point and translate it if so */
		oskit_file_query(file, &oskit_iunknown_iid, (void**)&iu);
		mountpoint:
		for (mp = fs_mountpoints; mp; mp = mp->next) {
			if (iu == mp->mountover_iu) {
				oskit_file_release(file);
				oskit_iunknown_release(iu);
				file = mp->subtree;
				oskit_file_addref(file);
				iu = mp->subtree_iu;
				oskit_iunknown_addref(iu);
				goto mountpoint; /* support multilevel mounts */
			}
		}
		oskit_iunknown_release(iu);

		/* See if it's a symlink */
		rc = oskit_file_readlink(file, link, sizeof(link), &linksize);
		if (!rc) {
			oskit_file_release(file);

			/* Put together the new path to traverse */
			if (linksize + strlen(s) + 1 > sizeof(link)) {
				oskit_dir_release(dir);
				return OSKIT_ENAMETOOLONG;
			}
			strcpy(link + linksize, s);
			strcpy(wname, link);
			s = wname;

			/* Avoid infinite symlink loops */
			if (symlinkdepth > MAXSYMLINKDEPTH) {
				/* too many nested symlinks */
				oskit_dir_release(dir);
				return OSKIT_ELOOP;
			}
			symlinkdepth++;

			/*
			 * If it's an absolute symlink,
			 * restart traversal at the root directory.
			 */
			if (wname[0] == '/') {
				oskit_dir_release(dir);
				dir = fs_root;
				oskit_dir_addref(dir);
			}

			continue;
		}

		oskit_dir_release(dir);

		if (!*s) {	/* this was the last component */
			*fileret = file;
			return 0;
		}

		/* 
		 * see if it's a directory
		 */
		rc = oskit_file_query(file, &oskit_dir_iid, (void **)&dir);
		oskit_file_release(file);
		if (rc)
			return (rc == OSKIT_E_NOINTERFACE) ? OSKIT_ENOTDIR : rc;
	}
}

