/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Handles details of read/write locking file descriptors
 */
#ifdef	THREAD_SAFE
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include "fd.h"

/*
 * Very simple model right now. An fd can be locked for read and/or
 * write access. This allows two threads to concurrently read/write,
 * but not read/read or write/write. Locks are also recursive.
 *
 * The FD mutex must already be held!
 */
int
fd_access_lock(int fd, int type, int block)
{
	fd_t		*fdp  = fd_array + fd;
	int		self  = pthread_self();

	if (!fdp->lock)
		return 1;

	/*
	 * If nonblocking, quick test.
	 */
	if (!block) {
		if ((type & FD_WRITE) && fdp->writer && fdp->writer != self)
			goto noblock;
		if ((type & FD_READ)  && fdp->reader && fdp->reader != self)
			goto noblock;
	}

	if (type & FD_WRITE) {
		/*
		 * Look for recursive write lock.
		 */
		if (fdp->writer == self) {
			fdp->writecount++;
			goto rlock;
		}

		/*
		 * Loop on the condition, waiting until the writer is null.
		 */
		while (fdp->writer)
			pthread_cond_wait(fdp->cond, fdp->lock);

		/*
		 * We got it. Change the state.
		 */
		fdp->writer     = self;
		fdp->writecount = 1;
	}
  rlock:
	if (type & FD_READ) {
		/*
		 * Look for recursive read lock. 
		 */
		if (fdp->reader == self) {
			fdp->readcount++;
			goto done;
		}

		/*
		 * Loop on the condition, waiting until the reader is null.
		 */
		while (fdp->reader)
			pthread_cond_wait(fdp->cond, fdp->lock);

		/*
		 * We got it. Change the state.
		 */
		fdp->reader    = self;
		fdp->readcount = 1;
	}
   done:
	/*
	 * Okay, now the fd mutex can be safely released.
	 */
	pthread_mutex_unlock(fdp->lock);
	return 1;
noblock:
	pthread_mutex_unlock(fdp->lock);
	return 0;
}

void
fd_access_unlock(int fd, int type)
{
	fd_t		*fdp  = fd_array + fd;
	int		self  = pthread_self();

	if (!fdp->lock)
		return;

	pthread_mutex_lock(fdp->lock);
	
	if (type & FD_WRITE) {
		assert(fdp->writer == self);
		
		/*
		 * Look for recursive write lock.
		 */
		if (--fdp->writecount != 0)
			goto rlock;

		/*
		 * Free the writelock and signal any waiters.
		 */
		fdp->writer = 0;
		pthread_cond_signal(fdp->cond);
	}
  rlock:
	if (type & FD_READ) {
		assert(fdp->reader == self);

		/*
		 * Look for recursive read lock. 
		 */
		if (--fdp->readcount != 0)
			goto done;

		/*
		 * Free the writelock and signal any waiters.
		 */
		fdp->reader = 0;
		pthread_cond_signal(fdp->cond);
	}
   done:
	/*
	 * Okay, now the fd mutex can be safely released.
	 */
	pthread_mutex_unlock(fdp->lock);
}
#endif
